
#include "CSManager.h"
#include <gio/gio.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <libgen.h>
#include "autogen/netConnmanManager.h"
#include "autogen/netConnmanSession.h"
#include "autogen/netConnmanNotification.h"

DLT_DECLARE_CONTEXT(CSMG_CTX)


typedef struct {
  csmanager_session_state_change_cb_func_t cb;
  void *user_data;
} cb_session_data;

typedef struct {
  csmanager_manager_state_change_cb_func_t cb;
  void *user_data;
} cb_manager_data;

typedef struct csmanager_session_context_struct {
gchar *ContextIdentifier;
gchar *session_path;
gchar *notifier_path;
gchar *bearer;
gchar *bind_ip;
GPtrArray  *allowed_bearers;
gchar *agent_name;
NetConnmanSession *session;
NetConnmanNotification *notification;
csmanager_session_state session_state;
cb_session_data *signal_change_cb;
csmanager_manager_context *manager_context;
GDBusConnection *dbus_connection;
GVariant *settings;
gboolean force_bind_ip;
} csmanager_session_context;

typedef struct csmanager_manager_context_struct {
gchar *agent_name;
GDBusConnection *dbus_connection;
NetConnmanManager *connman;
guint connmanwatch_id;
csmanager_manager_state manager_state;
csmanager_session_context *session_context;
cb_manager_data *signal_change_cb;
} csmanager_manager_context;



static inline  void csmanager_cb_session_data_new(csmanager_session_context *context, csmanager_session_state_change_cb_func_t cb, void *user_data)
{
	if (!context){
	  INF_TRC_CSM(("csmanager_cb_session_data_free called with NULL context"));
	  return;
	}
	cb_session_data *ret;

	ret = g_new0(cb_session_data, 1);
	ret->cb = cb;
	ret->user_data = user_data;
	context->signal_change_cb = ret;

}
static inline void  csmanager_cb_session_data_free(csmanager_session_context *context)
{
  if (!context){
    INF_TRC_CSM(("csmanager_cb_session_data_free called with NULL context"));
    return;
  }
  cb_session_data *scb = context->signal_change_cb;
  scb->cb = NULL;
  scb->user_data = NULL;
  g_free(scb);
  context->signal_change_cb = NULL;
}



static inline void csmanager_cb_manager_data_new(csmanager_manager_context *context,  csmanager_manager_state_change_cb_func_t cb, void *user_data)
{
  if (!context){
    INF_TRC_CSM(("csmanager_cb_manager_data_new called with NULL context"));
    return;
  }
  cb_manager_data *ret;
  ret = g_new0(cb_manager_data, 1);
  ret->cb = cb;
  ret->user_data = user_data;
  context->signal_change_cb = ret;
}

static inline void  csmanager_cb_manager_data_free(csmanager_manager_context *context)
{
  if (!context){
    INF_TRC_CSM(("csmanager_cb_manager_data_free called with NULL context"));
    return;
  }
  cb_manager_data *scb = context->signal_change_cb;
  scb->cb = NULL;
  scb->user_data = NULL;
  g_free(scb);
  context->signal_change_cb = NULL;
}


gchar *get_random_string(guint len)
{
  gchar *s;

  s = g_malloc0(len + 1);
  for (int i = 0; i < len; i++) {
    s[i] = g_random_int_range(65, 91);
  }

  return s;
}




csmanager_session_context* csmanager_session_context_new(csmanager_manager_context* manager_context){
  csmanager_session_context *session_context = g_new0(csmanager_session_context,1);
  session_context->ContextIdentifier          = NULL;
  session_context->session_path     = NULL;
  session_context->session_state    = CSMANAGER_SESSION_DISCONNECTED;
  session_context->notifier_path    = NULL;
  session_context->signal_change_cb = NULL;
  session_context->session          = NULL;
  session_context->notification     = NULL;
  session_context->bind_ip          = NULL;
  session_context->bearer           = NULL;
  session_context->allowed_bearers  = NULL;
  session_context->agent_name       = g_strdup(manager_context->agent_name);
  session_context->manager_context  = manager_context;
  session_context->dbus_connection  =  manager_context->dbus_connection;
  session_context->settings         = NULL;
  session_context->force_bind_ip      = FALSE;
  return session_context;
}



csmanager_manager_context* csmanager_new(gchar* agent_name){
  DBG_TRC_CSM(("Creating new manager context agent_name %s",agent_name));
  csmanager_manager_context *context = g_new0(csmanager_manager_context, 1 );
  context->dbus_connection   = NULL;
  context->connmanwatch_id   = 0;
  context->connman           = NULL;
  context->signal_change_cb = NULL;
  context->manager_state = CSMANAGER_UNINITIALIZED;
  context->agent_name = g_strdup(agent_name);
  context->session_context = NULL;
  DBG_TRC_CSM(("Done"));
  return context;
}



void csmanager_propagate_session_state(csmanager_session_context *context){
  DBG_TRC_CSM(("propagating session state update"));
  if(!context){
	  INF_TRC_CSM(("not propagating state as session_context is NULL"));
	  return;
  }
  cb_session_data *callback_data = context->signal_change_cb; 
  csmanager_session_state_change_cb_func_t cb = callback_data->cb;
  (*cb)( context, context->session_state, context->ContextIdentifier, callback_data->user_data);
  DBG_TRC_CSM(("done"));
}



void csmanager_propagate_manager_state(csmanager_manager_context *context){
  DBG_TRC_CSM(("propagating manager state update"));
  if(!context){
	  INF_TRC_CSM(("not propagating manager state as manager_context is NULL"));
	  return;
  }
  cb_manager_data *callback_data = context->signal_change_cb; 
  csmanager_manager_state_change_cb_func_t cb = callback_data->cb;
  DBG_TRC_CSM(("cb call"));
  (*cb)( context, context->manager_state, callback_data->user_data);
  DBG_TRC_CSM(("done"));
}



void csmanager_connman_appeared_handler (GDBusConnection *connection,
		const gchar *name,
		const gchar *name_owner,
		gpointer user_data)
{
  GError *error;
  DBG_TRC_CSM(("Connman appeared"));
  error = NULL;
  if (user_data == NULL){
    INF_TRC_CSM(("csmanager_connman_appeared_handler context is NULL"));
    return;
  }
    
  csmanager_manager_context *context = (csmanager_manager_context*) user_data;  
  DBG_TRC_CSM(("Creating connman manager proxy .. "));
  context->connman = net_connman_manager_proxy_new_for_bus_sync (
							G_BUS_TYPE_SESSION,
							G_DBUS_PROXY_FLAGS_NONE,
							"net.connman",
							"/",
							NULL,
							&error);
  if (error) {
    DBG_TRC_CSM(("failed to create connman manager proxy"));
    context->manager_state = CSMANAGER_FAILURE;

  } else {
    DBG_TRC_CSM(("connman manager proxy created"));
    context->manager_state = CSMANAGER_MANAGER_CONNECTED;   
  }
  csmanager_propagate_manager_state(context);
}



void csmanager_connman_vanished_handler (GDBusConnection *connection,
		const gchar *name,
		gpointer user_data)
{
  DBG_TRC_CSM(("Connman vanished"));
  if (user_data == NULL){
    INF_TRC_CSM(("csmanager_connman_vanished_handler context is NULL"));
    return;
  }
  csmanager_manager_context *context = (csmanager_manager_context*) user_data;  
  DBG_TRC_CSM(("Removing reference .. "));
  if ( context->session_context != NULL)
    csmanager_session_destroy(context->session_context);
  if (context->connman != NULL){
    g_object_unref(context->connman);
    context->connman = NULL;
  }
  context->manager_state = CSMANAGER_NOMANAGER;
  csmanager_propagate_manager_state(context);
  DBG_TRC_CSM(("Leaving connman_vanished_handler"));
}



gboolean csmanager_init(csmanager_manager_context *context, csmanager_manager_state_change_cb_func_t cb, void *user_data ){
  guint owner_id;
  GError *error = NULL;
  guint connmanwatch_id = 0;
  DLT_REGISTER_CONTEXT(CSMG_CTX, "CSMG", "CSManager context for Logging"); 
  if (context->manager_state != CSMANAGER_UNINITIALIZED){
    DBG_TRC_CSM(("INIT has already been called"));
    return FALSE;
  } else {
    csmanager_cb_manager_data_new(context, cb, user_data);  
    context->manager_state = CSMANAGER_INITIALIZED;
    context->dbus_connection =  g_bus_get_sync(G_BUS_TYPE_SESSION,NULL,&error);
    if (error)
      {
	DBG_TRC_CSM(("error occurred while connecting to bus."));
	return FALSE;  
      }
    context->connmanwatch_id = g_bus_watch_name (G_BUS_TYPE_SESSION,
						 "net.connman",
						 G_BUS_NAME_WATCHER_FLAGS_NONE,
						 csmanager_connman_appeared_handler,
						 csmanager_connman_vanished_handler,
						 context,
						 NULL);
    if(context->connmanwatch_id == 0){
      DBG_TRC_CSM(("failure to setup watch for connman"));
      return FALSE;
    }
  }
  return TRUE;  
}

void csmanager_notification_destroy(csmanager_session_context *context);

gboolean csmanager_notification_on_handle_release(NetConnmanNotification *object,
						  GDBusMethodInvocation *invocation,
						  GVariant *arg_settings,
						  gpointer *user_data){
  
  DBG_TRC_CSM(("Notification on handle release called"));
  if (!user_data){
    INF_TRC_CSM(("Got state update but session_context is NULL"));
    net_connman_notification_complete_release (object, invocation);
    return TRUE;
  }
  csmanager_session_context *context = (csmanager_session_context *) user_data;
  //Nothing to be done here?
  net_connman_notification_complete_release (object, invocation);
 

  return TRUE;
}



gboolean csmanager_update_state_from_notification(csmanager_session_context *context, gchar* state){
  csmanager_session_state new_session_state = CSMANAGER_SESSION_DISCONNECTED;
  gboolean read_state = TRUE;
  gboolean state_update = FALSE;
  if (!context){
    INF_TRC_CSM(("context is NULL"));
    return FALSE;
  }
  DBG_TRC_CSM(("Starting state mapping %s ",state));
  if (g_strcmp0(state,"online") == 0 ){
    DBG_TRC_CSM(("new state is online")); 
    new_session_state = CSMANAGER_SESSION_ONLINE;
  } else if (g_strcmp0(state, "connected") == 0){
    DBG_TRC_CSM(("new state is connected"));
    new_session_state = CSMANAGER_SESSION_CONNECTED;
  } else if  (g_strcmp0(state, "disconnected") == 0){
    DBG_TRC_CSM(("new state is disconected"));
    new_session_state = CSMANAGER_SESSION_DISCONNECTED;
  } else {
    DBG_TRC_CSM(("failed to map new state"));
    read_state = FALSE;
  }
  if (new_session_state!=context->session_state && read_state==TRUE){
    DBG_TRC_CSM(("state has been updated"));
    context->session_state = new_session_state; 
    state_update =TRUE;
  }
  DBG_TRC_CSM(("Done"));
  return state_update;
}



gboolean csmanager_notification_on_handle_update(NetConnmanNotification *object,
						  GDBusMethodInvocation *invocation,
						  GVariant *arg_settings,
						  gpointer *user_data){
  if (!user_data){
	  INF_TRC_CSM(("Got state update but session_context is NULL"));
	  net_connman_notification_complete_update (object, invocation);
	  return TRUE;
  }
  csmanager_session_context *context = (csmanager_session_context *) user_data;
  GError *error = NULL;
  gboolean ret = TRUE;
  gboolean got_update = FALSE;
  GVariantIter *variant_iter;
  GVariant *variant_value ;
  GVariantDict *dict;
  gchar *str = NULL;
  DBG_TRC_CSM(("update called"));
  DBG_TRC_CSM(("variant format value: %s\n",g_variant_get_type_string(arg_settings)));
  dict = g_variant_dict_new(arg_settings);
  variant_value = g_variant_dict_lookup_value(dict , "State", NULL);
  if (variant_value != NULL){
    DBG_TRC_CSM(("Got state update"));
    DBG_TRC_CSM(("variant format value: %s\n",g_variant_get_type_string(variant_value)));
    str = g_variant_dup_string(variant_value, NULL);
    got_update = csmanager_update_state_from_notification(context, str);
    g_free(str);
    DBG_TRC_CSM(("variant unref")); 
    g_variant_unref (variant_value);
  }

  variant_value = g_variant_dict_lookup_value(dict, "Bearer", NULL);
  if (variant_value != NULL){
    DBG_TRC_CSM(("Got bearer update"));
    context->bearer= g_variant_dup_string(variant_value, NULL);
    g_variant_unref (variant_value);
  }
 
 variant_value=g_variant_dict_lookup_value(dict, "AllowedBearers", NULL);
  if (variant_value != NULL){
    DBG_TRC_CSM(("Got AllowedBearer update"));
    DBG_TRC_CSM(("Found  Type string: %s", g_variant_get_type_string(variant_value)));
    g_variant_get (variant_value, "as", &variant_iter);
    if (context->allowed_bearers != NULL)
      g_ptr_array_free(context->allowed_bearers,TRUE);
    context->allowed_bearers = g_ptr_array_new();
    while (g_variant_iter_loop (variant_iter, "s", &str)){
      g_ptr_array_add(context->allowed_bearers,str);
      DBG_TRC_CSM(("b: %s", str));
    }
    g_variant_iter_free (variant_iter);    
    g_variant_unref (variant_value);
  }
  variant_value=g_variant_dict_lookup_value(dict, "IPv4", NULL);

  if (variant_value != NULL){
    DBG_TRC_CSM(("Found IPv4 dict: Type string: %s", g_variant_get_type_string(variant_value)));
      dict =  g_variant_dict_new(variant_value);
    variant_value = g_variant_dict_lookup_value(dict, "Address", NULL);
    if (variant_value != NULL){
      DBG_TRC_CSM(("Got bind ip update"));
      if (context->force_bind_ip ==TRUE){
	DBG_TRC_CSM(("Ignoring bind ip due to forced_bind_ip"));
      }else{
      context->bind_ip=g_variant_dup_string(variant_value,NULL);
      DBG_TRC_CSM(("bindip %s", context->bind_ip)); 
      }
    }
  }
  net_connman_notification_complete_update (object, invocation); 
  if (got_update == TRUE) {
    csmanager_propagate_session_state(context);
  }
  DBG_TRC_CSM(("update processed"));
  
  return ret;
}

void csmanager_notification_destroy(csmanager_session_context *context){
  if (!context){
    INF_TRC_CSM(("called with NULL context"));
    return;
  }
  if (context->notification != NULL ){
    DBG_TRC_CSM(("freeing notification "));
    g_dbus_interface_skeleton_unexport_from_connection
      (G_DBUS_INTERFACE_SKELETON(context->notification),
       context->dbus_connection);
    if (g_dbus_interface_skeleton_has_connection(G_DBUS_INTERFACE_SKELETON(context->notification),
						 context->dbus_connection)){
      //in this case we don't set new session state. This should be checked by the caller -and in case the session state
      //is different from CSMANAGER_SESSION_DESTROYED this methos should be called again
      INF_TRC_CSM(("failed to un-export notifier object. No error messag e avilable"));
      return;
    }					   
    DBG_TRC_CSM(("notification unexport done"));
    if (context->notification != NULL){
      DBG_TRC_CSM(("freeing notification pointer"));
      g_object_unref(context->notification);
      context->notification = NULL;
    }
    else
      DBG_TRC_CSM(("notification already none"));
  }
}


void csmanager_setup_notification_client(csmanager_session_context *context){
 
   DBG_TRC_CSM(("concat of notifier_path"));
   if(!context){
     INF_TRC_CSM(("context is NULL"));
     return;
   }
   gchar *new_notifier = NULL;
   gchar *tmp = NULL;
   GError *error = NULL;
   new_notifier = g_strconcat("/agw/",context->agent_name,NULL);
   if (context->ContextIdentifier != NULL)
     {
       
       tmp= g_strconcat(new_notifier, "/",
				  context->ContextIdentifier,
				  NULL);
       g_free(new_notifier);
       new_notifier = tmp;
     }

    tmp = g_strconcat(new_notifier, "/",
			      get_random_string(6),
				  NULL);
    g_free(new_notifier);
    new_notifier = tmp;
    DBG_TRC_CSM(("Creating notification object"));
    context->notification = net_connman_notification_skeleton_new();
   

    g_signal_connect (context->notification,
		      "handle-release",
		     G_CALLBACK (csmanager_notification_on_handle_release),
		     context);
   g_signal_connect (context->notification,
		     "handle-update",
		     G_CALLBACK (csmanager_notification_on_handle_update),
		     context);
   DBG_TRC_CSM(("Exporting notifier to dbus"));
   if (!g_dbus_interface_skeleton_export (G_DBUS_INTERFACE_SKELETON (context->notification),
					  context->dbus_connection,
					  new_notifier,
					  &error))
     {
	   if(error){
		   INF_TRC_CSM(("error setting up notification interface %d %s",error->code, error->message)); 
		   g_error_free(error);
	   }
       g_free(new_notifier);
       csmanager_notification_destroy(context); 
       DBG_TRC_CSM(("error setting up notification interface")); 
     } else {
     context->notifier_path = new_notifier;
   }
}



csmanager_session_context*  csmanager_session_create(csmanager_manager_context *manager_context, csmanager_session_state_change_cb_func_t cb, void *user_data, const gchar *ContextIdentifier, gchar *force_bind_ip){
  gchar *session_path = NULL;
  GVariant *settings = NULL;
  GVariantBuilder * builder = NULL;
  GError *error = NULL;
  csmanager_session_context *session_context = csmanager_session_context_new(manager_context);
  csmanager_cb_session_data_new(session_context, cb, user_data);  
  if (force_bind_ip != NULL){
    session_context->force_bind_ip = TRUE;
    session_context->bind_ip = g_strdup(force_bind_ip);
  }
  DBG_TRC_CSM(("Setting up session"));
  session_context->ContextIdentifier = g_strdup(ContextIdentifier);
  csmanager_setup_notification_client(session_context);
  if (session_context->notification == NULL){
	DBG_TRC_CSM(("notifier creation failed. Session creation aborted"));
	csmanager_session_free(session_context);
	return NULL;
  }											 
  DBG_TRC_CSM(("notifier created.. building variant"));
  builder =  g_variant_builder_new (G_VARIANT_TYPE ("a{sv}"));
  g_variant_builder_add (builder, "{sv}", "ContextIdentifier", g_variant_new_string(ContextIdentifier));
  settings =  g_variant_builder_end (builder);
  DBG_TRC_CSM(("creating session object"));
  net_connman_manager_call_create_session_sync(manager_context->connman,
					       settings,
					       session_context->notifier_path,
					       &session_path,
					       NULL,
					       &error);
  if (error){
    DBG_TRC_CSM(("ERROR creating session %d %s",error->code, error->message));
    g_error_free(error);
    csmanager_notification_destroy(session_context);
    g_variant_unref(settings);
    return NULL;
  } else {
    DBG_TRC_CSM(("Session created"));
    session_context->session_path = g_strdup(session_path);
    DBG_TRC_CSM(("connecting session object"));
    session_context->session = net_connman_session_proxy_new_for_bus_sync (G_BUS_TYPE_SESSION,
								   G_DBUS_PROXY_FLAGS_NONE,
								   "net.connman",
								   session_context->session_path,
								   NULL,
								   &error);
  }
  if (error){
    DBG_TRC_CSM(("error connecting session code %d message %s", error->code, error->message));
    g_error_free(error);
    csmanager_session_destroy(session_context);
    csmanager_session_free(session_context);
    return NULL;
  }
  session_context->session_state = CSMANAGER_SESSION_CONNECTING;
  session_context->settings = settings;
  manager_context->session_context = session_context;
  csmanager_propagate_session_state(session_context);
  DBG_TRC_CSM(("Done"));
  return session_context;
}



void csmanager_session_destroy(csmanager_session_context *context){
  /* This method will try to destroy session. On success session state is set accordingly - on 
  failure it will return without changing the session state.																		  
  TODO: add boolean return value here
  */
  GError *error = NULL;
  DBG_TRC_CSM(("CTOR: Destroying session.."));
  gboolean destroyed = FALSE;
  if (context->session == NULL) {
	INF_TRC_CSM(("session is already NULL"));
	return NULL;
  } 
  DBG_TRC_CSM(("calling session destroy"));
  destroyed = net_connman_session_call_destroy_sync(context->session,
						    NULL, &error);
  if (destroyed == FALSE){
    if(error){
      INF_TRC_CSM(("session destruction failed: code %d message: %s ", error->code, error->message));
      g_error_free(error);
    } else {
      INF_TRC_CSM(("session destruction failed: no error message "));
    }
    
  }
  g_object_unref(context->session);
  context->session = NULL;
  DBG_TRC_CSM(("session destroyed"));
  context->session_state = CSMANAGER_SESSION_DESTROYED;
  csmanager_propagate_session_state(context);
  DBG_TRC_CSM(("Done"));
}

gboolean csmanager_session_free(csmanager_session_context *context){
  /*
  csmanager_session_destroy must have been called before (and it must have been successfull). We return False, if session state is != 
  CSMANAGER_SESSION_DESTROYED. If you called csmanager_session_destroy
  before and get FALSE here something went seriously wrong. 
  */
  gboolean ret = FALSE;
  DBG_TRC_CSM(("CTOR: csmanager session free"));
  if ( context == NULL )
  {
	  DBG_TRC_CSM(("Context is null"));
	  return TRUE;
  }
  if (context->session_state != CSMANAGER_SESSION_DESTROYED){
    DBG_TRC_CSM(("Not freeing session as it is still active (not destroyed"));
    return FALSE;
  }
  DBG_TRC_CSM(("freeing strings"));
  csmanager_notification_destroy(context);
  g_free(context->ContextIdentifier);
  g_free(context->bearer);
  g_free(context->bind_ip);
  g_free(context->agent_name);
  g_free(context->notifier_path);
  DBG_TRC_CSM(("freeing cb"));
  csmanager_cb_session_data_free(context);
  if(context->allowed_bearers != NULL){
	DBG_TRC_CSM(("freeing allowed bearers array"));
	g_ptr_array_free(context->allowed_bearers,TRUE);
  }
  if(context->settings != NULL){
    DBG_TRC_CSM(("freeing settings"));
    g_variant_unref(context->settings);
  }
  //this was inherited by manager context and will be freed there
  context->dbus_connection = NULL;
  (context->manager_context)-> session_context = NULL;
  DBG_TRC_CSM(("freeing context"));
  g_free(context);
  return TRUE;
}

void csmanager_session_reconnect(csmanager_session_context *context){
  GError *error = NULL;
  DBG_TRC_CSM(("Reconnecting session .. disconnect"));
  net_connman_session_call_disconnect_sync (
					    context->session,
					    NULL,
					    &error);
  if (error){
    DBG_TRC_CSM(("failed to disconnect session %d %s",error->code, error->message));
	g_error_free(error);				 
  }
  net_connman_session_call_connect_sync (
					 context->session,
					 NULL,
					 &error);
  if(error){
    DBG_TRC_CSM(("failed to reconnect session code: %d message: %s",error->code, error->message));
	g_error_free(error);
  } else {
    DBG_TRC_CSM(("Session reconnected"));
  }
}


void csmanager_free(csmanager_manager_context *context){
  if (context == NULL){
    INF_TRC_CSM(("context already NULL"));
    return;
  }
  csmanager_session_destroy(context->session_context);
  csmanager_session_free(context->session_context);
  g_bus_unwatch_name(context->connmanwatch_id);
  if (context->connman != NULL )
    g_object_unref(context->connman);
  if (context->dbus_connection)
    g_dbus_connection_close_sync(context->dbus_connection, NULL, NULL);
  g_free(context->agent_name);
  csmanager_cb_manager_data_free(context);
  g_free(context);

}

gboolean csmanager_session_is_online(csmanager_session_context *context){
  gboolean online = FALSE;
  if (context->session_state > CSMANAGER_SESSION_CONNECTING)
    online = TRUE;
  return online;
}

GVariant* csmanager_session_get_settings(csmanager_session_context *context){
  return context->settings;
}

// get datagram or stream socket. this can also be used with e.g curl
int csmanager_session_get_bound_socketv4(csmanager_session_context *context, const uint port, gboolean datagram){
   struct sockaddr_in bindaddr;
   int sockid;
   int err;
   DBG_TRC_CSM(("Starting to create bound socket"));
   if (datagram == TRUE)
     sockid = socket(PF_INET, SOCK_DGRAM, 0 );
   else
      sockid = socket(PF_INET, SOCK_STREAM, 0 );
   if (sockid == -1){
     DBG_TRC_CSM(("Failed to create socket"));
	 return sockid;
   }
   memset(&bindaddr, 0, sizeof(bindaddr));
   bindaddr.sin_family = AF_INET;
   bindaddr.sin_port = htons(port);
   err = inet_pton(AF_INET, context->bind_ip, &(bindaddr.sin_addr));
   if (err < 0 ) {
	   DBG_TRC_CSM(("Failed toconvert IPv4 addresses from text to binary form "));
	   return -1;
   }
   if (bind(sockid,(struct sockaddr*) &bindaddr, sizeof bindaddr) == -1){
     DBG_TRC_CSM(("Failed to bind socket"));
     return -1;
   }
   return sockid; 
}

//return gateway interface ip. this may be used e.g. to bind a socket to this address
gchar* csmanager_session_get_interface_ip(csmanager_session_context *context){
	if (!context){
		return NULL;
	}
	return g_strdup(context->bind_ip);
}
    
gchar* csmanager_session_get_bearer(csmanager_session_context *context){
  if (!context){
		return NULL;
	}
  return g_strdup(context->bearer);
}

GPtrArray* csmanager_session_get_allowed_bearers(csmanager_session_context *context){
	if (!context){
		return NULL;
	}
  return context->allowed_bearers;
}

csmanager_session_state csmanager_session_get_state(csmanager_session_context *context){
  	if (!context){
		return CSMANAGER_SESSION_DESTROYED;
	}
  return context->session_state;
}

csmanager_manager_state csmanager_manager_get_state(csmanager_manager_context *context){
  if (!context){
	  return CSMANAGER_FAILURE;
  }
  return context->manager_state; 
}

void csmanager_session_set_cb(csmanager_session_context *context,  csmanager_session_state_change_cb_func_t cb, void *user_data){
  DBG_TRC_CSM(("setting callback for session"));
  csmanager_cb_session_data_free(context);
  csmanager_cb_session_data_new(context, cb, user_data); 
}
