/**************************************************************************
 * @file 	- defusePNG.c
 * @brief   - Utility to convert PNG files to raw images, and then back to PNG files. 
 *			  Most functions of this file has been taken from widely available usage examples. 
 *			   source 1 : http://www.libpng.org/pub/png/libpng-1.2.5-manual.html
 *			   source 2 : https://www.cs.ucsb.edu/~pconrad/cs32/15F/lect/11.25/libpng/libpngExample1.c 
 * @history - v 0.1 mpr3hi, 0612017			 
 *************************************************************************/
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <sys/types.h>

#define PNG_DEBUG 3
#include <png.h>

/**
 * @brief - structure to hold required data types to hold raw image.
 *		    The data types and naming is self-explanatory.
 */
typedef struct {
	int width;
	int height;
	png_byte color_type;
	png_byte bit_depth;
	// to hold raw-image
	png_bytep * row_pointers;
}image_t;

/**
 * @brief - supported error types.
 */
enum DEFUSE_PNG_ERRNO{
	DEFUSE_PNG_SUCCESS = 0,
	DEFUSE_PNG_INCORRECT_USAGE,
	DEFUSE_PNG_MISSING_SOURCE,
	DEFUSE_PNG_MISSING_DESTINATION,
	DEFUSE_PNG_NON_PNG_TYPE,
	DEFUSE_PNG_READ_STRUCT_FAILURE,
	DEFUSE_PNG_CREATE_STRUCT_FAILURE,
	DEFUSE_PNG_INIT_IO_FAILURE,
	DEFUSE_PNG_USER_NOT_ALLOWED	
};

/**
 * @brief - API to convert PNG image to raw image. row_pointers of image_t shall be holding the raw image.
 * @param - file_name - PNG file name, which needs to be converted.
 * @param - structure image_t to hold required parameters of image, before conversion .
 */
void read_png_file(char* file_name, image_t *image)
{		// required structures for raw image conversion from PNG file. Refer : libpng Manual.
		png_structp png_ptr;
		png_infop info_ptr;		
		
        char header[8];    // 8 is the maximum size that can be checked
		int y;

        //  Open the source file 
        FILE *fp = fopen(file_name, "rb");
        if (!fp){
				printf("\n Source file doesnt exist");
				exit(DEFUSE_PNG_MISSING_SOURCE);                
		}
		// copy the types [1-8] into the buffer : header
        fread(header, 1, 8, fp);
		
		// match the PNG signature
        if (png_sig_cmp(header, 0, 8)){
			// non-PNG file provided
			// exit with error code : DEFUSE_PNG_NON_PNG_TYPE
			exit(DEFUSE_PNG_NON_PNG_TYPE);
		}

		// Allocate and initialize structure png_structp - png_ptr,  for reading PNG file
        png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);	
        if (!png_ptr){
			// Unable to allocate/initlize structure
			// Exit with err code : DEFUSE_PNG_READ_STRUCT_FAILURE
			exit(DEFUSE_PNG_READ_STRUCT_FAILURE);
		}

		// Allocate and initialize structure: png_infop - info_ptr
        info_ptr = png_create_info_struct(png_ptr);
        if (!info_ptr){
			// Unable to allocate/initlize structure
			// Exit with err code : DEFUSE_PNG_CREATE_STRUCT_FAILURE
			exit(DEFUSE_PNG_CREATE_STRUCT_FAILURE);
		}

		// longjmp in order to back to this routine, if libpng encounters an error
        if (setjmp(png_jmpbuf(png_ptr))){
			exit(DEFUSE_PNG_INIT_IO_FAILURE);
		}

		// Initialize the default input/output functions for the PNG file to standard C streams. 
        png_init_io(png_ptr, fp);
		// store the number of bytes of the PNG file signature that have been read from the PNG stream.
        png_set_sig_bytes(png_ptr, 8);
		// Reads the information before the actual image data from the PNG file.
        png_read_info(png_ptr, info_ptr);

		// fetch image parameters and store to the image_t structure
        image->width = png_get_image_width(png_ptr, info_ptr);
        image->height = png_get_image_height(png_ptr, info_ptr);
        image->color_type = png_get_color_type(png_ptr, info_ptr);
        image->bit_depth = png_get_bit_depth(png_ptr, info_ptr);       
		
		// Update the structure pointed to by info_ptr to reflect any transformations that have been requested.
        png_read_update_info(png_ptr, info_ptr);

		// longjmp in order to back to this routine, if libpng encounters an error and start reading the file
        if (setjmp(png_jmpbuf(png_ptr))){
			exit(DEFUSE_PNG_INIT_IO_FAILURE);
		}
		// allocate memory from heap, to store the raw image
        image->row_pointers = (png_bytep*) malloc(sizeof(png_bytep) * image->height);
		// allocate each row
        for (y=0; y < image->height; y++)
                image->row_pointers[y] = (png_byte*) malloc(png_get_rowbytes(png_ptr,info_ptr));

		// store the entire image to  image->row_pointer
        png_read_image(png_ptr, image->row_pointers);
	
		// destroy png_ptr & info_ptr, as per the libpng manual
		if (png_ptr && info_ptr) {
			png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
			png_ptr = NULL;
			info_ptr = NULL;
		}

		// close the fild descriptor
        fclose(fp);
}
/**
 * @brief - API to write raw image holded by image_t->row_pointers pointer to provide PNG file name.
 * @param - file_name - destination PNG file name.
 * @param - structure image_t to hold required parameters to write PNG image .
 */
void write_png_file(char* file_name, image_t *image)
{
		// required structures for raw image conversion from PNG file. Refer : libpng Manual.
		png_structp png_ptr;
		png_infop info_ptr;	
	    int y;
		
         //  create the destination file 
        FILE *fp = fopen(file_name, "wb");
        if (!fp){
			printf("\nDestination file doesnt exist");
			exit(DEFUSE_PNG_MISSING_DESTINATION);                
		}

        // Allocate and initialize structure png_structp - png_ptr,  for writing PNG file
        png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);

        if (!png_ptr){
			// Unable to allocate/initlize structure
			// Exit with err code : DEFUSE_PNG_INIT_IO_FAILURE
			exit(DEFUSE_PNG_CREATE_STRUCT_FAILURE);
		}
		// Allocate and initialize structure: png_infop - info_ptr
        info_ptr = png_create_info_struct(png_ptr);
        if (!info_ptr){
			// Unable to allocate/initlize structure
			// Exit with err code : DEFUSE_PNG_CREATE_STRUCT_FAILURE
			exit(DEFUSE_PNG_CREATE_STRUCT_FAILURE);
		}
		
		// longjmp in order to back to this routine, if libpng encounters an error
        if (setjmp(png_jmpbuf(png_ptr))){
			exit(DEFUSE_PNG_INIT_IO_FAILURE);
		}
		// Initialize the default input/output functions for the PNG file to standard C streams. 
        png_init_io(png_ptr, fp);


        // longjmp in order to back to this routine, if libpng encounters an error
        if (setjmp(png_jmpbuf(png_ptr)))
		{
			exit(DEFUSE_PNG_INIT_IO_FAILURE);
		}
		// Set image header information in info_ptr. width is the image width in pixels. height is the image height in pixels. bit_depth is the bit depth of the image.
        png_set_IHDR(png_ptr, info_ptr, image->width, image->height,
                     image->bit_depth, image->color_type, PNG_INTERLACE_NONE,
                     PNG_COMPRESSION_TYPE_BASE, PNG_FILTER_TYPE_BASE);
		// writes the PNG information in info_ptr to file
        png_write_info(png_ptr, info_ptr);


        // longjmp in order to back to this routine, if libpng encounters an error
        if (setjmp(png_jmpbuf(png_ptr))){
			exit(DEFUSE_PNG_INIT_IO_FAILURE);
		}
		// Write the rows of given image data. 
        png_write_image(png_ptr, image->row_pointers);


        /* end write */
        if (setjmp(png_jmpbuf(png_ptr))){
			exit(DEFUSE_PNG_INIT_IO_FAILURE);
		}
		// Writes the end of a PNG file to which the image data has already been written.
        png_write_end(png_ptr, NULL);

        // clean up the raw image
        for (y=0; y < image->height; y++)
                free(image->row_pointers[y]);
        free(image->row_pointers);
		// close the file descriptor
        fclose(fp);
}
/**
 * @brief - Main function.
 * @param - argv[1] - source file (PNG file), which shall be converted to raw image.
 * @param - argv[2] - destination file name, which shall be created (PNG file) from raw image.
 */
int main(int argc, char **argv)
{
		// structure to hold parameters for raw image
		image_t image;
		
        if (argc != 3){
			printf("Usage: program_name <file_in> <file_out>");
			exit(DEFUSE_PNG_INCORRECT_USAGE);
		}
		
		if( getuid() == 0)
		{
			printf("Usage: can't run as root user");
			exit(DEFUSE_PNG_USER_NOT_ALLOWED);
		}
		// convert PNG image ( source -- argv[1]) to raw image
        read_png_file(argv[1], &image);  
		// convert raw image ( destination -- argv[2]) to PNG image
        write_png_file(argv[2], &image);
		
        return 0;
}