/**************************************************************************
 *\file     TestComponent.cpp
 *\brief    LXC Master Component in Container

 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2017
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 *
 * History -- 0.1 mpr3hi, first version
 *			  0.2 mpr3hi, integration connman proxy, addition LCM ex. via agw
 *************************************************************************/
#include "dbus/com/bosch/TestComponent.h"

namespace com {
namespace bosch {	

DLT_DECLARE_CONTEXT(TEST_CONTAINER);

uint  session_avail=0;
uint  manager_avail=0;
std::string session_path;

TestComponent *TestComponent::_instance=0; 

TestComponent::TestComponent():
 ::asf::core::BaseComponent(),
 m_NativeService1Proxy(NativeService1Proxy::createProxy(NATIVE_COMPONENT_SERV1_PORT , *this)),
 m_NativeService2Proxy(NativeService2Proxy::createProxy(NATIVE_COMPONENT_SERV2_PORT , *this)),
 m_ManagerProxy(ManagerProxy::createProxy(CMANAGER_PORT , *this)),
 m_SessionProxy(SessionProxy::createProxy(CSESSION_PORT , *this)),
 m_LCMProxy(LifeCycleManagerProxy::createProxy(AGW_LCM_PROXY_PORT , *this)),
 m_AMProxy(AudioManagerProxy::createProxy(AGW_AM_PROXY_PORT , *this)) ,
 NotificationStub(CNOTIFICATION_PORT),
 OnAllocateSignalRcvd(false),
 SourceActivityRcvd(false),
 restartConfigTimer(TEST_CASE_RESTART_SEC_DEFAULT) 
{ 
  _instance = this;
  /* register TEST_CONTAINER app for DLT logging */
  DLT_REGISTER_APP("MEAG","TEST_CONTAINER Component Registration for Logging");
  /* register TEST_CONTAINER context */
  DLT_REGISTER_CONTEXT(TEST_CONTAINER,"TEST_CONTAINER","TEST_CONTAINER context for Logging"); 	   
  DLT_REGISTER_INJECTION_CALLBACK(TEST_CONTAINER,TEST_DLT_COMMAND_ID_01, uIntDLTTraceInjectionCallback);
	
  DEBUG_TRACE((TEST_CONTAINER,"TestComponent::TestComponent Ctor"));	
  
  parseTestConfig();
}

// parse TestConfig.xml
void TestComponent::parseTestConfig(){	
	DEBUG_TRACE((TEST_CONTAINER,"TestComponent::parseTestConfig"));			
	
	TiXmlDocument doc;
	// Load the TestConfig.xml file via TinyXML utils
	if(!doc.LoadFile(TEST_CASE_CONFIG_FILE))
	{
		DEBUG_TRACE((TEST_CONTAINER,"TestConfig.xml not found"));	
		return;
	}
		
	// Get the root element, <TestConfig>
	TiXmlElement* root = doc.FirstChildElement();
	if(root == NULL)
	{
		// "Failed to load file: No root element."						
		doc.Clear();
		DEBUG_TRACE((TEST_CONTAINER,"Failed to load TestConfig.xml. No root element found."));	
		return;
	}
		
	// Itervate over the test cases, with <TestCase> tags
	for(TiXmlElement* elem = root->FirstChildElement(); \
		elem != NULL; elem = elem->NextSiblingElement())
	{
		 string elemName = elem->Value();
		 // check if reset time is configured in configuration
		 if(elemName == TEST_CASE_RESET_TIMER)			 
		 {				
			// fetch reset time value (in seconds)				
			string val_sec = elem->Attribute(TEST_CASE_ATTRVAL_TIME);
			restartConfigTimer = atoi(val_sec.c_str());
		 }
			 
		 // check the presence of base tag, TestCase
		 if(elemName == TEST_CASE_BASE_TAG)
		 {
			string attr = elem->Attribute(TEST_CASE_ATTR_TYPE);
			DEBUG_TRACE((TEST_CONTAINER,"type=%s",attr.c_str()));				
			
			
			if(attr == TEST_CASE_ATTRVAL_SOCKET)
			{			
				// fetch attribute for socket subtest
				string subtest = elem->Attribute(TEST_CASE_ATTR_SUBTEST);						
				parseSocketTest(subtest);
				
			}
			else if(attr == TEST_CASE_ATTRVAL_NETWORK) {
			  // fetch attribute for network subtest
			  string subtest = elem->Attribute(TEST_CASE_ATTR_SUBTEST);
			  parseNetworkTest(subtest);
			
			}
			else if(attr == TEST_CASE_ATTRVAL_DBUS)
			{		
				// fetch attribute service
				string serv = elem->Attribute(TEST_CASE_ATTR_SERVICE);
				// fetch attribute method
				string method = elem->Attribute(TEST_CASE_ATTR_METHOD);
				parseDbusTest(serv,method);
				
			}
			else if(attr == TEST_CASE_ATTRVAL_FILECREATE)
			{			
				DEBUG_TRACE((TEST_CONTAINER,"create file test.txt"));	
				// fetch attribute, close
				string file_close = elem->Attribute(TEST_CASE_ATTR_FILECLOSE);
				// fetch attribute, folder
				string folder = elem->Attribute(TEST_CASE_ATTR_FOLDER);	
				parseFileHandleTest(file_close,folder);						
			}
			else if(attr == TEST_CASE_ATTRVAL_LCM_REG)
			{		
				DEBUG_TRACE((TEST_CONTAINER,"register to LCM via AGW"));	
				sendLcmRegisteration();
			}	
			else if(attr == TEST_CASE_ATTRVAL_AUDIO)
			{
				// fetch attribute subtest
				string subtest = elem->Attribute(TEST_CASE_ATTR_SUBTEST);
				if(subtest == TEST_CASE_ATTRVAL_SUBTEST_AUD_ALLOC){						
					DEBUG_TRACE((TEST_CONTAINER,"Allocate Audio Channgel"));	
					allocateAudioChannel();					
				}						
			}
		}						
	}
	 DEBUG_TRACE((TEST_CONTAINER," Restarting container in %d second(s)",restartConfigTimer));		
	 signal(SIGALRM, _instance->resetContainer);
	 alarm(restartConfigTimer);	
}	

void TestComponent::resetContainer(int signal)
{	
	_instance->execSlave(TEST_CASE_BIN_ISORESTART);			
}

void TestComponent::parseSocketTest(std::string subtest)
{
	DEBUG_TRACE((TEST_CONTAINER,"parseSocketTest to handle subtest:%s",subtest.c_str()));	
	// runs a slave, and notify to native socket server
	if(subtest == TEST_CASE_ATTRVAL_SUBTEST_SLAVE){			
		DEBUG_TRACE((TEST_CONTAINER,"exec slave"));	
		execSlave(TEST_CASE_BIN_SLAVE);
	}
	// runs a slave, which allocates memory after 1 second, and doesnt free it
	else if(subtest == TEST_CASE_ATTRVAL_SUBTEST_MEMLEAK){
		DEBUG_TRACE((TEST_CONTAINER,"exec memleak"));	
		execSlave(TEST_CASE_BIN_MEMLEAK);
	}
	// runs a slave, which performs infinite stress computation
	// ToDo: add more parameters for stress 
	else if(subtest == TEST_CASE_ATTRVAL_SUBTEST_STRESS){
		DEBUG_TRACE((TEST_CONTAINER,"exec stress"));	
		execSlave(TEST_CASE_BIN_STRESS);	
	}
}

void TestComponent::parseNetworkTest(std::string subtest)
{
	// runs a network testcase
	if(subtest == TEST_CASE_ATTRVAL_SUBTEST_SSC){			
		DEBUG_TRACE((TEST_CONTAINER,"start single service container test"));
		testSessionSSC();
	}	
	else if (subtest == TEST_CASE_ATTRVAL_SUBTEST_HTTP_GET){
		DEBUG_TRACE((TEST_CONTAINER,"start http_get test without session"));
		fetch_page();
	}
}

void TestComponent::parseDbusTest(std::string serv, std::string method)	{	
	// invokes method1 of service1
	if((serv == TEST_CASE_ATTRVAL_SERVICE_1) && \
	(method == TEST_CASE_ATTRVAL_METHOD_1)){
		DEBUG_TRACE((TEST_CONTAINER,"sendService1Method1"));	
		sendMountNavishareReq();
	}
	// invokes method2 of service1
	else if( (serv == TEST_CASE_ATTRVAL_SERVICE_1) && \
	(method == TEST_CASE_ATTRVAL_METHOD_2)){
		DEBUG_TRACE((TEST_CONTAINER,"sendService1Method2"));	
		sendUmountNavishareReq();
	}
	// invokes method1 of service2
	else if((serv == TEST_CASE_ATTRVAL_SERVICE_2) && \
	(method == TEST_CASE_ATTRVAL_METHOD_1)){
		DEBUG_TRACE((TEST_CONTAINER,"sendService2Method1"));	
		sendService2Method1();
	}
}
void TestComponent::parseFileHandleTest(std::string file_close, std::string folder) {
	// folder tag found
	if( !folder.empty() ) {
		DEBUG_TRACE((TEST_CONTAINER,"create file:test.txt in folder %s", folder.c_str()));							
		if( file_close ==  TEST_CASE_ATTRVAL_FILECLOSE ){			
		// create file, close the file handle .			
			createFile(folder,true);
		}
		else{
		// create file, don't close the file handle .	
			createFile(folder,false);
		}
	}
}
	
// synchronize proxies, if required
void TestComponent::onAvailable(
      const boost::shared_ptr< asf::core::Proxy >& proxy,
      const ServiceStateChange &stateChange) 
{	
  DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onAvailable proxy %s", proxy->getPortName().c_str()));
  if (proxy == m_ManagerProxy){
    DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onAvailable identified ManagerProxy"));
    manager_avail= true;
  }
  else if (proxy == m_SessionProxy){
    DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onAvailable identified SessionProxy"));
    session_avail = true;
  } 
  else {
    DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onAvailable proxy, no sync required"));
  }
}

void TestComponent::onUnavailable(
      const boost::shared_ptr< asf::core::Proxy >& proxy,
      const ServiceStateChange &stateChange) 
{
    DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onUnavailable"));
    if (proxy == m_ManagerProxy){
      DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onUnavailable identified ManagerProxy"));
      manager_avail= false;
    } else if (proxy == m_SessionProxy){
      DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onUnavailable identified SessionProxy"));
      session_avail = false;
    } else {
      DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onUnavailable proxy ignored"));
    }
}

// handle response if required
void TestComponent::onMountNavishareResponse(
		const ::boost::shared_ptr< NativeService1Proxy >& proxy,
		const ::boost::shared_ptr< MountNavishareResponse >& response)
{	
	DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onMountNavishareResponse"));
}
// handle error if required
void TestComponent::onMountNavishareError(
		const ::boost::shared_ptr< NativeService1Proxy >& proxy, 
		const ::boost::shared_ptr< MountNavishareError >& error)
{	
	DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onMountNavishareError"));
}
// handle response if required
void TestComponent::onUmountNavishareResponse(
		const ::boost::shared_ptr< NativeService1Proxy >& proxy,
		const ::boost::shared_ptr< UmountNavishareResponse >& response)
{	
		DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onUmountNavishareResponse"));
}
// handle error if required
void TestComponent::onUmountNavishareError(
		const ::boost::shared_ptr< NativeService1Proxy >& proxy, 
		const ::boost::shared_ptr< UmountNavishareError >& error)
{
		DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onUmountNavishareError"));
}
// handle response if required
void TestComponent::onService2Method1Response(
		const ::boost::shared_ptr< NativeService2Proxy >& proxy,
		const ::boost::shared_ptr< Service2Method1Response >& response)
{	
		DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onService2Method1Response"));	
}
// handle error if required
void TestComponent::onService2Method1Error(
		const ::boost::shared_ptr< NativeService2Proxy >& proxy, 
		const ::boost::shared_ptr< Service2Method1Error >& error)
{	
		DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onService2Method1Error"));	
}
 void  TestComponent::onDestroySessionError(
			     const ::boost::shared_ptr< ManagerProxy >& proxy, 
			     const ::boost::shared_ptr< DestroySessionError >& error){   
    DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onDestroySessionError"));
}

void  TestComponent::onDestroySessionResponse(
				const ::boost::shared_ptr< ManagerProxy >& proxy, 
				const ::boost::shared_ptr< DestroySessionResponse >& response){
  
  DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onDestroySessionResponse"));    
}

void  TestComponent::onCreateSessionError(
			    const ::boost::shared_ptr< ManagerProxy >& proxy, 
			    const ::boost::shared_ptr< CreateSessionError >& error){
  DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onCreateSessionError"));
}

// Update func. description 
void  TestComponent::onCreateSessionResponse(
			       const ::boost::shared_ptr< ManagerProxy >& proxy, 
			       const ::boost::shared_ptr< CreateSessionResponse >& response){
  DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onCreateSessionResponse"));
  session_path = response->getSession();
  DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onDestroySessionResponse got response %s", session_path));
}

// Update func. description 
void TestComponent::onReleaseRequest (const ::boost::shared_ptr< ReleaseRequest >& request){
  DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onReleaseRequest"));  
}

// Update func. description 
void TestComponent::onUpdateRequest (const ::boost::shared_ptr< UpdateRequest >& request){
  DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onUpdateRequest"));
  ::std::string str_state;
  ::std::string signature;
  ::std::map< ::std::string, ::asf::dbus::DBusVariant > settings;
  settings  = request->getSettings();
  ::std::map< ::std::string, ::asf::dbus::DBusVariant >::iterator it = settings.begin();
  enum connman_session_state act_state;

  while(it != settings.end()){
    DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onUpdateRequest: settings key: %s", it->first));
    it++;
  }
  if (settings.find("State") == settings.end() ){
     DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onUpdateRequest: no State update"));
  } else {
    settings["State"].getSignature(signature);
    DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onUpdateRequest: state signature: %s" ,signature));
    dbus_message_iter_get_basic(settings["State"].getReadIterator(), &str_state);
    DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onUpdateRequest: string state: %s" , str_state.c_str()));
    act_state = ( enum connman_session_state) atoi(str_state.c_str()); 
    DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onUpdateRequest: State update found %i" , act_state));
    if( act_state == CONNMAN_SESSION_STATE_CONNECTED || act_state == CONNMAN_SESSION_STATE_ONLINE){
      DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onUpdateRequest: session is now online")); 
      
    } else {
      DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onUpdateRequest: session is offline"));
      //this might be a state change .. if session has been online before
    }
  }  
}
// handle error if required
void TestComponent::onRegisterError(
	const ::boost::shared_ptr< LifeCycleManagerProxy >& proxy, 
	const ::boost::shared_ptr< RegisterError >& error)
{	
	DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onRegisterError"));	
}
// handle response if required
void TestComponent::onRegisterResponse(
	const ::boost::shared_ptr< LifeCycleManagerProxy >& proxy, 
	const ::boost::shared_ptr< RegisterResponse >& response)
{	
	request_id = response->getRequest_id();
	DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onRegisterResponse, Recieved Request ID=%d",request_id));		
	
}
// handle error if required
void TestComponent::onLifecycleRequestError(
	const ::boost::shared_ptr< LifeCycleManagerProxy >& proxy, 
	const ::boost::shared_ptr< LifecycleRequestError >& error)
{	
	DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onLifecycleRequestError"));	
}
// handle Signal LifecycleRequest
void TestComponent::onLifecycleRequestSignal(
	const ::boost::shared_ptr< LifeCycleManagerProxy >& proxy,
	const ::boost::shared_ptr< LifecycleRequestSignal >& signal)
{	
	DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onLifecycleRequestSignal"));	
	m_LCMProxy->sendLifecycleRequestCompleteRequest(*this, request_id);
}

// handle error if required
void TestComponent::onLifecycleRequestCompleteError(
	const ::boost::shared_ptr< LifeCycleManagerProxy >& proxy,
	const ::boost::shared_ptr< LifecycleRequestCompleteError >& error)
{		
	DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onLifecycleRequestCompleteError"));	
}
// handle response if required
void TestComponent::onLifecycleRequestCompleteResponse(
	const ::boost::shared_ptr< LifeCycleManagerProxy >& proxy,
	const ::boost::shared_ptr< LifecycleRequestCompleteResponse >& response)
{	
	DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onLifecycleRequestCompleteResponse"));	
} 

// allocate audio channel, before playing a song
void TestComponent::allocateAudioChannel()
{
	DEBUG_TRACE((TEST_CONTAINER,"TestComponent::allocateAudioChannel"));	
	
	sleep(3);
	//register signals ( SourceActivity, OnAllocate)	
	m_AMProxy->sendSourceActivityRegister(*this);
	m_AMProxy->sendOnAllocateRegister(*this);
	
	DEBUG_TRACE((TEST_CONTAINER,"TestComponent::sending SourceAvailability"));	
	m_AMProxy->sendSourceAvailabilityRequest(*this,1,true);	
	sleep(1);
	DEBUG_TRACE((TEST_CONTAINER,"TestComponent::sending AudioRouteRequest"));	
	m_AMProxy->sendAudioRouteRequestRequest(*this,1,1);
	
}
 // Audio Manager proxy methods
void TestComponent::onSourceAvailabilityError(
	const ::boost::shared_ptr< AudioManagerProxy >& proxy, 
	const ::boost::shared_ptr< SourceAvailabilityError >& error)
{	
	DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onSourceAvailabilityError"));		
}

// handle response if required
void TestComponent::onSourceAvailabilityResponse(
	const ::boost::shared_ptr< AudioManagerProxy >& proxy,
	const ::boost::shared_ptr< SourceAvailabilityResponse >& response)
{	
	DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onSourceAvailabilityResponse"));	
} 
// handle error if required 
void TestComponent::onAudioRouteRequestError(
	const ::boost::shared_ptr< AudioManagerProxy >& proxy, 
	const ::boost::shared_ptr< AudioRouteRequestError >& error)
{	
	DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onAudioRouteRequestError"));	
}
// handle response if required
void TestComponent::onAudioRouteRequestResponse(
	const ::boost::shared_ptr< AudioManagerProxy >& proxy,
	const ::boost::shared_ptr< AudioRouteRequestResponse >& response)
{	
	DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onAudioRouteRequestResponse"));	
}   
// handle error if required 
void TestComponent::onSourceActivityResultError(
	const ::boost::shared_ptr< AudioManagerProxy >& proxy, 
	const ::boost::shared_ptr< SourceActivityResultError >& error)
{
	DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onSourceActivityResultError"));	
}
// handle response if required
void TestComponent::onSourceActivityResultResponse(
	const ::boost::shared_ptr< AudioManagerProxy >& proxy, 
	const ::boost::shared_ptr< SourceActivityResultResponse >& response)
{	
	DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onSourceActivityResultResponse"));	
	sleep(1);
	// play the song
	system("aplay  -vv -DAdevWebMainOut -c2 -r48000 -f S16_LE -d10 /home/lxc_agent_media/test.wav");
}

// handle error if required 
void TestComponent::onOnAllocateError(
	const ::boost::shared_ptr< AudioManagerProxy >& proxy, 
	const ::boost::shared_ptr< OnAllocateError >& error){
		DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onOnAllocateError"));	
	}

// handle signal if OnAllocate 	
void TestComponent::onOnAllocateSignal(
	const ::boost::shared_ptr< AudioManagerProxy >& proxy, 
	const ::boost::shared_ptr< OnAllocateSignal >& signal){
		DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onOnAllocateSignal"));	
		OnAllocateSignalRcvd=true;
		if(SourceActivityRcvd){
			//call method SourceActivityResult
			m_AMProxy->sendSourceActivityResultRequest(*this,1,1);					


		}
	}

// handle error if required 	
void TestComponent:: onSourceActivityError(
	const ::boost::shared_ptr< AudioManagerProxy >& proxy,
	const ::boost::shared_ptr< SourceActivityError >& error){
		DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onSourceActivityError"));	
	}
	
void TestComponent::onSourceActivitySignal(
	const ::boost::shared_ptr< AudioManagerProxy >& proxy,
	const ::boost::shared_ptr< SourceActivitySignal >& signal){
		DEBUG_TRACE((TEST_CONTAINER,"TestComponent::onSourceActivitySignal"));	
		SourceActivityRcvd=true;
		if(OnAllocateSignalRcvd){
			//call method SourceActivityResult
			m_AMProxy->sendSourceActivityResultRequest(*this,1,1);					


		}
	}
// calls method1 of service1 provided by native test component
void TestComponent::sendMountNavishareReq()
{
		DEBUG_TRACE((TEST_CONTAINER,"TestComponent::sendMountNavishareReq"));	
		m_NativeService1Proxy->sendMountNavishareRequest(*this);				
}
// calls method2 of service1 provided by native test component
void TestComponent::sendUmountNavishareReq()
{
		DEBUG_TRACE((TEST_CONTAINER,"TestComponent::sendUmountNavishareReq"));
		m_NativeService1Proxy->sendUmountNavishareRequest(*this);				
}
// calls method1 of service2 provided by native test component
void TestComponent::sendService2Method1()
{
		DEBUG_TRACE((TEST_CONTAINER,"TestComponent::sendService2Method1"));
		m_NativeService2Proxy->sendService2Method1Request(*this);				
}
// create file, test case
void TestComponent::createFile(string folder, int flag)
{
		DEBUG_TRACE((TEST_CONTAINER,"TestComponent::createFile"));
		FILE *fp; 
		string file = folder + TEST_CONTAINER_TEST_FILE;
		
		fp = fopen(file.c_str(), "w");
		fprintf(fp, "%s", "Test File");		
		if(flag){
			 DEBUG_TRACE((TEST_CONTAINER,"TestComponent::createFile close file handle"));
			 fclose(fp);	
		}
		else{
			 DEBUG_TRACE((TEST_CONTAINER,"TestComponent::createFile don't close file handle"));			
		}    				
}
void TestComponent::testSessionSSC(){
    DEBUG_TRACE((TEST_CONTAINER,"TestComponent::testSessionSCC"));
    std::map< std::string, asf::dbus::DBusVariant > settings;
    m_ManagerProxy->sendCreateSessionRequest(*this, settings, CNOTIFICATION_BUS_ADDRESS);    
 }

void TestComponent::fetch_page(){
   DEBUG_TRACE((TEST_CONTAINER,"TestComponent::fetch_page"));
   CURL *curl;
   CURLcode res;
 
   curl = curl_easy_init();
   if(curl) {
     curl_easy_setopt(curl, CURLOPT_URL, "http://example.com");
     /* example.com is redirected, so we tell libcurl to follow redirection */ 
     curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1L);
 
     /* Perform the request, res will get the return code */ 
     res = curl_easy_perform(curl);
     /* Check for errors */ 
     if(res != CURLE_OK)
       DEBUG_TRACE((TEST_CONTAINER,"TestComponent::testSession_fetch_page failed to fetch page %s", curl_easy_strerror(res)));
     else
       DEBUG_TRACE((TEST_CONTAINER,"TestComponent::testSession_fetch_page success"));
      /* always cleanup */ 
     curl_easy_cleanup(curl);
	}	
}

void TestComponent::sendLcmRegisteration(){
	DEBUG_TRACE((TEST_CONTAINER,"TestComponent::sendLcmRegisteration"));
	
	std::map< std::string, asf::dbus::DBusVariant > credentials;
	asf::dbus::DBusVariant agent_name;
	
	DBusMessageIter* iter = agent_name.getWriteIterator(); 		
	std::string agent_name_reg("agent_media");	
    dbus_message_iter_append_basic(iter,DBUS_TYPE_STRING,&agent_name_reg);      
    credentials.insert(std::pair<string,asf::dbus::DBusVariant>("Label",agent_name));  
	
	uint32 request_mode = shutdown_normal;
	//sending registeration for LifeCycleRequest signal, before registration
	m_LCMProxy->sendLifecycleRequestRegister(*this);
	// registering with LCM
	m_LCMProxy->sendRegisterRequest(*this, credentials, request_mode);
}

// notify to test socket server
void TestComponent::notifyTestServer(std::string msg){	

	int lxc_sock;	
	struct sockaddr_un address;
	int size;	

	if((lxc_sock = socket (AF_UNIX, SOCK_STREAM, 0)) > 0)		
		DEBUG_TRACE((TEST_CONTAINER,"TestComponent::notifyTestServer : Socket is created\n"));	
	else	
		return;
	
	address.sun_family = AF_UNIX;
	strcpy(address.sun_path, TEST_CONTAINER_SOCKET);
	
	if (connect (lxc_sock, (struct sockaddr *) &address, sizeof (address)) == 0) {
		DEBUG_TRACE((TEST_CONTAINER,"TestComponent::notifyTestServer : sends message::%s is executed", msg.c_str()));	
		send(lxc_sock, msg.c_str(), msg.length(), 0);
		close (lxc_sock);
    }
    else 
		DEBUG_TRACE((TEST_CONTAINER,"TestComponent::notifyTestServer : Unable to connect"));			
}
// execs the binary provided 
void TestComponent::execSlave(std::string bin_path)
{		
	pid_t parent = getpid();
	pid_t pid = fork();	
	
	DEBUG_TRACE((TEST_CONTAINER,"TestComponent::execSlave, bin:%s",bin_path.c_str()));			

	if (pid == -1)
	{		
		DEBUG_TRACE((TEST_CONTAINER,"TestComponent::execSlave : Fork Error"));			
		return ;
	} 
	else if (pid > 0)
	{
		int status;
		// wait for slave to exit
		waitpid(pid, &status, 0);					
		notifyTestServer(bin_path.c_str());		
	}
	else 
	{		
		char *argv[] = {strdup(bin_path.c_str()), strdup("-n"), strdup("agent_media"), 0};			
		// exec, the slave
		if(bin_path != TEST_CASE_BIN_ISORESTART){				
			argv[1] = NULL;
			// currently, only isorestart requires, additional parameters out of provided cases
			// TODO:: write a switch,to handle all such cases
		}		
		char *envp[] =
		{
			strdup(TEST_CONTAINER_SLAVE_HOME),
			strdup(TEST_CONTAINER_SLAVE_PATH),
			strdup(TEST_CONTAINER_SLAVE_USER),
			strdup(TEST_CONTAINER_SLAVE_LOG),
			NULL
		};	
		
		execve(argv[0], &argv[0], envp);
		DEBUG_TRACE((TEST_CONTAINER,"TestComponent::execSlave : Exec Failure in LXC !"));						
	}	
}

int TestComponent::uIntDLTTraceInjectionCallback(unsigned int service_id, void *data, unsigned int length)
{
	DEBUG_TRACE((TEST_CONTAINER,"Entered inside the uIntDLTTraceInjectionCallback"));
	return 0;
}
TestComponent::~TestComponent()
{  	
	DLT_UNREGISTER_CONTEXT(TEST_CONTAINER);
	DLT_UNREGISTER_APP();
}

}  //namespace com
}  //namespace bosch 

