/**
 * TODO: Add fancy header
 */

// c99 headers
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <limits.h>
#include <errno.h>

// posix headers
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <pwd.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sysexits.h> /* Well, rather BSD than POSIX */

// linux specific
#include <sys/prctl.h>

// libraries
#include <lxc/lxccontainer.h>
#include <systemd/sd-daemon.h>

// local headers
#include "lcm_defines.h"

#define STAT_RECV_BUFLEN 128 // Some room for arbitrary STATUS="..." messages

LCM_DECLARE_CONTEXT(IRUN_CTX1);

int DBG_LEVEL;
//char* __dbgstr;
FILE *stream = NULL;

static void print_usage(const char *name)
{
    fprintf(stderr, "usage: %s -n LXC_NAME -u LXC_USER -i LXC_DLTID [-s[<strsize>]] [-d] [-h] [-v]\n", name);
}

static void print_ver(const char *name)
{
    fprintf(stderr, "%s - built: %s %s, designed: %s\n", name, __DATE__, __TIME__, DESIGN_COMPLETED);
}

void sighandler(int signo)
{
    switch (signo)
    {
        case SIGTERM:
        case SIGABRT:
            exit(0);
            break;
        default:
            break;
   }
}

#ifdef LCM_ENABLE_LOG
static int checkRestarts(char *path)
{
    int count = 0;
    FILE *fp = fopen( path, "r");
    if( fp == NULL ) {
        DBG(DBG_ERR, "Error opening %s\n", path);
        DBG_PERROR("fopen");
        return count;
    }
    char buffer[8];
    while (fgets(buffer, 8, fp)) {
        count++;
    }
    if (ferror(fp)) {
        DBG(DBG_ERR, "Error reading from %s\n", path);
    }
    fclose(fp);
    return count;
}
#endif

void setupLogging(void)
{
#ifdef LCM_ENABLE_LOG
    char *isorun_logfile = getenv("ISORUN_LOGFILE");

    if ((isorun_logfile == NULL) || (strlen(isorun_logfile) == 0))
    {
        DBG_SETOUTPUT(stdout);
    }
    else
    {
        FILE *fp = fopen(isorun_logfile, "a");
        if (fp != NULL)
        {
            //stream = fp; // same as
            DBG_SETOUTPUT(fp);
        }
        else {
            DBG_SETOUTPUT(stdout);
        }
    }

    char *isorun_loglevel = getenv("ISORUN_LOGLEVEL");

    if ((isorun_loglevel == NULL) || (strlen(isorun_loglevel) == 0))
    {
        DBG_SET_LEVEL(DBG_WARN);
    }
    else {
        long int val = 0;
        val = strtol(isorun_loglevel, NULL, 10);
        DBG_SET_LEVEL((int)val);
    }
#endif
}


int isorun_main(int argc, char *argv[])
{
    signal(SIGABRT, sighandler);
    signal(SIGTERM, sighandler);

    if (argc == 1)
    {
        print_usage(argv[0]);
        return -1;
    }

    char *lxc_name = getenv("LXC_NAME");
    char *lxc_user = getenv("LXC_USER");
    char *dltid = getenv("LXC_DLTID");
    char *lxc_path = getenv("LXC_PATH");

    if (lxc_path == NULL)
    {
        lxc_path = "/var/lib/lxc";
    }

    int o;
    int strace = 0;
    int redirect = 0;
    int os_container = 0;
    char *strsize = "32";

    while ((o = getopt(argc, argv, "n:u:i:s::hdvo")) != -1)
    {
        switch (o)
        {
            case 'n':
                lxc_name = optarg;
                break;
            case 'u':
                lxc_user = optarg;
                break;
            case 'i':
                dltid = optarg;
                break;
            case 'h':
                print_usage(argv[0]);
                return 0;
            case 'v':
                print_ver(argv[0]);
                return 0;
            case 's':
                strace = 1;
                if (optarg)
                    strsize = optarg;
                break;
            case 'd':
#ifdef LCM_ENABLE_LOG
                redirect = 1;
#endif
                break;
            case 'o':
                os_container = 1;
                break;
            default:
                // error
                break;
        }
    }

    if (lxc_name == NULL)
    {
        fprintf(stderr, "Missing LXC_NAME\n");
        return -1;
    }
    else if (lxc_user == NULL)
    {
        fprintf(stderr, "Missing LXC_USER\n");
        return -1;
    }
    else if (dltid == NULL)
    {
        fprintf(stderr, "Missing LXC_DLTID\n");
        return -1;
    }

    setupLogging();

    DBGF(DBG_DBG, "lxc_name: '%s'  lxc_user: '%s'  dltid: '%s'\n", lxc_name, lxc_user, dltid);

#ifdef LCM_ENABLE_LOG
    // Count number of restarts
    // systemd service file writes a line to failcnt.txt every restart on failure
    char restartfilepath[80];
    snprintf(restartfilepath, 80, "/tmp/shared/%s_lcm/failcnt.txt", lxc_name);
    int restarts = checkRestarts(restartfilepath);
    DBGF(DBG_DBG, "Isolation has restarted %d time(s)\n", restarts);
#endif
    char *watchdog_usec = getenv("WATCHDOG_USEC");
    int wdtime_secs = 0;
    if ((watchdog_usec != NULL) && (strlen(watchdog_usec) > 0))
    {
        char *endptr;
        long int val;
        errno = 0;
        val = strtol(watchdog_usec, &endptr, 10);

        /* Check for various possible errors */
        if ((errno == ERANGE && (val == LONG_MAX || val == LONG_MIN))
             || (errno != 0 && val == 0))
        {
            DBG_PERROR("strtol");
            val = 0;
        }
        if (val < 0) {
            DBGF(DBG_WARN, "WATCHDOG_USEC < 0\n");
        }
        if (endptr == watchdog_usec) {
            DBGF(DBG_WARN, "WATCHDOG_USEC: No digits were found\n");
            val = 0;
        }
        if (*endptr != '\0') {
            DBGF(DBG_WARN, "WATCHDOG_USEC: characters after number: %s\n", endptr);
            val = 0;
        }
        if (val > 0) {
            wdtime_secs = (int) (val / 1000000);
            DBGF(DBG_DBG, "Setting watchdog timer to %ds from WATCHDOG_USEC=%s\n", wdtime_secs, watchdog_usec);
        }
    }

    pid_t npid = fork();
    if ( npid < 0 )
    {
        DBG_PERROR("fork");
        return -1;

    }
    else if (npid == 0)
    {
        /* Child */
        if (prctl(PR_SET_NAME, "isorun_child", NULL, NULL, NULL) < 0)
        {
            DBG_PERROR("prctl");
        }

        struct lxc_container *c = lxc_container_new(lxc_name, lxc_path);
        if (c == NULL)
        {
            DBGF(DBG_ERR, "Failed creating container '%s'\n", lxc_name);
            return -1;
        }

        // Build DLT id for isoinit, append I at the end of passed 3 bytes, ignore other bytes beyond first three
        char dltid_i[5] = {0};
        strncpy(dltid_i, dltid, 3);
        strcat(dltid_i,"I");

        char **priv_argv;

        char stracefilepath[80];
        snprintf(stracefilepath, 80, "/tmp/shared/debug/%s-strace.txt", lxc_name);
        char cWDTIME[10];
        snprintf(cWDTIME, 10, "%d", wdtime_secs);
        unsigned int arguments;
        if (strace)
        {
            DBGF(DBG_INFO, "Using strace with string size: %s, output in: %s\n", strsize, stracefilepath);
            arguments = 16;
            if (redirect == 1) {
                arguments = 17;
            }
            priv_argv = (char**)malloc(arguments * sizeof(char*));
            priv_argv[0] = "/usr/bin/strace";
            priv_argv[1] = "-f";
            priv_argv[2] = "-v";
            priv_argv[3] = "-y";
            priv_argv[4] = "-s";
            priv_argv[5] = strsize;
            priv_argv[6] = "-o";
            priv_argv[7] = stracefilepath;
            priv_argv[8] = "/opt/bosch/lxc/isoinit";
            priv_argv[9] = "-u";
            priv_argv[10] = lxc_user;
            priv_argv[11] = "-i";
            priv_argv[12] = dltid_i;
            priv_argv[13] = "-w";
            priv_argv[14] = cWDTIME;
            if (redirect == 1) {
                priv_argv[15] = "-d";
                priv_argv[16] = NULL;
            }
            else {
                priv_argv[15] = NULL;
            }
        }
        else
        {
            arguments = 8;
            if (redirect == 1) {
                arguments = 9;
            }
            priv_argv = (char**)malloc(arguments * sizeof(char*));
            priv_argv[0] = "/opt/bosch/lxc/isoinit";
            priv_argv[1] = "-u";
            priv_argv[2] = lxc_user;
            priv_argv[3] = "-i";
            priv_argv[4] = dltid_i;
            priv_argv[5] = "-w";
            priv_argv[6] = cWDTIME;
            if (redirect == 1) {
                priv_argv[7] = "-d";
                priv_argv[8] = NULL;
            }
            else {
                priv_argv[7] = NULL;
            }
        }

        DBGF(DBG_DBG, "Starting container '%s'\n", lxc_name);

        /**
         * Do we really need lxc-init ?
         *
         * start() blocks... Why?
         */

        int useinit = 1;

	if(os_container)
	{
                if (c->start(c, 0, NULL) == false)
                {
                    DBGF(DBG_DBG, "Failed starting container '%s'\n", lxc_name);
                    return -1;
                }	
	}
	else
	{           
                c->want_daemonize(c, false);
                if (c->start(c, useinit, priv_argv) == false)
                {
                    DBGF(DBG_DBG, "Failed starting container '%s'\n", lxc_name);
                    return -1;
                }
	}

        //DBGF(DBG_DBG, "container started\n");

        if (c->is_running(c))
        {
            DBGF(DBG_DBG, "Container '%s' seems to be running\n", lxc_name);
        }

        DBGF(DBG_DBG, "Container seems to be stopped\n");
        //DBGF(DBG_DBG, "TODO: What to do here? Let's die!\n");

        /**
         * TODO: What to do here?
         */
        return 0;
    }
    else
    {
        /* Parent */

        // Build DLT id for isorun, append R at the end of passed 3 bytes, ignore other bytes beyond first three
        char dltid_r[5] = {0};
        strncpy(dltid_r, dltid, 3);
        strcat(dltid_r,"R");

        /* register app for DLT logging */
        LCM_REGISTER_APP(dltid_r, "Isorun Isolation monitor");
        LCM_REGISTER_CONTEXT(IRUN_CTX1, "IRU1", "Isorun DLT context1");
        DEBUG_TRACE_LEV((IRUN_CTX1, DBG_INFO, "child pid=%d\n", npid));
        int sock;
        struct sockaddr_un addr;
        addr.sun_family = AF_UNIX;

        char *sock_path = getenv("ISORUN_SOCK_PATH");
        char sock_path_tmp[255];
        if (sock_path == NULL)
        {
            DBG(DBG_INFO, "ISORUN_SOCK_PATH not set. Using default\n");
            snprintf(sock_path_tmp, 255, "/tmp/shared/%s_lcm/isorun.sock", lxc_name);
            sock_path = sock_path_tmp;
        }

        DEBUG_TRACE_LEV((IRUN_CTX1, DBG_INFO, "Using socket path: '%s'\n", sock_path));

        // create socket

        sock = socket(AF_UNIX, SOCK_DGRAM, 0);
        if (sock < 0)
        {
            DEBUG_TRACE_PERROR((IRUN_CTX1,"socket"));
            return -1;
        }

        // TODO: Take care of different error cases
        if (unlink(sock_path) < 0)
        {
            DEBUG_TRACE_PERROR((IRUN_CTX1,"unlink"));
            DBG(DBG_WARN, "Error deleting %s\n", sock_path);
        }

        strcpy(addr.sun_path, sock_path);

        if (bind(sock, (struct sockaddr*) &addr, sizeof( addr )) < 0)
        {
            DEBUG_TRACE_PERROR((IRUN_CTX1,"bind"));
        }

        // Make socket writable for main isolation user
        // by changing the group for the socket file
        if ((lxc_user != NULL) && (strlen(lxc_user) > 0))
        {
            struct passwd *pwd;
            // Try getting GID for username
            pwd = getpwnam(lxc_user);
            if (pwd == NULL)
            {
                if (errno != 0) {
                    DEBUG_TRACE_PERROR((IRUN_CTX1,"getpwnam"));
                } else {
                    DEBUG_TRACE_LEV((IRUN_CTX1, DBG_ERR, "Problem getting GID for LXC_USER. Failed to change socket write permissions!\n"));
                }
            }
            else
            {
                DEBUG_TRACE_LEV((IRUN_CTX1, DBG_DBG, "Setting GID to %u (%s) for %s\n", pwd->pw_gid, lxc_user, sock_path));
                if (chown(sock_path, (uid_t)-1, pwd->pw_gid) < 0)
                {
                    DEBUG_TRACE_PERROR((IRUN_CTX1,"chown"));
                }
            }
        }
        else
        {
            DEBUG_TRACE_LEV((IRUN_CTX1, DBG_ERR, "LXC_USER not set. Failed to change socket write permissions!\n"));
        }

        // Make socket writable for owner and group
        if (chmod(sock_path, 0660) < 0)
        {
            DEBUG_TRACE_PERROR((IRUN_CTX1,"chmod"));
        }

        // listen to socket
        // send sd_notify on trigger

        char buf[STAT_RECV_BUFLEN] = { 0 };
        ssize_t n;
        int restarting = 0;
        for (;;)
        {
            DEBUG_TRACE_LEV((IRUN_CTX1, DBG_ALL, "Isorun, waiting to receive on socket...\n"));
            n = recv(sock, buf, STAT_RECV_BUFLEN, 0);
            // should be recvfrom() here? recv() is recvfrom() with NULL sockaddr
            if (n < 0) {
                DEBUG_TRACE_PERROR((IRUN_CTX1,"recv"));
                sleep(1);
                continue;
            }
            buf[n] = 0;
            DEBUG_TRACE_LEV((IRUN_CTX1, DBG_INFO, "Isorun received on socket: '%s'\n", buf));

            if (strcmp(buf, "READY=1") == 0)
            {
                DEBUG_TRACE_LEV((IRUN_CTX1, DBG_DBG, "Sending READY=1 to systemd (sd_notify)\n"));
                //sd_pid_notify((pid_t)0, 0, "READY=1"); //pid version works in new systemd
                sd_notify(0, "READY=1");
            }

            if (strcmp(buf, "WATCHDOG=1") == 0)
            {
                if (restarting == 0) {
                    DEBUG_TRACE_LEV((IRUN_CTX1, DBG_DBG, "Sending WATCHDOG=1 to systemd (sd_notify)\n"));
                    //sd_pid_notify((pid_t)0, 0, "READY=1"); //pid version works in new systemd
                    sd_notify(0, "WATCHDOG=1");
                }
            }

            if (strcmp(buf, "RESTART=1") == 0)
            {
                restarting = 1;
                DEBUG_TRACE_LEV((IRUN_CTX1, DBG_WARN, "    RESTART request received, stopping...\n"));
#if 0
                struct lxc_container *c2 = lxc_container_new(lxc_name, lxc_path);
                if (c2 == NULL)
                {
                    DBGF(DBG_DBG, "Failed defining container '%s'\n", lxc_name);
                }
                else
                {
                    DBGF(DBG_DBG, "Stopping container '%s' ...\n", lxc_name);
                    if (c2->stop(c2) == true)
                    {
                        LCM_UNREGISTER_CONTEXT(IRUN_CTX1);
                        LCM_UNREGISTER_APP();
                        // Cannot have "clean" exit code for systemd restart on-failure
                        return -1;
                    } else {
                        DBGF(DBG_DBG, "Failed stopping container '%s'\n", lxc_name);
                    }
                }
                if (wdtime_secs == 0) {
                    return -1;
                }
                // Failed to stop container, wait for watchdog timeout.
            }
#endif
                LCM_UNREGISTER_CONTEXT(IRUN_CTX1);
                LCM_UNREGISTER_APP();
                return EX_SOFTWARE;
            }

            char* tag = "STATUS=";
            if (strncmp(buf, tag, strlen(tag)) == 0)
            {
                DEBUG_TRACE_LEV((IRUN_CTX1, DBG_DBG, "Sending '%s' to systemd (sd_notify)\n", buf));
                sd_notify(0, buf);
            }
            sleep(1);
            fflush(0);
        }
    }
    return 0;
}
