#define _GNU_SOURCE
#include <stdlib.h>
#include <ctype.h> // for isspace()
#include <dirent.h>
#include <errno.h>
#include <limits.h> // strtol error checking
#include <signal.h>
#include <stddef.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>

#include "service.h"
#include "lcm_defines.h"

#define MAX_SRV         10
#define MAX_SRVFILE_LEN 1024
#define ENVLINE_MAX     1024

static Service mySrv[MAX_SRV];
static const char *initsockpath;
extern char *dltid;
static const char *isoinit_debug_dir = NULL;
extern int notificationsExpected;
extern int execfailed;

LCM_DECLARE_CONTEXT(IINI_FORK);
LCM_IMPORT_CONTEXT(IINI_MAIN)


/**
 * Initialize service array
 */
void SERVICE_InitSrvList(void)
{

    int n;
    for (n = 0; n < MAX_SRV; n++)
    {
        memset(&mySrv[n], 0, sizeof(Service));
    }
}

/**
 * Removes all chars of value "away" from a string
 * @param str string to operate on
 * @param away char to remove from str
 */
static void removeChar(char *str, char away)
{
    if (str == NULL) {
        return;
    }
    char *src, *dst;
    for (src = dst = str; *src != '\0'; src++) {
        *dst = *src;
        if (*dst != away) dst++;
    }
    *dst = '\0';
}

/**
 * @brief   split up a  cmd string into and array of string
 *
 * This is used to split up and command string, fetched from the service file, into an char** environ. The environ
 * array can be passed to execvlp(). /// execvp()
 * Take care to free the alocated memory after use, e.g. by using freeArgv()
 * @param   cmd - a string containing an command with multiple arguments
 * @return  An array of char*. Each char* points to an string, containing an argument. The array is terminated with NULL.
 */
static char** splitArgv(char* cmd)
{
    size_t n = 0;
    int argc = 0;
    size_t start = 0;
    char** env = (char**)malloc(20 * sizeof(char*));
    enum states { FIND_END, FIND_START } state = FIND_END;
    while (n <= strlen(cmd))
    {
        if (argc >= 20) {
            DEBUG_TRACE_LEV((IINI_FORK, DBG_ERR, "Error, too many arguments (max 20)\n"));
           break;
        }
        switch (state)
        {
        case FIND_END:
            if (*(cmd + n) == ' ' || *(cmd + n) == 0) {
                size_t len = n - start;
                //DBGF(DBG_DBG, "len is: %d\n", len);
                env[argc] = (char*) malloc(len + 1); // space for termination
                strncpy(env[argc], cmd + start, len);
                env[argc][len] = 0;
                DEBUG_TRACE_LEV((IINI_FORK, DBG_DBG, "arg[%d]='%s'\n", argc, env[argc]));
                argc++;
                start = n + 1;
                state = FIND_START;
            }
            break;

        case FIND_START:
            if (*(cmd + n) != ' ' || *(cmd + n) == 0) {
                state = FIND_END;
            }
            else {
                start++;
            }
            break;
        }

        n++;
    }
    //DBGF(DBG_DBG, "Setting arg[%d]=NULL\n", argc);
    env[argc] = NULL;
    return env;
}

/**
 * @brief   Split up a string of double quoted substrings or words separated by
 *          white space into an array of strings.
 *
 * This is used to split up a string of envorinment variables, fetched from the
 * service file, into an char** environ. The substrings can be double quoted
 * like "VAR1=val1 val2" or plain words like VAR2=val. The environ array can be
 * passed to execvlp(). /// execvp() Take care to free the alocated memory after
 * use, e.g. by using freeArgv()
 *
 * @param   envline - a string containing a string from the Environment= tag with
 *                    multiple arguments
 * @return  An array of char*. Each char* points to a string, containing an
 *          environment variable. The array is terminated with NULL.
 */
static char **splitEnvArgv(char *envline)
{
    int argc = 0;
    char **env = (char**)malloc(20 * sizeof(char*));

    char* p;
    char* start_of_word;
    int c;
    enum states { DULL, IN_WORD, IN_STRING } state = DULL;

    for (p = envline; *p != '\0'; p++)
    {
        if ((argc >= 20) && (state != DULL)) {
            DEBUG_TRACE_LEV((IINI_FORK, DBG_ERR, "Error, too many environment variables (max 20)\n"));
           break;
        }
        c = (unsigned char) *p; /* convert to unsigned char for is* functions */
        switch (state)
        {
        case DULL: /* not in a word, not in a double quoted string */
            if (isspace(c))
            {
            /* still not in a word, so ignore this char */
                continue;
            }
            /* not a space -- if it's a double quote we go to IN_STRING, else to IN_WORD */
            if (c == '"')
            {
                state = IN_STRING;
                start_of_word = p + (short)1; /* word starts at *next* char, not this one */
                continue;
            }
            /* else to IN_WORD */
            state = IN_WORD;
            start_of_word = p; /* word starts here */
            continue;

        case IN_STRING:
        /* we're in a double quoted string, so keep going until we hit a close " */
            if (c == '"')
            {
               /* word goes from start_of_word to p */
                size_t len = (size_t)p + 1 - (size_t)start_of_word;
                //DBG( DBG_DBG, "QUOTED STRING, LEN:%d, p=%i, sow=%i\n", len, (int)p, (int)start_of_word);
                env[argc] = (char*) malloc(len);
                strncpy(env[argc], start_of_word, len);
                env[argc][len - 1] = 0;       // add trailing zero
                DEBUG_TRACE_LEV((IINI_FORK, DBG_DBG, "\"env\"[%d]='%s'\n", argc, env[argc]));
                argc++;
                state = DULL; /* back to "not in word, not in string" state */
            }
            continue; /* either still IN_STRING or we handled the end above */

      case IN_WORD:
        /* TODO: Make it work without trailing whitespace */
        /* we're in a word, so keep going until we get to a space */
            if (isspace(c))
            {
                /* word goes from start_of_word to p  */
                size_t len = (size_t)p + 1 - (size_t)start_of_word;
                env[argc] = (char*)malloc(len);
                strncpy(env[argc], start_of_word, len);
                env[argc][len - 1] = 0;       // add trailing zero
                DEBUG_TRACE_LEV((IINI_FORK, DBG_DBG, "env[%d]='%s'\n", argc, env[argc]));
                argc++;
                state = DULL; /* back to "not in word, not in string" state */
            }
            continue; /* either still IN_WORD or we handled the end above */
        }
    }
    env[argc] = NULL;

    return env;
}

/**
 * @brief Frees memory of an NULL terminated array of char* and its referenced char[]
 *
 * @param argv - the array to free
 */
static void freeArgv(char** argv)
{
    if (argv == NULL)
    {
        return;
    }

    size_t n;
    for (n = 0; argv[n] != NULL; n++)
    {
        free(argv[n]);
    }
    free(argv);
}

/**
 * @brief filter function for scandir
 *
 * This function is called by scandir on each directory entry. If the entry
 * is ending with "'.service', it will pass.
 * @param *dir - dirent structure from scandir
 * @return 
 */
static int srvFilter(const struct dirent *dir)
{
    if (dir == NULL)
    {
        return 0;
    }

    size_t len = strlen(dir->d_name);
    // DBGF( DBG_DBG, " %s ", dir->d_name);
    if (len < 8)
    {
        // DBGF( DBG_DBG, " -> ret=0;\n");
        return 0;
    }

    int ret = strcmp(dir->d_name + len - 8, ".service") == 0;
    // DBGF( DBG_DBG, " -> ret=%d\n", ret);
    return ret;
}

/**
 * adds variables to the environment
 * @param env Array of strings formatted like "NAME=VALUE". Array should be
 *            NULL terminated
 */
static void modifyEnv(char **env)
{
    size_t idx = 0;

    if (env == NULL)
    {
        DBGF(DBG_DBG, "env == NULL");
        return;
    }

    char *value = NULL;
    while (env[idx] != NULL)
    {
        // strtok will alter env[idx] but we don't need it after this
        char *name = strtok_r(env[idx], "=", &value);

        removeChar(value, '\''); // Strip away any "'" from value

        if (name == NULL || value == NULL || strlen(name) == 0 || strlen(value) == 0)
        {
            // env[idx] might be changed here, should we print it?
            DEBUG_TRACE_LEV((IINI_FORK, DBG_WARN, "Corrupt environment variable found: %s\n", env[idx]));
            idx++;
            continue;
        }

        DEBUG_TRACE_LEV((IINI_FORK, DBG_DBG, "Setting environment variable -  name: %s, value: %s\n", name, value));

        if (setenv(name, value, 1) < 0)
        {
            DEBUG_TRACE_PERROR((IINI_FORK, "setenv"));
        }
        idx++;
    }
}

static void redirectSTDIO(void)
{
    if (isoinit_debug_dir == NULL)
    {
        return;
    }

    int retval = 0;
    char hostname[64] = { 0 };
    retval = gethostname(hostname, 64);
    hostname[63] = 0;

    if ((retval == 0) && (strlen(hostname) > 0))
    {
        char fname[PATH_MAX];
        snprintf(fname, PATH_MAX, "%s/%s%s", isoinit_debug_dir, hostname,
                 "-stdout.txt");

        int fd = open(fname, O_WRONLY | O_CREAT | O_APPEND, 00666);

        if (fd < 0)
        {
            DEBUG_TRACE_PERROR((IINI_FORK, "open()"));
            DEBUG_TRACE_LEV((IINI_FORK, DBG_INFO, "Failed to open %s\n", fname));
        }
        else
        {
            dup2(fd, STDOUT_FILENO);
            close(fd);
        }

        snprintf(fname, PATH_MAX, "%s/%s%s", isoinit_debug_dir, hostname,
                "-stderr.txt");

        fd = open(fname, O_WRONLY | O_CREAT | O_APPEND, 00666);

        if (fd < 0)
        {
            DEBUG_TRACE_PERROR((IINI_FORK, "open()"));
            DEBUG_TRACE_LEV((IINI_FORK, DBG_INFO, "Failed to open %s\n", fname));
        }
        else
        {
            dup2(fd, STDERR_FILENO);
            close(fd);
        }
    }
}


/**
 * @brief start a service from given commandline
 *
 * @param srv Service to start
 * @return negative on failure, zero on success
 */
static int startSrv(Service* srv)
{
    /* for(int n=0; argv[n]; n++){ */
    /*     DBG( DBG_DBG, "arg[%d]='%s'\n", n, argv[n]); */
    /* } */

    //  We'll do fork() & exec() below and need a way to find out if exec() in child process was successful.
    //  This can be achieved by creating a pipe with close-on-exec. The pipe available in other PID after fork
    //  and will be closed after successful exec() by new process image. This gives the parent a trigger
    //  and ensures either successful exec() or a returned error through the pipe.
    // https://www.codeproject.com/Articles/1200531/Executing-and-verifying-successful-process-executi
    int pd[2];                          // pipe descriptors
    int ps = pipe2(pd, O_CLOEXEC);    // create pipe with close-on-exec flag
    if (ps == -1)
    {
        DEBUG_TRACE_PERROR((IINI_MAIN, "pipe2"));
        return -1;
    }
    pid_t pid = fork();

    if (pid < 0)
    {
        // Get the DLT magic started again...
        // it stops working when fork() fails
        LCM_REGISTER_APP(dltid, "Isoinit Isolation manager");
        /* register context */
        LCM_REGISTER_CONTEXT(IINI_MAIN, "INI1", "Isoinit main CTX");

        DEBUG_TRACE_PERROR((IINI_MAIN, "fork"));
        return -1;
    }
    else if (pid != 0)
    {
        // Parent goes here after succesful fork()
        int err;        // storage for errno
        close(pd[1]); // close write end of pipe
        int n = read(pd[0], &err, sizeof(int));    // blocking read on pipe from child.
        close(pd[0]); // close read end of pipe
        if (n == 0)
        {
            // Read returned zero bytes. Write end was automatically closed, that is, exec() was successful!

            srv->pid = pid;
            DEBUG_TRACE_LEV((IINI_MAIN, DBG_INFO, "forked pid=%d\n", pid));

            if (srv->watchdogsec)
            {
                /*
                 * Set countdown timer if Type=notify
                 * For Type=forking we do not monitor watchdog but
                 * make sure it is set in the environment so that
                 * any forked process may use it.
                 * For Type=notify with NotifyAccess=all we do the same.
                 *
                 */
                if (strncmp("forking", srv->typestr, strlen(srv->typestr)) == 0)
                {
                    srv->watchdogcd = 0;
                    srv->watchdogsec = 0;
                    DEBUG_TRACE_LEV((IINI_MAIN, DBG_DBG, "Type=forking, clearing watchdog timer for child pid\n"));
                }
                else if (strncmp("notify", srv->typestr, strlen(srv->typestr)) == 0)
                {
                    if ((strlen(srv->notifyaccessstr) == strlen("all")) &&
                        (strncmp("all", srv->notifyaccessstr, strlen(srv->notifyaccessstr)) == 0))
                    {
                        srv->watchdogcd = 0;
                        srv->watchdogsec = 0;
                        DEBUG_TRACE_LEV((IINI_MAIN, DBG_DBG, "Type=notify, NotifyAccess=all, clearing watchdog timer for child pid\n"));
                    }
                }
                else { // Set watchdog timer for all other cases
                    srv->watchdogcd = srv->watchdogsec;
                }
            }

            if (strncmp("notify", srv->typestr, strlen(srv->typestr)) == 0)
            {
                /*
                 * For Type=notify we expect "READY=1" to be sent before we consider
                 * the service started.
                 * We count the total number of expected READY=1 and will count down later
                 * for every READY=1 received.
                 */
                notificationsExpected++;
                /*
                 * Set whether service has notified us about successful start.
                 * For Type=notify that does not have NotifyAccess=all the READY=1 must be sent by
                 * the child pid.
                 */
                if ((strlen(srv->notifyaccessstr) == strlen("all")) &&
                    (strncmp("all", srv->notifyaccessstr, strlen(srv->notifyaccessstr)) == 0))
                {
                /*
                 * For Type=notify with NotifyAccess=all the READY=1 is expected to be sent by
                 * a grand child pid.
                 */
                    srv->mustnotify = false;
                }
                else {
                    srv->mustnotify = true;
                }
            }
            else { // We don't need READY=1 from this pid for all other cases
                srv->mustnotify = false;
            }
        }
        else {
            // Read on pipe returned some bytes. This must be an errno, written by child. exec() did fail!
            errno = err;
            DEBUG_TRACE_PERROR((IINI_MAIN, "Parent: exec() failed with "));
            return -1;
        }
    }
    else // Child
    {

        LCM_REGISTER_APP(dltid, "Isoinit service starter");
        /* register context */
        LCM_REGISTER_CONTEXT(IINI_FORK, "INI2", "Isoinit child process");

        // split cmd and args
        char **argv = splitArgv(srv->cmdline);

        /* child */
        // by default, the parents environ is passed
        // No need to define **environ for that

        clearenv();

        char **envv = NULL;
        if (srv->envline)
        {
            envv = splitEnvArgv(srv->envline);
            modifyEnv(envv);

        }
        else
        {
            DEBUG_TRACE_LEV((IINI_FORK, DBG_INFO, "No environment to parse\n"));
        }

        if (srv->watchdogsec)
        {
            char tmp[100];
            sprintf(tmp, "%u", srv->watchdogsec * 1000000);
            setenv("WATCHDOG_USEC", tmp, 1);
        }

        if (strlen(srv->workingdirstr) > 0)
        {
            if (chdir(srv->workingdirstr) == -1) {
                int tmperrno = errno;
                DEBUG_TRACE_PERROR((IINI_FORK, "chdir()"));
                freeArgv(argv);

                close(pd[0]);     // close read end of pipe
                write(pd[1], &tmperrno, sizeof(int));   // send errno to parent
                close(pd[1]);     // close write end of pipe
                exit(-1);   // child dies here due to failure to change directory

            } else {
                DEBUG_TRACE_LEV((IINI_FORK, DBG_INFO, "Changing working directory to: %s\n", srv->workingdirstr));
            }
        }

        setenv("ISOINIT_SOCK_PATH", initsockpath, 1);

        redirectSTDIO();

        // Start the service
        DEBUG_TRACE_LEV((IINI_FORK, DBG_INFO, "Starting pid %d from cmdline %s\n", getpid(), srv->cmdline));

        // un-register DLT id before exec
        LCM_UNREGISTER_APP();

        if (execvp(argv[0], argv) < 0)
        {
            // exec() failed !. Child must die ...
            int tmperrno = errno;
            // re-register again as the exec fails
            LCM_REGISTER_APP(dltid, "Isoinit Isolation manager");
            DEBUG_TRACE_PERROR((IINI_FORK, "exec() failure"));
            freeArgv(argv);

            close(pd[0]);     // close read end of pipe
            write(pd[1], &tmperrno, sizeof(int));   // send errno to parent
            close(pd[1]);     // close write end of pipe
            exit(-1);   // child dies here due to fail on exec()
        }

        exit(-1);  // This is never reached as child executes new process image loaded by exec()
    }

    return 0;
}

/**
 * Removes NL (CR / LF) from a string
 * @param line string to operate on
 */
static void removeNL(char *line)
{
    char *nl;
    while ((nl = index(line, '\n')) != NULL)
    {
        *nl = '\0';
    }

    while ((nl = index(line, '\r')) != NULL)
    {
        *nl = '\0';
    }
}

/**
 * Parses a line for "WatchdogSec=n" and stores the value in a Service
 * @param srv Service to write watchdog value to
 * @param line Line to parse
 * @return false if parsing fails, true otherwise
 */
static bool parseWatchdogSec(Service* srv, char *line)
{
    const char tag[] = "WatchdogSec=";
    size_t tag_len = strlen(tag);

    if ((srv == NULL) || (line == NULL))
    {
        return false;
    }

    if (strncmp(line, tag, tag_len) != 0)
    {
        return false;
    }

    size_t line_len = strlen(line);

    if (line_len <= tag_len)
    {
        return false;
    }

    char *number = line + tag_len;

    removeNL(number);

    char *endptr;
    long int val;
    errno = 0;

    val = strtol(number, &endptr, 10);

    /* Check for various possible errors */

    if ((errno == ERANGE && (val == LONG_MAX || val == LONG_MIN))
            || (errno != 0 && val == 0))
    {
        DEBUG_TRACE_PERROR((IINI_MAIN, "strtol"));
        return false;
    }

    if (val < 0)
    {
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_WARN, "WatchdogSec < 0\n"));
        return false;
    }

    if (endptr == number)
    {
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_WARN, "parseWatchdogSec: No digits were found\n"));
        return false;
    }

    //printf("strtol() returned %ld\n", val);

    if (*endptr != '\0')
    {
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_WARN, "parseWatchdogSec: characters after number: %s\n",
                endptr));
        return false;
    }

    srv->watchdogsec = (unsigned) val;
    DEBUG_TRACE_LEV((IINI_MAIN, DBG_INFO, "WatchdogSec=%ld\n", val));

    return true;
}

/**
 * Parses a line for "Restart=<str>" and stores the string in a Service
 * @param srv Service to write Restart string to
 * @param line Line to parse
 * @return false if parsing fails, true otherwise
 */
static bool parseRestart(Service* srv, char* line)
{
    if ((srv == NULL) || (line == NULL))
    {
        return false;
    }

    const char tag[] = "Restart=";
    size_t tag_len = strlen(tag);
    size_t line_len = strlen(line);

    if (line_len <= tag_len)
    {
        return false;
    }

    if (strncmp(line, tag, strlen(tag)) == 0)
    {

        char* res = line + strlen(tag);
        removeNL(res);

        strncpy(srv->restartstr, res, sizeof(srv->restartstr));
        srv->restartstr[sizeof(srv->restartstr) - 1] = 0;
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_DBG, "Restart=%s\n", srv->restartstr));
        return true;
    }
    return false;
}

/**
 * Parses a line for "Type=n" and stores the string in a Service
 * @param srv Service to write Type string to
 * @param line Line to parse
 * @return false if parsing fails, true otherwise
 */
static bool parseType(Service* srv, char* line)
{
    if ((srv == NULL) || (line == NULL))
    {
        return false;
    }

    const char tag[] = "Type=";
    size_t tag_len = strlen(tag);
    size_t line_len = strlen(line);

    if (line_len <= tag_len)
    {
        return false;
    }

    if (strncmp(line, tag, tag_len) == 0)
    {

        char* res = line + tag_len;
        removeNL(res);

        strncpy(srv->typestr, res, sizeof(srv->typestr));
        srv->typestr[sizeof(srv->typestr) - 1] = 0;
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_INFO, "Type=%s\n", srv->typestr));
        return true;
    }
    return false;
}

/**
 * Parses a line for "ExecStart=<str>" and stores the cmdline in a Service
 * @param srv Service to write cmdline string to
 * @param line Line to parse
 * @return false if parsing fails, true otherwise
 */
static bool parseExecStart(Service* srv, char* line)
{
    const char tag[] = "ExecStart=";

    if ((srv == NULL) || (line == NULL))
    {
        return false;
    }

    size_t tag_len = strlen(tag);
    size_t line_len = strlen(line);

    if (line_len <= tag_len)
    {
        return false;
    }

    if (strncmp(line, tag, strlen(tag)) == 0)
    {
        // found an ExecStart line ...
        if ((srv->cmdline = (char*) malloc(1024)) == NULL)
        {
            DEBUG_TRACE_PERROR((IINI_MAIN, "malloc():"));
            return false;
        }

        if (line_len >= 1024)
        {
            DEBUG_TRACE_LEV((IINI_MAIN, DBG_WARN, "ExecStart= command line too long (max 1024)"));
            return false;
        }

        char* cmd = line + tag_len;
        removeNL(cmd);
        strcpy(srv->cmdline, cmd);
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_INFO, "CMD='%s'\n", srv->cmdline));
        return true;
    }
    return false;
}

/**
 * Parses a string for "Environment="
 * @param srv Service to write environment string to
 * @param line Line to parse
 * @return true for success, false for failure
 */
static bool parseEnvironment(Service *srv, char *line)
{
    if ((srv == NULL) || (line == NULL))
    {
        return false;
    }

    const char *tag = "Environment=";
    size_t tag_len = strlen(tag);

    if (strncmp(line, tag, tag_len) != 0)
    {
        return false;
    }

    size_t line_len = strlen(line);

    if (line_len <= tag_len)
    {
        return false;
    }

    size_t envline_end = 0;
    if (srv->envline != NULL)
    {
        // Continue to build srv->envline
        envline_end = strlen(srv->envline);
         // Insert new string after a white space
         // strcpy(srv->envline + position, " ");
         // position++;
    }
    else
    {
        if ((srv->envline = (char*)malloc(ENVLINE_MAX)) == NULL)
        {
            DEBUG_TRACE_PERROR((IINI_MAIN, "malloc"));
            return false;
        }
    }

    char *line_tuples = line + tag_len;
    size_t tuples_len = strlen(line_tuples);

    if (tuples_len + envline_end >= 1024)
    {
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_WARN, "Environment= line too long (max 1024)"));
        return false;
    }

    line_tuples[tuples_len - 1] = '\0';  // NULL terminate
    strncpy(srv->envline + envline_end, line_tuples, ENVLINE_MAX - envline_end);
    strcat(srv->envline, " "); // add whitespace, current splitter needs it
    DBGF(DBG_DBG, "srv->envline: '%s'\n", srv->envline);

    return true;
}

/**
 * Parses a string for "EnvironmentFile="
 * @param srv Service to write environment string to
 * @param line Line to parse
 * @return true for success, false for failure
 */
static bool parseEnvironmentFile(Service *srv, char *line)
{
    if ((srv == NULL) || (line == NULL))
    {
        return false;
    }

    const char *tag = "EnvironmentFile=";
    size_t tag_len = strlen(tag);

    if (strncmp(line, tag, tag_len) != 0)
    {
        return false;
    }

    size_t line_len = strlen(line);

    if (line_len <= tag_len)
    {
        return false;
    }

    char* fname = line + tag_len;
    removeNL(fname);

    FILE *fd;
    char rline[MAX_SRVFILE_LEN];

    DEBUG_TRACE_LEV((IINI_MAIN, DBG_INFO, "Reading from EnvironmentFile=%s\n", fname));

    if ((fd = fopen(fname, "r")) == NULL)
    {
        DEBUG_TRACE_PERROR((IINI_MAIN, "fopen"));
        return false;
    }

    char *bkslsh;
    int multiline;
    size_t envline_end = 0;
    while (fgets(rline, MAX_SRVFILE_LEN, fd) != NULL)
    {
        if (*rline == ';' || *rline == '#') {
            continue;
        }

        // If we find '\' at the END of the line
        // the definition is supposed to continue on next line(s)
        multiline = 0;
        if ((bkslsh = strstr(rline, "\\\n")) != NULL)
        {
            multiline = 1;
            *bkslsh = '\0';
        }

        if (srv->envline != NULL)
        {
            // Continue to build srv->envline
            envline_end = strlen(srv->envline);
        }
        else
        {
            if ((srv->envline = (char*)malloc(ENVLINE_MAX)) == NULL)
            {
                DEBUG_TRACE_PERROR((IINI_MAIN, "malloc"));
                fclose(fd);
                return false;
            }
        }

        removeNL(rline);

        if (strlen(rline) + envline_end >= 1024)
        {
            DEBUG_TRACE_LEV((IINI_MAIN, DBG_WARN, "Environment line too long (max 1024)"));
            fclose(fd);
            return false;
        }

        strncpy(srv->envline + envline_end, rline, ENVLINE_MAX - envline_end);
        // If we are concatenating multiple lines (ended with \)
        // we don't want any whitespace
        if (0 == multiline) {
            strcat(srv->envline, " "); // add whitespace, current splitter needs it
        }
        DBGF(DBG_DBG, "srv->envline: '%s'\n", srv->envline);

    }
    fclose(fd);
    return true;
}

/**
 * Parses a line for "WorkingDirectory=<path>" and stores the <path> string in a Service
 * @param srv Service to write Type string to
 * @param line Line to parse
 * @return false if parsing fails, true otherwise
 */
static bool parseWorkingDirectory(Service* srv, char* line)
{
    if ((srv == NULL) || (line == NULL))
    {
        return false;
    }

    const char tag[] = "WorkingDirectory=";
    size_t tag_len = strlen(tag);
    size_t line_len = strlen(line);

    if (line_len <= tag_len)
    {
        return false;
    }

    if (strncmp(line, tag, tag_len) == 0)
    {

        char* res = line + tag_len;
        removeNL(res);

        if (strlen(res) < sizeof(srv->workingdirstr)) {
            strncpy(srv->workingdirstr, res, sizeof(srv->workingdirstr));
            srv->workingdirstr[sizeof(srv->workingdirstr) - 1] = 0;
            DEBUG_TRACE_LEV((IINI_MAIN, DBG_INFO, "WorkingDirectory=%s\n", srv->workingdirstr));
            return true;
        } else {
            DEBUG_TRACE_LEV((IINI_MAIN, DBG_WARN, "WorkingDirectory line too long (max 128)"));
        }
    }
    return false;
}

/**
 * Parses a line for "NotifyAccess=" and stores the string in a Service
 * @param srv Service to write NotifyAccess string to
 * @param line Line to parse
 * @return false if parsing fails, true otherwise
 */
static bool parseNotifyAccess(Service* srv, char* line)
{
    if ((srv == NULL) || (line == NULL))
    {
        return false;
    }

    const char tag[] = "NotifyAccess=";
    size_t tag_len = strlen(tag);
    size_t line_len = strlen(line);

    if (line_len <= tag_len)
    {
        return false;
    }

    if (strncmp(line, tag, tag_len) == 0)
    {

        char* res = line + tag_len;
        removeNL(res);

        strncpy(srv->notifyaccessstr, res, sizeof(srv->notifyaccessstr));
        srv->notifyaccessstr[sizeof(srv->notifyaccessstr) - 1] = 0;
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_INFO, "NotifyAccess=%s\n", srv->notifyaccessstr));
        return true;
    }
    return false;
}

/**
 * get a free, unused service entry from our list
 * @return  A pointer to a Service or NULL
 */
static Service* getFreeSrv(void)
{
    int n;
    for (n = 0; n < MAX_SRV; n++)
    {
        if (mySrv[n].pid == 0)
        {
            return &mySrv[n];
        }
    }
    return NULL;
}

/**
 * Remove service entry from our list
 * @param  pid  Pid of service to remove
 * @return  Positive on success, zero on failure
 */
int SERVICE_RemoveSrv(pid_t pid)
{
    size_t n;
    int found = 0;
    for (n = 0; n < MAX_SRV; n++)
    {
        if (mySrv[n].pid == pid)
        {
            found = 1;
            //DBGF( DBG_DBG, "Service with PID=%d found in list!\n", pid);

            if (mySrv[n].cmdline)
            {
                free(mySrv[n].cmdline);
            }

            if (mySrv[n].envline)
            {
                free(mySrv[n].envline);
            }

            memset(&mySrv[n], 0, sizeof(Service));
        }
    }
    if (found == 0)
    {
        //DBGF( DBG_ERR, "Error, service with PID=%d not found in list!\n", pid);
    }

    return found;
}

/**
 * Add grand child service to our list
 * @param pid PID to add
 * @param wdsec Watchdog seconds
 * @param type Service type (e.g. "notify")
 * @return Zero if adding was unsuccessful. Return 1 if successful. Return
 *         negative on error
 */
int SERVICE_AddSrv(int pid, unsigned int wdsec, const char *type)
{
    if (type == NULL)
    {
        return -1;
    }

    size_t n;
    for (n = 0; n < MAX_SRV; n++)
    {
        if (mySrv[n].pid == pid)
        {
            DBGF( DBG_DBG, "Grandchild with PID=%d already known, new watchdog value: %d\n", pid, wdsec);
            mySrv[n].watchdogcd = wdsec;
            mySrv[n].watchdogsec = wdsec;
            return 0;
        }
    }

    Service* srv;

    if ((srv = getFreeSrv()) == NULL)
    {
        // error, service list is full
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_ERR, "Error, service list full!\n"));
        return -1;
    }

    DEBUG_TRACE_LEV((IINI_MAIN, DBG_INFO, "Grandchild with PID=%d added, watchdog value: %d\n", pid, wdsec));

    srv->pid         = pid;
    srv->mustnotify  = false;
    srv->watchdogsec = wdsec;
    srv->watchdogcd  = wdsec;
    srv->envline     = NULL;
    srv->cmdline     = NULL;
    strncpy(srv->typestr, type, sizeof(srv->typestr));
    srv->typestr[sizeof(srv->typestr) - 1] = 0;
    srv->workingdirstr[0] = 0;
    srv->notifyaccessstr[0] = 0;

    return 0;
}

/**
 * Reset watchdog counter for service in our list
 * @param pid PID of the service to reset watchdog counter for
 */
// TODO: Add one more if we always decrease by one directly after?
void SERVICE_ResetSrvWatchdog(int pid)
{
    size_t n;
    for (n = 0; n < MAX_SRV; n++)
    {
        if (mySrv[n].pid == pid)
        {
            DBGF(DBG_DBG, "Service with PID=%d found in list!\n", pid);
            mySrv[n].watchdogcd = mySrv[n].watchdogsec;
        }
    }
}

/**
 * Check if at least one service has expired watchdog timer
 * @return 1 if a watchdog counter has expired, 0 otherwise
 */
int SERVICE_ChkExpiredWatchdog(void)
{
    size_t n;
    for (n = 0; n < MAX_SRV; n++)
    {
        if (mySrv[n].pid != 0 && mySrv[n].watchdogsec > 0)
        {
            DBGF( DBG_DBG, "Service with PID=%d checked...\n", mySrv[n].pid);
            if (mySrv[n].watchdogcd == 0)
            {
                DBGF( DBG_DBG, "Service with PID=%d has not checked in in time!...\n", mySrv[n].pid);
                return 1;
            }
        }
    }
    return 0;
}

/**
 * Count down all running watchdog timers
 */
void SERVICE_CountDownWatchdogs(void)
{
    size_t n;
    for (n = 0; n < MAX_SRV; n++)
    {
        if (mySrv[n].pid != 0 && mySrv[n].watchdogcd > 0)
        {
            mySrv[n].watchdogcd--;
        }
    }
}

/**
 * Check type of service
 * @param  pid  Pid of service to check
 * @param  type  string with type to compare
 * @return  Positive if service is of type, zero otherwise
 */
int SERVICE_ChkSrvType(pid_t pid, const char *type)
{
    size_t n;

    if (type == NULL)
    {
        return 0;
    }

    for (n = 0; n < MAX_SRV; n++)
    {
        if (mySrv[n].pid == pid)
        {
            size_t a_len = strlen(mySrv[n].typestr);
            size_t b_len = strlen(type);
            if ((a_len == b_len) && (a_len < sizeof(mySrv->typestr)))
            {
                if (strncmp( type, mySrv[n].typestr, strlen(mySrv[n].typestr)) == 0)
                {
                    return 1;
                }
            }
            break;
        }
    }
    return 0;
}

/**
 * Check if we take action when a service died
 * @param  pid  Pid of service to check
 * @return  Positive if we take action when service has died, zero otherwise
 */
int SERVICE_ChkSrvDiedAction(pid_t pid)
{
    size_t n;

    for (n = 0; n < MAX_SRV; n++)
    {
        if (mySrv[n].pid == pid)
        {
            if (strncmp("simple", mySrv[n].typestr, strlen(mySrv[n].typestr)) == 0)
            {
                return 1;
            }
            if (strncmp("notify", mySrv[n].typestr, strlen(mySrv[n].typestr)) == 0)
            {
                if ((strlen(mySrv[n].notifyaccessstr) == strlen("all")) &&
                    (strncmp("all", mySrv[n].notifyaccessstr, strlen(mySrv[n].notifyaccessstr)) == 0))
                {
                    return 0;
                }
                else
                {
                    return 1;
                }
            }
        }
        break;
    }
    return 0;
}

/**
 * Check if service is on watchdog timer
 * @param pid PID to search for
 * @return 1 if PID is found and has a watchdog value, 0 otherwise
 */
int SERVICE_ChkSrvWatchdog(pid_t pid)
{
    size_t n;

    for (n = 0; n < MAX_SRV; n++)
    {
        if (mySrv[n].pid == pid)
        {
            if (mySrv[n].watchdogsec > 0)
            {
                return 1;
            }
            break;
        }
    }

    return 0;
}

/**
 * READY=1 was received from a PID
 *
 * @param  pid  Pid of service
 *
 */
void SERVICE_ReadyReceived(pid_t pid)
{
    if (notificationsExpected > 0) {
        notificationsExpected--;
    }
    size_t n;
    for (n = 0; n < MAX_SRV; n++)
    {
        if (mySrv[n].pid == pid)
        {
            DBGF(DBG_DBG, "Service with PID=%d found in list!\n", pid);
            mySrv[n].mustnotify = false;
        }
    }
}

/**
 * Check if at least one service has not yet sent READY=1
 *
 * @return 1 if we have outstanding READY=1, 0 otherwise
 */
int SERVICE_ChkWaitNotify(void)
{
    size_t n;
    for (n = 0; n < MAX_SRV; n++)
    {
        if (mySrv[n].pid != 0)
        {
            DBGF( DBG_DBG, "Service with PID=%d checked...\n", mySrv[n].pid);
            if (mySrv[n].mustnotify == true)
            {
                DBGF( DBG_DBG, "Service with PID=%d has not sent READY=1\n", mySrv[n].pid);
                return 1;
            }
        }
    }
    return 0;
}

/**
 * Parse service file and start service
 * @param  fname  filename of service to start
 */
static void parseSrvFile(const char *fname)
{
    int validSrvFile = 0;
    Service* srv;
    FILE *fd;
    char line[MAX_SRVFILE_LEN];

    if( (srv = getFreeSrv()) == NULL )
    {
        // error, service list is full
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_ERR, "Error, service list full!\n"));
        return;
    }

    DEBUG_TRACE_LEV((IINI_MAIN, DBG_INFO, "file=%s\n", fname));

    if ((fd = fopen(fname, "r")) == NULL)
    {
        DEBUG_TRACE_PERROR((IINI_MAIN, "fopen"));
        return;
    }

    while (fgets(line, MAX_SRVFILE_LEN, fd) != NULL)
    {
        //DBGF(DBG_DBG, "line: %s", line);

        if ( parseExecStart( srv, line ) )
        {
            validSrvFile = 1;
            continue;
        }
        else if (parseEnvironment(srv, line))
            continue;
        else if (parseEnvironmentFile(srv, line))
            continue;
        else if(parseRestart(srv, line))
            continue;
        else if(parseType(srv, line))
            continue;
        else if(parseWatchdogSec(srv, line))
            continue;
        else if (parseWorkingDirectory(srv, line))
            continue;
        else if(parseNotifyAccess(srv, line))
            continue;
    }

    if (validSrvFile)
    {
        if (startSrv(srv) < 0)
        {
            DEBUG_TRACE_LEV((IINI_MAIN, DBG_ERR, "Error, failed to start service from file %s\n", fname));
            // Set external parameter execfailed for isoinit main loop
            execfailed = 1;
        }
    } else {
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_ERR, "Error, unsuccessful parsing of service file %s\n", fname));
    }

    fclose(fd);

}

void SERVICE_SetIsoinitSockPath(const char *path)
{
    initsockpath = path;
}

void SERVICE_SetDbgPath(const char *dbgpath)
{
    isoinit_debug_dir = dbgpath;
}

int SERVICE_ScanSrvDir(const char *dir)
{
    struct dirent **namelist;
    int num_entries;

    if (dir == NULL)
    {
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_FATAL, "dir = NULL\n"));
        return -1;
    }

    // get a filtered and sorted list of dir
    num_entries = scandir( dir, &namelist, srvFilter, alphasort);

    if (num_entries < 0)
    {
        DEBUG_TRACE_PERROR((IINI_MAIN, "scandir"));
        return -1;
    }

    int idx = 0;

    while (idx < num_entries)
    {
        char fname[PATH_MAX];

        DBGF(DBG_DBG, "\t%s\n", namelist[idx]->d_name);

        snprintf(fname, PATH_MAX, "%s/%s", dir, namelist[idx]->d_name);
        parseSrvFile(fname);
        free(namelist[idx]);
        idx++;
    }
    free(namelist);

    return 0;
}

