/**
 * TODO: Add fancy header
 */

#ifndef _GNU_SOURCE
#define _GNU_SOURCE     // needed for SCM_CRED
#endif

// c99 headers
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

// posix headers
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <signal.h>
#include <pwd.h>
#include <grp.h>
#include <limits.h>
#include <unistd.h>
#include <sys/time.h>

// local headers
#include "service.h"
#include "lcm_defines.h"

#define SOCK_INIT          "isoinit.sock"
#define SOCK_RUN           "isorun.sock"
#define SOCK_PATH_INIT     "/tmp/" SOCK_INIT
#define SOCK_PATH_RUN      "/tmp/" SOCK_RUN
#define SRV_DIR            "/etc/isoinit.d/"
#define ISOINIT_DEBUG_DIR  "/tmp/shared/debug"
#define RECV_BUFLEN 1024
#define MAX_PATH_LEN     108 // sizeof(sockaddr_un.sun_path)

LCM_DECLARE_CONTEXT(IINI_MAIN);

// Set socket paths to default, will most likely be changed later
static char sock_path_init[MAX_PATH_LEN] = SOCK_PATH_INIT;
static char sock_path_run[MAX_PATH_LEN]  = SOCK_PATH_RUN;
static int isorunSock;
static struct sockaddr_un isorunAddr;
int execfailed = 0;
static char *lxc_user;
char *dltid;

static int tickledog = 0;
static int maddog = 0;
static int wd_sec = 0;
static int redirect = 0;
struct itimerval timer;
static int isoinitSock;
static const char *isoinit_debug_dir = ISOINIT_DEBUG_DIR;

int DBG_LEVEL;
char* __dbgstr;
FILE *stream = NULL;
static int sentnotify = 0;
int notificationsExpected = 0;

static int isorunOpen()
{
    // open socket to isorun
    isorunSock = socket(AF_UNIX, SOCK_DGRAM, 0);

    if (isorunSock < 0)
    {
        DEBUG_TRACE_PERROR((IINI_MAIN,"socket"));
        return -1;
    }

    isorunAddr.sun_family = AF_UNIX;
    DEBUG_TRACE_LEV((IINI_MAIN, DBG_INFO, "Using socket path to outside: '%s'\n", sock_path_run));
    strncpy(isorunAddr.sun_path, sock_path_run, sizeof(isorunAddr.sun_path) - 1);
    isorunAddr.sun_path[sizeof(isorunAddr.sun_path) - 1] = 0;
    return 0;
}

static void isorunSendStartupNotify()
{
    if (sentnotify == 0) {
        char buf[]="READY=1";
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_INFO, "Sending to socket: %s\n", buf));
        ssize_t res = sendto(isorunSock, buf, sizeof(buf), 0, (struct sockaddr*) &isorunAddr, sizeof(isorunAddr));
        if (res == -1)
        {
            DEBUG_TRACE_PERROR((IINI_MAIN,"sendto"));
        }
        else {
            sentnotify = 1;
        }
    }
}

static void isorunSendRestart()
{
    char buf[]="RESTART=1";
    DEBUG_TRACE_LEV((IINI_MAIN, DBG_INFO, "Sending to socket: %s\n", buf));
    ssize_t res = sendto(isorunSock, buf, sizeof(buf), 0,(struct sockaddr*) &isorunAddr, sizeof(isorunAddr));
    if (res == -1)
    {
        DEBUG_TRACE_PERROR((IINI_MAIN,"sendto"));
    }
}

static void isorunSendWatchdog()
{
    DEBUG_TRACE_LEV((IINI_MAIN, DBG_INFO, "Sending watchdog message using socket %s\n", isorunAddr.sun_path));
    char buf[]="WATCHDOG=1";
    ssize_t res = sendto(isorunSock, buf, sizeof(buf), 0,(struct sockaddr*) &isorunAddr, sizeof(isorunAddr));
    if (res == -1)
    {
        DEBUG_TRACE_PERROR((IINI_MAIN,"sendto"));
    }
}

static void setWatchdogTimer()
{
    if (wd_sec > 0)
    {
        int half_time = wd_sec / 2;
        if (half_time > 0) {
            /* Configure the timer to expire after half_time seconds */
            timer.it_value.tv_sec = half_time;
			timer.it_value.tv_usec = 0;
            /* ... and every half_time seconds after that. */
            timer.it_interval.tv_sec = half_time;
			timer.it_interval.tv_usec = 0;
        }
        else { // It must be set to 1 second, use 0.5 s timer (unlikely)
            /* Configure the timer to expire after 0.5 seconds */
            timer.it_value.tv_usec = 500000;
			timer.it_value.tv_sec = 0;
            /* ... and every 0.5 seconds after that. */
            timer.it_interval.tv_usec = 500000;
			timer.it_interval.tv_sec = 0;
            DEBUG_TRACE_LEV((IINI_MAIN, DBG_WARN,
                           "Setting watchdog timer interval SHORTER THAN 1 SECOND, might not behave as expected...\n"));
        }
        setitimer (ITIMER_REAL, &timer, NULL);
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_INFO, "Setting watchdog timer interval (full seconds): %d\n", half_time));

    }
}

static void isorunSendStatus(const char* message)
{
    char buf[128] = "STATUS=";
    strncat(buf, message, sizeof(buf) - strlen(buf) - 1);
    ssize_t res = sendto(isorunSock, buf, sizeof(buf), 0,(struct sockaddr*) &isorunAddr, sizeof(isorunAddr));
    if (res == -1)
    {
        DEBUG_TRACE_PERROR((IINI_MAIN,"sendto"));
    }
}

static void print_ver(const char *name)
{
    fprintf(stderr, "%s - built: %s %s, designed: %s\n", name, __DATE__, __TIME__, DESIGN_COMPLETED);
}

int ISOINIT_CheckArgs(int argc, char *argv[])
{
    int opt;

    while ((opt = getopt(argc, argv, "u:i:w:dv")) != -1)
    {
        switch (opt)
        {
            case 'u':
                lxc_user = optarg;
                break;
            case 'i':
                dltid = optarg;
                break;
            case 'w':
                wd_sec = atoi(optarg); // Assume correct format (checked by isorun)
                break;
            case 'd':
                redirect = 1;
                break;
            case 'v':
                print_ver(argv[0]);
                break;
            default:
                break;
        }
    }

    if ((lxc_user == NULL) || (dltid == NULL))
    {
        DBGF(DBG_DBG, "usage: %s -u LXC_USER -i LXC_DLTID [-w WATCHDOG_SEC] [-d] [-v]\n", argv[0]);
        return -1;
    }

    int n = argc - 1;
    char* var = NULL;
    char buildstr[MAX_PATH_LEN] = { 0 };
    DBG_START(DBG_DBG, "self='%s', args='", argv[0]);
    while (n)
    {
        DBG_ADD(DBG_DBG, "%s, ", argv[argc - n]);

        /**
         * Fetch the path to the folder for the isoinit socket as an argument
         * on the command line. Overwrite default path to socket if found.
         * Will also overwrite isoinit socket set by buildSocketPaths().
         */
        var = strstr(argv[argc - n], "ISOINIT_SOCK_DIR=");
        if (var != NULL
                && (strlen(argv[argc - n]) > strlen("ISOINIT_SOCK_DIR=")))
        {
            var = var + strlen("ISOINIT_SOCK_DIR=");
            strncpy(buildstr, var, MAX_PATH_LEN);
            buildstr[MAX_PATH_LEN - 1] = 0;
            char last = var[strlen(var) - 1];
            if (last != '/')
            {
                strncat(buildstr, "/", 1);
            }
            strncpy(sock_path_init, buildstr, MAX_PATH_LEN);
            strncat(sock_path_init, SOCK_INIT, strlen(SOCK_INIT));
            sock_path_init[MAX_PATH_LEN - 1] = 0;

        }
        /**
         * Fetch the path to the folder for the isorun socket as an argument
         * on the command line. Overwrite default path to socket if found.
         * Will also overwrite isorun socket set by buildSocketPaths().
         */
        var = strstr(argv[argc - n], "ISORUN_SOCK_DIR=");
        if (var != NULL
                && (strlen(argv[argc - n]) > strlen("ISORUN_SOCK_DIR=")))
        {
            var = var + strlen("ISORUN_SOCK_DIR=");
            strncpy(buildstr, var, MAX_PATH_LEN);
            buildstr[MAX_PATH_LEN - 1] = 0;
            char last = var[strlen(var) - 1];
            if (last != '/')
            {
                strncat(buildstr, "/", 1);
            }
            strncpy(sock_path_run, buildstr, MAX_PATH_LEN);
            strncat(sock_path_run, SOCK_RUN, strlen(SOCK_RUN));
            sock_path_run[MAX_PATH_LEN - 1] = 0;
        }
        n--;
    }
    DBG_END(DBG_DBG, "'\n");
    return 0;
}

/**
 * Set socket path for isorun
 */
void ISOINIT_BuildSocketPaths(void)
{
    char hostname[MAX_PATH_LEN] = { 0 };

    if (gethostname(hostname, MAX_PATH_LEN) == 0)
    {
        snprintf(sock_path_run, MAX_PATH_LEN, "/tmp/shared/%s_lcm/" SOCK_RUN, hostname);
    }
}

/**
 * Sets socket path from ISOINIT_SOCK_DIR environment variable. If variable
 * is missing, the socket path defaults to "/tmp"
 * Isoinit uses this socket to communicate with the services
 */
void ISOINIT_SetSocketPath(void)
{
    char *isoinit_sock_dir = getenv("ISOINIT_SOCK_DIR");

    if ((isoinit_sock_dir != NULL) && (strlen(isoinit_sock_dir) > 0))
    {
        DBG(DBG_INFO, "Setting isolation socket dir from ISOINIT_SOCK_DIR=%s\n", isoinit_sock_dir);
        snprintf(sock_path_init, MAX_PATH_LEN, "%s/%s", isoinit_sock_dir, SOCK_INIT);
        DBG(DBG_INFO, "sock_path_init: '%s'\n", sock_path_init);
    }
}

/**
 * Gets socket path
 * Isoinit uses this socket to communicate with the services
 * @return Path to socket
 */
char *ISOINIT_GetSocketPath(void)
{
    return sock_path_init;
}

static const char *getDebugPath(void)
{
    char *debug_dir_tmp = getenv("ISOINIT_DEBUG_DIR");

    if ((debug_dir_tmp != NULL) && (strlen(debug_dir_tmp) > 0))
    {
        DBG(DBG_INFO, "Setting redirect STDOUT/STDERR folder from ISOINIT_DEBUG_DIR=%s\n", debug_dir_tmp);
        return debug_dir_tmp;
    }

    return isoinit_debug_dir;
}

/**
 * Signal handler
 */
static void handleSignal(int sig, siginfo_t *si, void *context)
{
    (void)context;

    switch (sig)
    {
        case SIGINT:
            close(isorunSock);
            exit(0);
            break;
        case SIGCHLD:
            if ((si->si_code == CLD_EXITED) || (si->si_code == CLD_KILLED))
            {
                // We only care if Type=simple or Type=notify with NotifyAccess=main/exec
                if (SERVICE_ChkSrvDiedAction(si->si_pid))
                {
                    execfailed = 1;
                }
                SERVICE_RemoveSrv(si->si_pid);

                int status, options = 0;
                pid_t p = waitpid(si->si_pid, &status, options);
                if (p < 0)
                {
                    DBG_PERROR("waitpid");
                }
            }
            break;
      case SIGALRM:
            tickledog = 1;
            break;
   }
}

void ISOINIT_InstallSignalHandler(void)
{
    struct sigaction sa;

    memset(&sa, 0, sizeof(struct sigaction));
    sa.sa_flags = SA_SIGINFO;
    sa.sa_sigaction = handleSignal;

    sigaction(SIGINT, &sa, NULL);
    sigaction(SIGCHLD, &sa, NULL);
    sigaction(SIGALRM, &sa, NULL);
}

/**
 * @param socket Incoming socket
 * @param timeout Seconds to wait
 * @return false if queue of socket is empty, true otherwise
 */
static int socketIncoming(int socket, long timeout)
{
    fd_set readfds;
    struct timeval tv;

    FD_ZERO(&readfds);
    FD_SET(socket, &readfds);

    tv.tv_sec = timeout;
    tv.tv_usec = 0;

    int ret = select(socket + 1, &readfds, NULL, NULL, &tv);
    if (ret < 0)
    {
        if (errno == EINTR)
        {
            /**
             * We were interrupted. As tv is undefined at this point, we don't
             * know how long time we spent here, let the user try again
             */
            return 0;
        }
        DBG_PERROR("select");
        return -1;

    }
    else if (ret == 0)
    {
        // timeout
        return 0;
    }

    if (FD_ISSET(socket, &readfds))
    {
        // socket is readable
        return 1;
    }

    // Something is wrong
    return -1;
}

static void list_groups(size_t num, gid_t *groups)
{
    size_t idx;

    for (idx = 0; idx < num; idx++)
    {
        errno = 0;
        struct group *gr = getgrgid(groups[idx]);
        if (gr == NULL)
        {
            if (errno)
            {
                DEBUG_TRACE_PERROR((IINI_MAIN,"getgrgid"));
            }
            else
            {
                DEBUG_TRACE_LEV((IINI_MAIN, DBG_DBG, "No match for gid %d\n", groups[idx]));
            }
        } else {
            DEBUG_TRACE_LEV((IINI_MAIN, DBG_DBG, "%zu: %d (%s)\n", idx, groups[idx], gr->gr_name));
        }
    }
}

/**
 * Change uid (and gid / supplementary groups) to what was set by ISOINIT_CheckArgs()
 */
int ISOINIT_ChangeUser(void)
{
    uid_t uid = getuid();
#ifdef LCM_ENABLE_LOG
    uid_t euid = geteuid();
    gid_t egid = getegid();
#endif
    gid_t gid = getgid();

    DBGF(DBG_DBG, "uid: %d, euid: %d, gid: %d, egid: %d\n", uid, euid, gid, egid);

    errno = 0;
    struct passwd *pwd = getpwuid(uid);
    if (pwd == NULL)
    {
        if (errno)
        {
            DEBUG_TRACE_PERROR((IINI_MAIN,"getpwuid"));
            return -1;
        }
        else
        {
            DEBUG_TRACE_LEV((IINI_MAIN, DBG_INFO, "No matching entry for uid %d\n", uid));
        }
    } else {

        DEBUG_TRACE_LEV((IINI_MAIN, DBG_DBG, "uid: %d (%s)\n", uid, pwd->pw_name));
    }

    errno = 0;
    struct group *gr = getgrgid(gid);
    if (gr == NULL)
    {
        if (errno)
        {
            DEBUG_TRACE_PERROR((IINI_MAIN,"getgrgid"));
            return -1;
        }
        else
        {
            DEBUG_TRACE_LEV((IINI_MAIN, DBG_INFO, "No matching entry for gid %d\n", gid));
        }
    } else {

        DEBUG_TRACE_LEV((IINI_MAIN, DBG_DBG, "gid: %d (%s)\n", gid, gr->gr_name));
    }
    /**
     *    ^^^^ CURRENT IDENTITY ^^^^
     *
     *    Let's change it....
     */

    errno = 0;
    struct passwd *pwd_new = getpwnam(lxc_user);
    if (pwd_new == NULL)
    {
        if (errno)
        {
            DEBUG_TRACE_PERROR((IINI_MAIN, "getpwuid"));
        }
        else
        {
            DEBUG_TRACE_LEV((IINI_MAIN, DBG_ERR, "No matching entry for user %s\n", lxc_user));
        }
        return -1;
    }

    errno = 0;
    struct group *gr_new = getgrgid(pwd_new->pw_gid);
    if (gr_new == NULL)
    {
        if (errno)
        {
            DEBUG_TRACE_PERROR((IINI_MAIN,"getgrgid"));
            return -1;
        }
        else
        {
            DEBUG_TRACE_LEV((IINI_MAIN, DBG_WARN, "No matching entry for gid %d\n", pwd_new->pw_gid));
        }
    }

    gid_t groups[64];
    int ngroups = 64;
    int i = getgrouplist(lxc_user, pwd_new->pw_gid, groups, &ngroups);
    if (i < 0)
    {
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_WARN, "%u (%s) is member of more that %d groups.\n", pwd_new->pw_uid, lxc_user, 64));
        return -1;
    }

    DEBUG_TRACE_LEV((IINI_MAIN, DBG_INFO, "Setting %d supplementary groups (max %d)\n", i, NGROUPS_MAX));

    list_groups((size_t)i, groups);

#if 0
    if (initgroups(user, pwd_new->pw_gid) < 0)
    {
        DEBUG_TRACE_PERROR((IINI_MAIN, "initgroups"));
        //return -1;
    }

#else
    if (setgroups((size_t)i, groups) < 0)
    {
        DEBUG_TRACE_PERROR((IINI_MAIN, "setgroups"));

        // Try to find the offending group
        int grcnt;
        gid_t supgroup[64];
        for (grcnt = 0; grcnt < i; grcnt++)
        {
            supgroup[grcnt] = groups[grcnt];
            DBGF(DBG_DBG, "Adding GID: %d \n", supgroup[grcnt]);
            if (setgroups((size_t)grcnt + 1, supgroup) < 0)
            {
                DEBUG_TRACE_LEV((IINI_MAIN, DBG_WARN,
                "Failed to add supplementary groups, first offending group: %d." \
                " Please check container configuration file with regards to group mapping\n",
                supgroup[grcnt]));
                break;
            }
        }
        return -1;
    }
#endif

    DEBUG_TRACE_LEV((IINI_MAIN, DBG_INFO, "Morphing into uid %u (%s), gid %d\n",
                    pwd_new->pw_uid, lxc_user, pwd_new->pw_gid));
    if (setgid(pwd_new->pw_gid) < 0 )
    {
        DEBUG_TRACE_PERROR((IINI_MAIN, "setgid"));
        return -1;
    }

    if (setuid(pwd_new->pw_uid) < 0 )
    {
        DEBUG_TRACE_PERROR((IINI_MAIN, "setuid"));
        return -1;
    }
    return 0;
}

void ISOINIT_PrintCredentials(void)
{
#if defined(LCM_ENABLE_LOG) || defined(LCM_ENABLE_DLT)
    uid_t uid = getuid();
    uid_t euid = geteuid();
    gid_t gid = getgid();
    gid_t egid = getegid();
#endif

    DEBUG_TRACE_LEV((IINI_MAIN, DBG_DBG, "uid: %d, euid: %d, gid: %d, egid: %d\n", uid, euid, gid, egid));

    gid_t list[64];
    int num = getgroups(64, list);
    if (num < 0)
    {
        DBG_PERROR("getgroups");
        return;
    }

    DEBUG_TRACE_LEV((IINI_MAIN, DBG_DBG, "I'm part of %d supplementary\n", num));

    list_groups((size_t)num, list);
}


static int addGrandChild(int pid, char *watchdog_usec)
{
    unsigned int wdsec;
    char *endptr;
    long int val;
    errno = 0;

    if (watchdog_usec == NULL)
    {
        return -1;
    }

    val = strtol(watchdog_usec, &endptr, 10);

    /* Check for various possible errors */
    if ((errno == ERANGE && (val == LONG_MAX || val == LONG_MIN))
         || (errno != 0 && val == 0))
    {
        DEBUG_TRACE_PERROR((IINI_MAIN, "strtol"));
        val = -1;
    }

    if (val == 0) {
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_DBG, "WATCHDOG_USEC = 0, assume watchdog turned off\n"));
    }
    else {

        if (val < 0) {
            DEBUG_TRACE_LEV((IINI_MAIN, DBG_WARN, "WATCHDOG_USEC < 0\n"));
        }
        // Must be more than 1 second, at least for now
        else if (val < 1000000) {
            DEBUG_TRACE_LEV((IINI_MAIN, DBG_WARN, "WATCHDOG_USEC < 1000000\n"));
            val = -1;
        }
    }

    if (endptr == watchdog_usec) {
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_WARN, "WATCHDOG_USEC: No digits were found\n"));
        val = -1;
    }
    if (*endptr != '\0') {
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_WARN, "WATCHDOG_USEC: characters after number: %s\n", endptr));
        val = -1;
    }

    if (val < 0) {
        return -1;
    }

    wdsec = (unsigned int) (val / 1000000);
    DEBUG_TRACE_LEV((IINI_MAIN, DBG_INFO, "WATCHDOG_USEC from PID %d, setting watchdog timer to %ds from WATCHDOG_USEC=%s\n", pid, wdsec, watchdog_usec));
    // Add this PID to service list, Type does not really matter for now
    return SERVICE_AddSrv(pid, wdsec, "notify");
}

static void tickleTheDog()
{
    tickledog = 0;
    // Only check if we have no other restart situation
    if (execfailed == 0)
    {
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_INFO, "Checking all watchdog timers...\n"));
        if (SERVICE_ChkExpiredWatchdog() == 0)
        {
             DEBUG_TRACE_LEV((IINI_MAIN, DBG_DBG, "All services on watchdog have checked in\n"));
             isorunSendWatchdog();
              maddog = 0;
        }
        else { // Some tardiness ongoing, panic mode!
               DEBUG_TRACE_LEV((IINI_MAIN, DBG_WARN, "Service on watchdog is late, start panicking...\n"));
               maddog = 1;
        }
    } // execfailed == 1
    else { // SIGCHLD or exec() failed
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_WARN, "At least one process has terminated unexpectedly!"));
        maddog = 0;  // No point in checking often since we are failed anyway
    }
    if (notificationsExpected)
    {
         // Check and print PID(s) that did not send READY=1
        SERVICE_ChkWaitNotify();
    }
}

void ISOINIT_SetupDebug(void)
{
#ifdef LCM_ENABLE_LOG
    char *isoinit_logfile = getenv("ISOINIT_LOGFILE");

    if ((isoinit_logfile == NULL) || (strlen(isoinit_logfile) == 0))
    {
        DBG_SETOUTPUT(stdout);
    }
    else
    {
        FILE *fp = fopen(isoinit_logfile, "a");
        if (fp != NULL)
        {
            //stream = fp; // same as
            DBG_SETOUTPUT(fp);
        }
        else {
            DBG_SETOUTPUT(stdout);
        }
    }

    char *isoinit_loglevel = getenv("ISOINIT_LOGLEVEL");

    if ((isoinit_loglevel == NULL) || (strlen(isoinit_loglevel) == 0))
    {
        DBG_SET_LEVEL(DBG_WARN);
    }
    else
    {
        long int val = 0;
        val = strtol(isoinit_loglevel, NULL, 10);
        DBG_SET_LEVEL((int)val);
    }
#endif
}

int ISOINIT_SetupSocket(void)
{
    // Set up socket for clients
    struct sockaddr_un addr;
    memset(&addr, 0, sizeof(addr));

    isoinitSock = socket(AF_UNIX, SOCK_DGRAM, 0);

    if (isoinitSock < 0)
    {
        DEBUG_TRACE_PERROR((IINI_MAIN,"socket"));
        return -1;
    }

    /**
     * Enables  the  receiving  of  the  credentials of the sending process in
     * an ancillary message.
     */
    int optval = 1;
    if (setsockopt(isoinitSock, SOL_SOCKET, SO_PASSCRED, &optval, sizeof(optval)) < 0)
    {
        DEBUG_TRACE_PERROR((IINI_MAIN,"setsockopt"));
        return -1;
    }

    unlink(sock_path_init);

    addr.sun_family = AF_UNIX;
    strncpy(addr.sun_path, sock_path_init, sizeof(addr.sun_path) - 1);
    addr.sun_path[sizeof(addr.sun_path) - 1] = 0;

    /**
     * If path of isolation internal socket starts with a '#' we
     * should use an abstract socket.
     */
    if ('#' == sock_path_init[0])
    {
        DBG(DBG_INFO, "Using abstract socket (path name begins with '#')\n");
        addr.sun_path[0] = 0;
    }

    DEBUG_TRACE_LEV((IINI_MAIN, DBG_INFO, "Using isolation socket path: '%s'\n", sock_path_init));

    if (bind(isoinitSock, (struct sockaddr*) &addr, sizeof(addr)) < 0)
    {
        DEBUG_TRACE_PERROR((IINI_MAIN,"bind"));
        return -1;
    }
    return 0;
}

int ISOINIT_ScanSrvDir(void)
{
    static const char *srv_dir;
    srv_dir = getenv("ISOINIT_SRV_DIR");

    if (srv_dir == NULL)
    {
        DBG(DBG_INFO, "ISOINIT_SRV_DIR not set. Using default\n");
        srv_dir = SRV_DIR;
    }

    DEBUG_TRACE_LEV((IINI_MAIN, DBG_INFO, "Using service file path: '%s'\n", srv_dir));
    // Redirection of STDOUT/STDERR requested (-d switch), check if
    // default save dir was changed with environment variable
    if (redirect == 1)
    {
        const char *dbgpath = getDebugPath();
        SERVICE_SetDbgPath(dbgpath);
    }

    if (SERVICE_ScanSrvDir(srv_dir) < 0)
    {
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_FATAL, "Failed scanning service dir, exiting!\n"));
        return -1;
    }
    return 0;
}

int ISOINIT_Daemonize(void)
{
    int nochdir = 0;
    int noclose = 1;

    DBGF(DBG_DBG, "daemonizing...\n");

    if (daemon(nochdir, noclose) < 0)
    {
        DBG_PERROR("daemon");
        return -1;
    }
    return 0;
}

int ISOINIT_MainLoop(void)
{
    char buf[RECV_BUFLEN] = {0};

    struct msghdr msgh;
    struct iovec iov;
    /* Set fields of 'msgh' to point to buffer used to receive (real)
       data read by recvmsg() */
    msgh.msg_iov = &iov;
    msgh.msg_iovlen = 1;        // we use only one buffer for rx
    iov.iov_base = &buf;        // point to our buffer
    iov.iov_len = sizeof(buf);
    msgh.msg_name = NULL;       // no address needed
    msgh.msg_namelen = 0;
    msgh.msg_flags = 0;

    /* Set 'control_un' to describe ancillary data that we want to receive */
    union {
        struct cmsghdr cmh;
        char   control[CMSG_SPACE(sizeof(struct ucred))];
                        /* Space large enough to hold a ucred structure */
    } control_un;
    control_un.cmh.cmsg_len   = CMSG_LEN(sizeof(struct ucred));
    control_un.cmh.cmsg_level = SOL_SOCKET;
    control_un.cmh.cmsg_type  = SCM_CREDENTIALS;

    /* Set 'msgh' fields to describe 'control_un' */
    msgh.msg_control = control_un.control;
    msgh.msg_controllen = sizeof(control_un.control);


    /*
     * Monitoring resides in here for now instead of in a separate thread
     * in service.c. Messages received on the socket will buffer so we
     * can start a bunch before going into monitoring mode. We can also
     * handle SIGCHLD without interrupting this loop.
     */


    if (isorunOpen() < 0) {
        // Failed to open socket to isorun
        return -1;
    }

    // Unless exec failed was set from failing to start a service
    // all services of type simple and forking are up and running.
    // If we started any service of type notify we have to wait for
    // READY=1 to be received, i.e., we expect notifications.

    if (execfailed == 0)
    {
        if (notificationsExpected == 0)
        {
            //  Time to inform isorun:
            isorunSendStartupNotify();
        }
    }
    else {
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_ERR, "Failed to start one or more service(s)\n"));
        return -1;
    }


    // Start monitoring watchdogs and start our own tickle signal
    // (only is we got a wd_time > 0 from isorun)
    setWatchdogTimer();

    for (;;)
    {
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_ALL, "waiting to receive on isolation socket...\n"));

        for (;;)
        {
            int readable = socketIncoming(isoinitSock, 1);
            if (readable < 0)
            {
                // this is bad
                DEBUG_TRACE_LEV((IINI_MAIN, DBG_ERR, "main is dying\n"));
                return -1;
            }
            else if (readable == 0)
            {
                //DBGF(DBG_DBG, "Waiting for readable socket\n");

                SERVICE_CountDownWatchdogs();

                if (tickledog)
                {
                    tickleTheDog();
                }
                continue;
            }
            break;
        }

        if (recvmsg(isoinitSock, &msgh, 0) < 0) {
            DEBUG_TRACE_LEV((IINI_MAIN, DBG_WARN, "Failed to receive message on socket\n"));
            sleep(1); // Sleep here instead before next round.
            continue;
        }

        DEBUG_TRACE_LEV((IINI_MAIN, DBG_DBG, "isoinit got buf: '%s'\n", buf));

        struct cmsghdr *cmhp;
        struct ucred* ucredp;
        cmhp = CMSG_FIRSTHDR(&msgh);

        if (cmhp->cmsg_type != SCM_CREDENTIALS) {
            DEBUG_TRACE_LEV((IINI_MAIN, DBG_WARN, "Failed to receive message credentials\n"));
            sleep(1); // Sleep here instead before next round.
            continue;
        }

        ucredp = (struct ucred *) CMSG_DATA(cmhp);
        DEBUG_TRACE_LEV((IINI_MAIN, DBG_DBG, "Received credentials pid=%ld, uid=%ld, gid=%ld\n",
            (long) ucredp->pid, (long) ucredp->uid, (long) ucredp->gid));

        if (strcmp(buf, "WATCHDOG=1") == 0)
        {
            SERVICE_ResetSrvWatchdog(ucredp->pid);

            // Only check if we are running late (dog will be mad)
            if ((maddog) && (execfailed == 0) && (SERVICE_ChkExpiredWatchdog() == 0))
            {
                DEBUG_TRACE_LEV((IINI_MAIN, DBG_DBG, "All services in list checked in\n"));
                isorunSendWatchdog();
                maddog = 0;
            }
            if (execfailed)
            {
                isorunSendStatus("Isoinit: WARNING, at least one process has terminated unexpectedly!");
                maddog = 0;  // No point in checking often since we are failed anyway
            }
        }

        if (strcmp(buf, "READY=1") == 0)
        {
            SERVICE_ReadyReceived(ucredp->pid);

            if (execfailed == 0 && notificationsExpected == 0 && SERVICE_ChkWaitNotify() == 0)
            {
                DEBUG_TRACE_LEV((IINI_MAIN, DBG_INFO, "All expected startup notifications received\n"));
                // Time to inform isorun:
                isorunSendStartupNotify();
            }
        }

        char *match = NULL;
        if ((match = strstr(buf, "WATCHDOG_USEC=")) != NULL)
        // We have a grand child process starting to report watchdog
        {
            char wd_usec[16];
            match = match + strlen("WATCHDOG_USEC=");
            strncpy(wd_usec, match, sizeof(wd_usec));
            wd_usec[sizeof(wd_usec) - 1] = 0;

            if (addGrandChild(ucredp->pid, wd_usec) < 0)
            {
                DBG(DBG_ERR, "addGrandChild() failed\n");
            }
        }

        if (strcmp(buf, "RESTART=1") == 0)
        {
            isorunSendRestart();
        }

        SERVICE_CountDownWatchdogs();

        if (tickledog)
        {
            tickleTheDog();
        }

        sleep(1);
        fflush(0);
    }

    close(isorunSock);
    return 0;
}


