/**
* \file sdc_session.h
*
* \brief libSDC: Definition of functions etc. for session management - <b>should not be included directly!</b>
*
* Application should not include this header directly!
*
* \author Christoph Gellner (cgellner@de.adit-jv.com)
*
* \copyright (c) 2015 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/

/**
 * \ingroup generic
 * \defgroup sessionmanagement Session management
 * Functions to manage sessions
 */

#ifndef __SDC_LIB_SESSION_H_
#define __SDC_LIB_SESSION_H_

#ifdef __cplusplus
extern "C"
{
#endif

#include "sdc.h"

/* Definitions types and defaults */

/* Functions */

/**
 * \ingroup sessionmanagement
 * \brief Session handle
 *
 * In order to start a new SDC session a new session handle needs to be created
 * using ::sdc_open_session.
 * Afterwards the session handle needs to be passed to any session related
 * operation (e.g. ::sdc_session_load_storage_key).
 * After usage it needs to be freed using ::sdc_close_session.
 *
 * For each session it is possible to execute a single cryptographic
 * operation (wrap, unwrap, encrypt, decrypt, sign, verify or
 * random number generation) at once.
 * Starting a second operation while the first is still ongoing results
 * in an error.
 *
 */
typedef struct sdc_session sdc_session_t;



/**
 * \ingroup sessionmanagement
 * \brief Create a SDC session
 *
 * Creates a new session and allocates the \ref sdc_session_t.
 * \ref sdc_session_t is needed internally to keep track of the session.
 * After usage it needs to be freed using ::sdc_close_session
 *
 * \param session   pointer to return the session handle.
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_NO_MEM - failed to allocate memory for session struct
 * \return \ref SDC_OPEN_DEVICENODE_FAILED - failed to open the libSDC kernel device
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_open_session(sdc_session_t **session);


/* Note: validity of session needs to be checked internally */
/**
 * \ingroup sessionmanagement
 * \brief Close a SDC session
 *
 * Close and free a SDC session and all associated resources.
 *
 * \param session   session handle created using ::sdc_open_session
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_close_session(sdc_session_t *session);

/* Note: validity of session needs to be checked internally */
/**
 * \ingroup sessionmanagement
 * \brief Assign the key storage key with the given ID to the session.
 *
 * This function is used to assign a key storage key with ID \p kid to the session.
 * All cryptographic operations (see \ref operations) will use this key until the
 * key is unset or replaced by a different key.
 *
 * \param session   session handle created using ::sdc_open_session
 * \param kid    ID of the key storage key
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_KEY_INVALID - given KID is invalid
 * \return \ref SDC_KID_NOT_AVAILABLE - the given KID is valid but not available in the key storage
 * \return \ref SDC_ACCESS_DENIED - the process has no permission to perform any operation using the given key
 * \return \ref SDC_KEY_TAMPERED - loading the key aborted as it seems to be tampered
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_session_load_storage_key(sdc_session_t *session, sdc_key_id_t kid);

/* Note: validity of session needs to be checked internally */
/**
 * \ingroup sessionmanagement
 * \brief Remove assigned key from session
 *
 * \todo implement function and tests
 *
 * This function is used to unset the assigned key of a session.
 * All cryptographic operations (which require a key) will fail afterwards.
 *
 * \param session   session handle created using ::sdc_open_session
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_session_unset_key(sdc_session_t *session);

#ifdef __cplusplus
}
#endif

#endif
