/**
* \file sdc_perm.h
*
* \brief libSDC: Definition of functions etc. required by to manage permissions of
* keys - <b>should not be included directly!</b>
*
* Application should not include this header directly!
*
* \author Christoph Gellner (cgellner@de.adit-jv.com)
*
* \copyright (c) 2015 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/


/**
 * \ingroup keymanagement
 * \defgroup keymanagement_perm Permission managment
 * Functions to manage permission used during key creation (see \ref keymanagement_keystore)
 */


#ifndef __SDC_LIB_PERM_H_
#define __SDC_LIB_PERM_H_

#ifdef __cplusplus
extern "C"
{
#endif

#include "sdc.h"

/* Definitions types and defaults */

/**
 * \ingroup keymanagement_perm
 * \brief Type to specify the permission and group during key creation
 *
 * <b>The elements of this struct must not be accessed directly!</b>
 * In order to allocate, free, read or modify its elements the corresponding
 * functions need to be used.
 * See ::sdc_permissions_alloc, ::sdc_permissions_free ::sdc_set_default_permissions_and_gid ::sdc_set_default_permissions_current_gid
 */
typedef struct sdc_permissions sdc_permissions_t;

/* Functions */

/**
 * \ingroup keymanagement_perm
 * \brief Allocate permissions structure
 *
 * After allocation the content of the struct needs to be filled using functions
 * like ::sdc_set_default_permissions_and_gid, ::sdc_set_default_permissions_current_gid.
 * After usage the structure needs to be freed using ::sdc_permissions_free.
 *
 * \param permissions a pointer to the structure pointer
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_permissions_alloc (sdc_permissions_t **permissions);

/**
 * \ingroup keymanagement_perm
 * \brief Free permissions structure
 *
 * Free the structure allocated by ::sdc_permissions_alloc.
 *
 * \param permissions the structure pointer
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_permissions_free (sdc_permissions_t *permissions);

/**
 * \ingroup keymanagement_perm
 * \brief Set default permissions and explicitly configured group
 *
 * Set default permissions and an explicitly configured group.
 *
 * \param permissions the structure pointer
 * \param gid group identifier of the desired group
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_set_default_permissions_and_gid (sdc_permissions_t *permissions, gid_t gid);

/**
 * \ingroup keymanagement_perm
 * \brief Set default permissions
 *
 * In contrast to ::sdc_set_default_permissions_and_gid the group is configured implicitly.
 * This function will set the group to the current effective group of the calling process.
 *
 * \param permissions the structure pointer
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_set_default_permissions_current_gid (sdc_permissions_t *permissions);

#ifdef __cplusplus
}
#endif

#endif
