/**
* \file sdc_op_conv.h
*
* \brief libSDC: Definition of functions etc. of crytographic operations (convenience API)
*
* Applications needing to perform wrap/unwrap, encrypt/decrypt or sign/verify
* in an easy way should include this header.\n
* The common operation header or generic types like error codes are included internally.
*
* \author Christoph Gellner (cgellner@de.adit-jv.com)
*
* \copyright (c) 2015 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/

/**
 * \ingroup operations_wrapunwrap
 * \defgroup operations_wrapunwrap_conv Convenience API
 * Convenience API
 *
 */
/**
 * \ingroup operations_wrapunwrap_conv
 * \defgroup operations_wrapunwrap_conv_form Formatted
 * All parameters (data,IV,...) are handled in a formatted memory buffer having
 * an proprietary format
 */
/**
 * \ingroup operations_wrapunwrap_conv
 * \defgroup operations_wrapunwrap_conv_norm Non-Formatted
 * All parameters (data,IV,...) are specified by application
 */
/**
 * \ingroup  operations_wrapunwrap_conv
 * \defgroup operations_wrapunwrap_conv_buffer_handling Buffer handling rules
 *
 * For buffers providing input data only a simple pointer is used.\n
 * In case a buffer can provide output data a pointer to a pointer is used.\n
 * \n
 * In case the value of the pointer to the actual buffer is != NULL the data is provided externally.\n
 * In case this pointer is NULL the data is internally generated. If the buffer is an input/output or an output only buffer
 * the content can be used by application after successful execution of the function.\n
 * In case the buffer is externally provided the buffer length has to be provided explicitly as well.\n
 * Some internal generated buffers require an externally specified length (e.g. IV of \ref sdc_wrap).\n
 * This specification can be done either explicitly or the architecture and type specific default can be referenced.
 * In order to use this default the special values need to be assigned to the length values (i.e. SDC_IV_USE_DEFAULT, SDC_SIG_USE_DEFAULT).\n
 * In case the length and the buffer content can not be specified externally (e.g. output data buffer of \ref sdc_wrap) the pointer to the buffer has to be NULL while the length is ignored.\n
 * \n
 * Internally generated output buffers will return length 0 and pointer NULL in case of an API error.\n
 * In case of API success the application has to free internal generated buffers after usage (e.g. using \ref sdc_plain_buffer_free, \ref sdc_cipher_buffer_free).\n
 *
 * Please note: For externally provided buffers with explicit length 0 the buffer pointer might be NULL.
 */



#ifndef __SDC_LIB_OP_CONV_H_
#define __SDC_LIB_OP_CONV_H_

#ifdef __cplusplus
extern "C"
{
#endif

#include "sdc.h"
#include "sdc_session.h"
#include "sdc_op_common.h"

/* Definitions types and defaults */

/* Functions */

/**
 * \ingroup operations_wrapunwrap_conv_norm
 * \brief Perform wrap of a memory buffer with a single call from application
 *
 * Internally the request is handled by one or multiple calls to the kernel.\n
 * In case of an error the buffer pointer and the length of output buffers is NULL/0.\n
 * This is not true for the IV in case application has specified the IV explicitly (*\p iv != NULL).
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param session   session handle created using ::sdc_open_session
 * \param type      type descriptor
 * \param in_data   pointer to the buffer containing the plain data to be wrapped (externally provided buffer)
 * \param in_len    length of the \p in_data buffer (external specified length)
 * \param aad_data  pointer to buffer containing additional authenticated data (externally provided buffer)
 * \param aad_len   length of \p aad_data \n
 *                  Currently not supported (has to be 0)\n
 *                  Reserved for future use
 * \param iv        pointer to the IV buffer pointer\n
 *                  IV can be provided externally (*\p iv != NULL) or\n
 *                  internally generated (*\p iv == NULL)\n
 *                  In both cases \p iv must not be NULL\n
 * \param iv_len    pointer to the length of the IV\n
 *                  The length has to be specified externally\n
 *                  For the internally generated IV the architecture and \p type specific default can be selected by using
 *                  *\p iv_len == SDC_IV_USE_DEFAULT. In this case *\p iv_len will contain the actual length on success\n
 *                  \p iv_len must not be NULL
 * \param out_data  pointer to return a pointer for the internally allocated buffer storing the wrapped data.\n
 *                  \p out_data must not be NULL\n
 *                  Initially *\p out_data has to be NULL!\n
 *                  Application has to free this buffer after usage
 * \param out_len   pointer to return the length of \p out_data \n
 *                  The length is determined internally - therefore the value of *\p out_len is ignored\n
 *                  \p out_len must not be NULL\n
 * \param sig_data  pointer to return a pointer for the internally allocated buffer storing the signature.\n
 *                  \p sig_data must not be NULL\n
 *                  Initially *\p sig_data has to be NULL\n
 *                  Application has to free this buffer after usage
 * \param sig_len   pointer to the length of the signature\n
 *                  The length of the signature needs to be specified externally\n
 *                  In case *\p sig_len == SDC_SIG_USE_DEFAULT the architecture and \p type specific default is used.\n
 *                  In this case *\p sig_len will contain the actual length on success\n
 *                  \p sig_len must not be NULL
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_IV_INVALID - iv data or length pointer is invalid or invalid length specified
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return \ref SDC_SIG_DATA_INVALID - signature data or length pointer is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_wrap(sdc_session_t *session,
        const sdc_wrap_unwrap_type_t *type,
        const uint8_t *in_data, const size_t in_len,
        const uint8_t *aad_data, const size_t aad_len,
        uint8_t **iv, size_t *iv_len,
        uint8_t **out_data, size_t *out_len,
        uint8_t **sig_data, size_t *sig_len);

/**
 * \ingroup operations_wrapunwrap_conv_norm
 * \brief Perform unwrap of a memory buffer with a single call from application
 *
 * Internally the request is handled by one or multiple calls to the kernel.\n
 * In case of an error the buffer pointer and the length of the output buffer is NULL/0.
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param session   session handle created using ::sdc_open_session
 * \param type      type descriptor
 * \param in_data   pointer to the buffer containing the cipher data to be unwrapped (externally provided buffer)
 * \param in_len    length of the \p in_data buffer (external specified length)
 * \param aad_data  pointer to buffer containing additional authenticated data (externally provided buffer)
 * \param aad_len   length of \p aad_data \n
 *                  Currently not supported (has to be 0)\n
 *                  Reserved for future use
 * \param iv        pointer to the IV buffer (externally provided buffer)
 * \param iv_len    length of the IV  (external specified length)\n
 * \param sig_data  pointer to the signature buffer (externally provided buffer)\n
 * \param sig_len   length of the signature  (external specified length)\n
 * \param out_data  pointer to return a pointer for the internally allocated buffer storing the unwrapped data.\n
 *                  \p out_data must not be NULL\n
 *                  Initially *\p out_data has to be NULL!\n
 *                  Application has to free this buffer after usage
 * \param out_len   pointer to return the length of \p out_data \n
 *                  The length is determined internally - therefore the value of *\p out_len is ignored\n
 *                  \p out_len must not be NULL\n
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_IV_INVALID - iv data or length pointer is invalid or invalid length specified
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return \ref SDC_SIG_DATA_INVALID - signature data or length pointer is invalid
 * \return \ref SDC_AUTHENTICATION_FAILED - failed to check the signature of the input data
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_unwrap(sdc_session_t *session,
        const sdc_wrap_unwrap_type_t *type,
        const uint8_t *in_data, const size_t in_len,
        const uint8_t *aad_data, const size_t aad_len,
        const uint8_t *iv, const size_t iv_len,
        const uint8_t *sig_data, const size_t sig_len,
        uint8_t **out_data, size_t *out_len);



/**
 * \ingroup operations_wrapunwrap_conv_form
 * \brief Perform wrap of a memory buffer and generate a formatted output buffer
 *
 * The used key for wrapping is the key currently assigned to the session.
 * This formatted output buffer contains\n
 * - key information,
 * - information on algorithm and block mode,
 * - automatically generated iv
 * - wrapped data
 * - signature information
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param session   session handle created using ::sdc_open_session
 * \param type      type descriptor
 * \param in_data   pointer to the buffer containing the plain data to be wrapped (externally provided buffer)
 * \param in_len    length of the \p in_data buffer (external specified length)
 * \param formatted_data  pointer to return a pointer for the internally allocated buffer storing the formatted wrapped data.\n
 *                  \p formatted_data must not be NULL\n
 *                  Initially *\p formatted_data has to be NULL!\n
 *                  Application has to free this buffer after usage
 * \param formatted_len   pointer to return the length of \p formatted_data \n
 *                  The length is determined internally - therefore the value of *\p formatted_len is ignored\n
 *                  \p formatted_len must not be NULL\n
 *
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_wrap_formatted(sdc_session_t *session,
        const sdc_wrap_unwrap_type_t *type,
        const uint8_t *in_data, const size_t in_len,
        uint8_t **formatted_data, size_t *formatted_len);



/**
 * \ingroup operations_wrapunwrap_conv_form
 * \brief Perform unwrap of a the formatted memory buffer and provide output data
 *
 * The used key for wrapping is the key currently assigned to the session.
 * This key might be set using \ref sdc_unwrap_formatted_autoload_key.\n
 * \n
 * Furthermore it is required by the application to extract the algorithm,
 * blockmode using \ref sdc_unwrap_formatted_extract_type, check it
 * (algorithm and block mode sufficient to fulfill its own security requirements)
 * and provide it as argument.\n
 * \n
 * IV and signature are extracted internally.
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param session   session handle created using ::sdc_open_session
 * \param type      type descriptor
 * \param formatted_data   pointer to the buffer containing the formatted data to be unwrapped (externally provided buffer)
 * \param formatted_len    length of the \p formatted_data buffer (external specified length)
 * \param out_data  pointer to return a pointer for the internally allocated buffer storing the unwrapped data.\n
 *                  \p out_data must not be NULL\n
 *                  Initially *\p out_data has to be NULL!\n
 *                  Application has to free this buffer after usage
 * \param out_len   pointer to return the length of \p out_data \n
 *                  The length is determined internally - therefore the value of *\p out_len is ignored\n
 *                  \p out_len must not be NULL\n
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return \ref SDC_AUTHENTICATION_FAILED - failed to check the signature of the input data
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_unwrap_formatted(sdc_session_t *session,
        const sdc_wrap_unwrap_type_t *type,
        const uint8_t *formatted_data, const size_t formatted_len,
        uint8_t **out_data, size_t *out_len);


/**
 * \ingroup operations_wrapunwrap_conv_form
 * \brief Try to load the key described in the formatted memory buffer
 *
 * Load the key storage key with the id described in the formatted
 * memory buffer.\n
 * Other kinds of keys will result in error!
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param session   session handle created using ::sdc_open_session
 * \param formatted_data   pointer to the buffer containing the formatted data to be unwrapped (externally provided buffer)
 * \param formatted_len    length of the \p formatted_data buffer (external specified length)
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_NOT_SUPPORTED - Key kind is not supported
 * \return \ref SDC_KEY_INVALID - given KID is invalid
 * \return \ref SDC_KID_NOT_AVAILABLE - the given KID is valid but not available in the key storage
 * \return \ref SDC_ACCESS_DENIED - the process has no permission to perform any operation using the given key
 * \return \ref SDC_KEY_TAMPERED - loading the key aborted as it seems to be tampered
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_unwrap_formatted_autoload_key(sdc_session_t *session,
        const uint8_t *formatted_data, const size_t formatted_len);

/**
 * \ingroup operations_wrapunwrap_conv_form
 * \brief Extract \ref sdc_wrap_unwrap_type_t from formatted memory buffer
 *
 * Based on the results the application has to check if its own
 * security requirements are fulfilled.\n
 * After using the type for unwrapping it needs to be freed using
 * \ref sdc_wrap_unwrap_type_free.
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param formatted_data   pointer to the buffer containing the formatted data to be unwrapped (externally provided buffer)
 * \param formatted_len    length of the \p formatted_data buffer (external specified length)
 * \param type      type descriptor
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_unwrap_formatted_extract_type(
        const uint8_t *formatted_data, const size_t formatted_len,
        sdc_wrap_unwrap_type_t **type);

#ifdef __cplusplus
}
#endif

#endif
