/**
* \file sdc_keystore_keys.h
*
* \brief LIBSDC: Definition of functions etc. required by applications to manage
* keystorage keys
*
* Applications requiring to install/remove key-storage keys need to
* include this header.\n
* The functions required to manage permissions are included automatically.\n
* The same is true for generic types like error codes.
*
* \author Christoph Gellner (cgellner@de.adit-jv.com)
*
* \copyright (c) 2015 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/

/**
 * \ingroup keymanagement
 * \defgroup keymanagement_keystore Key storage
 * Functions to generate, insert, import or remove keys of stored in the
 * key storage (persistent or volatile)
 */

#ifndef __SDC_LIB_KEYSTORE_KEYS_H_
#define __SDC_LIB_KEYSTORE_KEYS_H_

#ifdef __cplusplus
extern "C"
{
#endif

#include "sdc.h"
#include "sdc_perm.h"

/* Definitions types and defaults */

/**
 * \ingroup keymanagement_keystore
 * \brief Specify if the KID is specified or automatically determined during key creation
 */
typedef enum {
    SDC_CREATE_KEY_AUTOMATIC_ID,    /**< determine KID automatically */
    SDC_CREATE_KEY_FIXED_ID         /**< use explicitly specified KID */
} sdc_key_id_options_t;

/**
 * \ingroup keymanagement_keystore
 * \brief Specify if the created key shall be volatile or persistent
 */
typedef enum {
    SDC_VOLATILE_STORAGE_KEY,   /**< keystorage key will be volatile (lost in next power cycle) */
    SDC_PERSISTENT_STORAGE_KEY  /**< keystorage key will persist over power cycles */
} sdc_key_storage_options_t;


/* Functions */


/**
 * \ingroup keymanagement_keystore
 * \brief Install a key to the key storage using a user specified plain key secret
 *
 * If \p kid_opt is \ref SDC_CREATE_KEY_FIXED_ID \p kid is used as input.\n
 * If \p kid_opt is \ref SDC_CREATE_KEY_AUTOMATIC_ID the kernel will search for a free ID and return it using \p kid.
 *
 * \param kid key identifier
 * \param kid_opt specify if \p kid is used as input or output
 * \param plain_key buffer containing the key secret of the key
 * \param plain_key_len length of the plain_key buffer
 * \param key_stor_opt specify if the key shall be volatile or persistent
 * \param permissions permissions (incl. group) of the key
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_KEY_INVALID - given \p kid, \p plain_key or \p plain_key_len is invalid
 * \return \ref SDC_PERM_INVALID - given \p permissions are invalid
 * \return \ref SDC_INVALID_PARAMETER - any other parameter is invalid
 * \return \ref SDC_KID_EXISTS - the fixed (\ref SDC_CREATE_KEY_FIXED_ID) KID is already used
 * \return \ref SDC_KID_NOT_AVAILABLE - failed to find any free KID (\ref SDC_CREATE_KEY_AUTOMATIC_ID)
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return \ref SDC_CONFIG_FAILED - failed to fetch SDC daemon socket config
 * \return \ref SDC_DAEMON_COMMUNICATION_ERROR - communication to SDC daemon failed
 * \return \ref SDC_DAEMON_TIMEOUT_ERROR - response of daemon not received in time
 * \return \ref SDC_KEY_STORAGE_ACCESS_FAILED - daemon failed to access key storage
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_insert_plain_storage_key(
        sdc_key_id_t *kid,
        const sdc_key_id_options_t kid_opt,
        const uint8_t *plain_key,
        const size_t plain_key_len,
        const sdc_key_storage_options_t key_stor_opt,
        const sdc_permissions_t *permissions);

/**
 * \ingroup keymanagement_keystore
 * \brief Install a key to the key storage using internally generated random key secret
 *
 * If \p kid_opt is \ref SDC_CREATE_KEY_FIXED_ID \p kid is used as input.\n
 * If \p kid_opt is \ref SDC_CREATE_KEY_AUTOMATIC_ID the kernel will search for a free ID and return it using \p kid.
 *
 * \param kid key identifier
 * \param kid_opt specify if \p kid is used as input or output
 * \param key_len desired key length
 * \param key_stor_opt specify if the key shall be volatile or persistent
 * \param permissions permissions (incl. group) of the key
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_KEY_INVALID - given \p kid or \p key_len is invalid
 * \return \ref SDC_PERM_INVALID - given \p permissions are invalid
 * \return \ref SDC_INVALID_PARAMETER - any other parameter is invalid
 * \return \ref SDC_KID_EXISTS - the fixed (\ref SDC_CREATE_KEY_FIXED_ID) KID is already used
 * \return \ref SDC_KID_NOT_AVAILABLE - failed to find any free KID (\ref SDC_CREATE_KEY_AUTOMATIC_ID)
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return \ref SDC_CONFIG_FAILED - failed to fetch SDC daemon socket config
 * \return \ref SDC_DAEMON_COMMUNICATION_ERROR - communication to SDC daemon failed
 * \return \ref SDC_DAEMON_TIMEOUT_ERROR - response of daemon not received in time
 * \return \ref SDC_KEY_STORAGE_ACCESS_FAILED - daemon failed to access key storage
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_generate_random_storage_key(
        sdc_key_id_t *kid,
        const sdc_key_id_options_t kid_opt,
        const size_t key_len,
        const sdc_key_storage_options_t key_stor_opt,
        const sdc_permissions_t *permissions);

/**
 * \ingroup keymanagement_keystore
 * \brief Remove key with given ID from the key storage
 *
 * \param kid        ID of key to remove
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_KEY_INVALID - given KID is invalid
 * \return \ref SDC_KID_NOT_AVAILABLE - the given KID is valid but not available in the key storage
 * \return \ref SDC_KEY_LOCKED - the key can't be removed as it is currently assigned to one or multiple sessions
 * \return \ref SDC_ACCESS_DENIED - the process has no permission to delete the key
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return \ref SDC_CONFIG_FAILED - failed to fetch SDC daemon socket config
 * \return \ref SDC_DAEMON_COMMUNICATION_ERROR - communication to SDC daemon failed
 * \return \ref SDC_DAEMON_TIMEOUT_ERROR - response of daemon not received in time
 * \return \ref SDC_KEY_STORAGE_ACCESS_FAILED - daemon failed to access key storage
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_remove_storage_key(const sdc_key_id_t kid);

#ifdef __cplusplus
}
#endif

#endif
