/***************************************************************************
 *                                                                         *
 * Copyright                                                               *
 *     escrypt GmbH, Bochum, Germany                                       *
 *     Lise-Meitner-Allee 4                                                *
 *     D-44801 Bochum, Germany                                             *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************/
/*!
 \file        pkcs10_utils.c

 \brief       PKCS#10 Utilities Module

 $Rev: 937 $
 */
/***************************************************************************
 $Author: mlange $
 $Date: 2019-02-20 16:27:46 +0100 (Mi, 20. Feb 2019) $
 ****************************************************************************/

/***************************************************************************
 * 1. INCLUDES                                                              *
 ****************************************************************************/

#include "../../pkcs10/inc/pkcs10_utils.h"
#include <unistd.h>
#include <pthread.h>
#include <dbus/dbus.h>

/* ASN.1 Header */
#include "../asn1/CertificationRequest.h"
#include "../../common/inc/esc_common.h"

/* ASN1 Utilities header */
#include "../../cmp/inc/asn1_utils.h"
#include "../../common/inc/esc_asn1_utils.h"

/* common headers */
#include "../../common/inc/esc_debug.h"
#include "../../ecm/inc/ecm.h"
#include "../../common/inc/esc_utils.h"
#include "../../crypto/inc/crypto.h"


#include <dbus/dbus.h>
#include <unistd.h>
#include <pthread.h>

/* ECM header */
#include "../../ecm/inc/ecm_cm_EventHandler.h"

/* Local header */
#include "../inc/pkcs10_utils.h"

#ifdef UNIT_TESTING
#include "ecmUnitTests.h"
#endif



/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/***************************************************************************
 * 3. DEFINITIONS                                                          *
 ***************************************************************************/
// PKCS#10 Version - 0 for all known Versions (at least until 1.7)
#define pkcs10Version 0

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/

/***************************************************************************
 * 5. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/

BOOL createPKCS10CSR(
    PKCS10_DATA_s *pkcs10Data,
    UINT8 **dataOut,
    UINT32 *dataOutLen)
{
    BOOL failed = FALSE;
    CertificationRequest_t *certSignReq = NULL;
    RDNSequence_t *rdnSeqPtr;
    Name_t *subj = NULL;
    OID_t oid;

    dbgIn(__func__);

    failed = alloc((void **) &certSignReq, sizeof(CertificationRequest_t));
    subj = &certSignReq->certificationRequestInfo.subject;


    // Assign Version Number
    ifNotFailed {
        failed = (asn_long2INTEGER(
                &(certSignReq->certificationRequestInfo.version), pkcs10Version)
                != 0);
    }

    // Subject Info:
    // C=FR, L=Paris, O=PSA Peugeot Citroen,
    // OU=Open Vehicle Infotainment Platform,
    // OU=(c) PSA Peugeot Citroen - For authorized use only/initials=01,
    // CN=0D01180AC9D4DBC8675B

    // Assign unique Identifier
    ifNotFailed {
        /* place the device unique ID somehow within the subject field */
        subj->present = Name_PR_rdnSequence;
        rdnSeqPtr = &subj->choice.rdnSequence;

        if (pkcs10Data->setUniqueIdAsCn == TRUE) {
            oid = oid_commonName;
        } else {
            if(pkcs10Data->uniqueIdSrc){
                if (strncmp(pkcs10Data->uniqueIdSrc, "mac", strlen ( "mac" )) == 0) {
                    oid = oid_uniqueIdentifier;
                } else if (strncmp(pkcs10Data->uniqueIdSrc, "serial", strlen ( "serial" )) == 0) {
                    oid = oid_serialNumber;
                } else {
                    failed = TRUE;
                }
            }
            else failed = TRUE;
        }

        ifNotFailed {
            failed = addToRdnList(rdnSeqPtr, oid, pkcs10Data->deviceUniqueId);
        }
        ifNotFailed {
            certSignReq->certificationRequestInfo.subject = *subj;
        }
    }

    // Set Public Key and Algorithm Identifier
    ifNotFailed {

        /* set algorithm id */
        failed =
                setObjIdentifier(
                        &certSignReq->certificationRequestInfo.subjectPKInfo.algorithm.algorithm,
                        oid_rsaEnc);
    }

    ifNotFailed {

        /* set public key */
        failed = alloc((void**)& certSignReq->certificationRequestInfo.subjectPKInfo.subjectPublicKey.buf,
                pkcs10Data->pkcs1PubKeySize);
    }
    ifNotFailed{

        if (certSignReq->certificationRequestInfo.subjectPKInfo.subjectPublicKey.buf
                != NULL) {
            memcpy(
                    certSignReq->certificationRequestInfo.subjectPKInfo.subjectPublicKey.buf,
                    pkcs10Data->pkcs1PubKey, pkcs10Data->pkcs1PubKeySize);

            certSignReq->certificationRequestInfo.subjectPKInfo.subjectPublicKey.size =
                    pkcs10Data->pkcs1PubKeySize;
            certSignReq->certificationRequestInfo.subjectPKInfo.subjectPublicKey.bits_unused =
                    0;
        } else {
            failed = TRUE;
        }

    }

    // serialize CRI
    UINT8* temp = NULL;
    UINT32 tlength = 0;
    ifNotFailed {

        failed = asn1Encode(&certSignReq->certificationRequestInfo,
                &asn_DEF_CertificationRequestInfo, &temp, &tlength, FALSE);
    }

    // Sign CR
    ifNotFailed {
        failed = alloc((void**) &certSignReq->signature.buf,
                pkcs10Data->keySize / 8);
    }
    ifNotFailed{
        void* keyPair = NULL;
        if (certSignReq->signature.buf != NULL) {
            keyPair = pkcs10Data->keyPair;

        } else
            failed = TRUE;

        if (!failed && keyPair)
            failed = EscCrypto_SignMessage(keyPair, temp, tlength,
                    certSignReq->signature.buf,
                    pkcs10Data->hashAlg, pkcs10Data->keySize);
        else
            failed = TRUE;

        ifNotFailed {
            certSignReq->signature.size = pkcs10Data->keySize / 8;
            certSignReq->signature.bits_unused = 0;
        }

    }

    free(temp);
    temp = NULL;

    // Add Signature and used Signature Algorithm to CR
    ifNotFailed {
        // Set Signing Algorithm
        if (pkcs10Data->hashAlg == hash_sha1) {
            failed = setObjIdentifier(
                    &certSignReq->signatureAlgorithm.algorithm,
                    oid_shaWithRsaEnc);
        } else if (pkcs10Data->hashAlg == hash_sha256) {
            failed = setObjIdentifier(
                    &certSignReq->signatureAlgorithm.algorithm,
                    oid_sha256WithRsaEnc);
        } else {
            failed = TRUE;
        }

    }

    ifNotFailed {

        dbgPrint( DBG_INFO, "Encode PKCS#10 CSR message\n");
        failed = asn1Encode(certSignReq, &asn_DEF_CertificationRequest, dataOut,
                dataOutLen, FALSE);
    }

    ifFailed {
        *dataOutLen = 0;
    }

    ASN_STRUCT_FREE ( asn_DEF_CertificationRequest, certSignReq );

    // Subject is part of certSignReq

    if(!failed){
        pkcs10Data->csrRequested = TRUE;
    }

    dbgOut(__func__);

    return (failed);
}

void freePkcs10Data(
    PKCS10_DATA_s *pkcs10data)
{
    BOOL failed = FALSE;
    UINT8 *ptr;

    /* sanity checks */
    if (!pkcs10data) {
        dbgPrint( DBG_ERR, DBG_FLS "Invalid parameters!\n", DBG_FL);
        failed = TRUE;
    }

    ifNotFailed {

        // only free if pkcs10 owns the memory or its a copy (cmp and ecm also have the uniqueID field)
        ptr = (UINT8 *)pkcs10data->deviceUniqueId;
        escFreePtr ( &ptr );

        ptr = pkcs10data->pkcs1PubKey;
        escFreePtr(&ptr);

        ptr = (UINT8 *)pkcs10data->devCertUri;
        escFreePtr(&ptr);
    }
}

/***************************************************************************
 * 6. END                                                                  *
 ***************************************************************************/

