/***************************************************************************
 *                                                                         *
 * Copyright                                                               *
 *     escrypt GmbH, Bochum, Germany                                       *
 *     Lise-Meitner-Allee 4                                                *
 *     D-44801 Bochum, Germany                                             *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        ecm_cm_tools.c

   \brief       Utilities module of Embedded Certificate Manger

   $Rev: 814 $
 */
/***************************************************************************
$Author: mlange $
$Date: 2017-08-28 10:42:47 +0200 (Mo, 28. Aug 2017) $
****************************************************************************/

/***************************************************************************
* 1. INCLUDES                                                              *
****************************************************************************/

#include <dirent.h>
#include <unistd.h>
#include <sys/stat.h>
#include <errno.h>
#include <dbus/dbus.h>

#include "../inc/ecm.h"
#include <asn1_utils.h>
//#include "ecm_cm_EventHandler.h"
#include "esc_asn1_utils.h"
#include "../inc/ecm_utils.h"
#include "esc_debug.h"
#include "esc_utils.h"
//#include "pkcs10_utils.h"
#include "cmp_utils.h"
#include <base64.h>
//#include <ecm_cm_sdc_utils.h>

#include "../inc/ecm_cm_tools.h"

#ifdef UNIT_TESTING
#include "../inc/ecmUnitTests.h"
#endif

/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

#define CONFIG_FILE "./ecm.conf"

/***************************************************************************
 * 3. DEFINITIONS                                                          *
 ***************************************************************************/

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/

/***************************************************************************
 * 5. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/

BOOL setupFileNames (
    ECM_PARAMS_t *ecm );

void ecmCmCleanup (
    ECM_PARAMS_t *ecm,
    ECM_CONFIG_t *conf )
{
    if ( !ecm || !conf ) {
        return;
    }

    base64_cleanup();
    escCloseLogging();
    freeEcmData(ecm);
    escFreePtr((UINT8 **) &conf->serial);
}

char* getUID(
    ECM_PARAMS_t *ecm )
{
    BOOL failed = FALSE;
    char *uid = NULL;
    UINT32 uidSize = 0;

    /* sanity checks */
    if ( !ecm ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        if ( !ecm->str.uidFile ) {
            dbgPrint ( DBG_ERR, DBG_FLS "UIN file name not available "\
                "(Not set in configuration file?)\n", DBG_FL );
            failed = TRUE;
        }
    }

#ifdef CONFIG_ECM_USE_SDC

    /* Setup session */
    ifNotFailed {
        failed = setupSdcSession( &ecm->sdc, ecm->sdc.uinWrapKeyId );
    }

#endif /* CONFIG_ECM_USE_SDC */

    /* load UID file content */
    ifNotFailed {
        failed = loadFromFileToBuffer( &ecm->sdc, ecm->str.uidFile, (UINT8 **) &uid, &uidSize);
    }

#ifdef CONFIG_ECM_USE_SDC

    /* Release session */
    ifNotFailed {
        releaseSdcSession( &ecm->sdc);
    }

#endif /* CONFIG_ECM_USE_SDC */

    /* Adjust UID string */
    ifNotFailed {
        uid = realloc ( uid, uidSize + 1);
        uid[uidSize] = 0;
        remTrailingCrLf( &uid );
        uid = realloc ( uid, strlen ( uid ) + 1);
    }

    ifFailed {
        escFreePtr( (UINT8 **) &uid );
        uid = NULL;
    }

    return uid;
}

BOOL loadLocalCRLs(
    char* filepath,
    const CR_LIST_t** crList,
    CRL_HASHES_t** hashes)
{
    CRL_HASHES_t** hash = hashes;
    BOOL failed = FALSE;
    DIR *dir = NULL;
    struct dirent *entry = NULL;
    size_t s = 0;
    char* path = NULL;

    if(!filepath || !crList || !hashes) {
        return TRUE;
    }

    if( (dir = opendir(filepath)) != NULL){
        while (((entry = readdir(dir)) != NULL) && (failed == FALSE )) {

            //failed = TRUE;

            if(strstr(entry->d_name, ".crl") != NULL){
                s = strlen(filepath)+strlen(entry->d_name)+1;
                if ( escCheckSizetForOv( s )) {
                    failed = TRUE;
                    break;
                }
                if ( escMalloc ((void **)&path, s + 1)) {
                    failed = TRUE;
                    break;
                }

                strncpy(path, filepath, strlen ( filepath)+strlen (entry->d_name) + 2);
                strcat(path, "/");
                strcat(path, entry->d_name);
                if(!(*hash)) {
                    *hash = calloc(1, sizeof(CRL_HASHES_t));
                    if ( NULL == *hash ) {
                        failed = TRUE;
                        break;
                    }
                }
                failed = loadCRL(path, crList, NULL, *hash);
                hash = &(*hash)->next;
                escFreePtr((UINT8 **)&path);
                path = NULL;
            }
        }
        closedir(dir);
    }

    return failed;
}

BOOL cm_init(
    ECM_PARAMS_t* ecm,
    ECM_CONFIG_t* ecm_conf)
{
    BOOL failed = FALSE;
    size_t s = 0;
#ifdef CONFIG_ECM_USE_SDC
    sdc_error_t ret;
#endif /* CONFIG_ECM_USE_SDC */

    if ( !ecm || !ecm_conf ) {
        failed = TRUE;
    }

    ifNotFailed {
        ecm_conf->confFiles[1].name = "/etc/ecm/ecm.conf" ;
        ecm_conf->confFiles[2].name = "./ecm.conf" ;

        ecm->certFormat = ID_FORMAT_CONTAINER_DER;
        ecm->keyFormat = ID_FORMAT_CONTAINER_DER;
        ecm->keyContainer = ID_FORMAT_CONTAINER_PKCS1;

        if(readConfFile(ecm_conf->confFiles, ecm)){
            dbgPrint(DBG_ERR, DBG_FLS "Could not read configuration file\n", DBG_FL);
            failed = TRUE;
        }
    }

    ifNotFailed {
        /* initialize logging */
        failed = escInitLogging(ecm->logLevel, ecm_conf->quietFlag, ecm->str.logFile);
    }

    ifNotFailed {
        if(ecm->keySize == 0) {
            ecm->keySize = 1024;
        }

        if(setupFileNames(ecm)){
            dbgPrint(DBG_ERR, DBG_FLS "Could not set up File Names\n", DBG_FL);
            failed = TRUE;
        }
    }

    ifNotFailed {

        /* Setup certificate file and key store directory (CFKSDir) if necessary */
        if ( access ( ecm->str.fileStorePath, F_OK ) != 0 ) {
            dbgPrint ( DBG_INFO, "No CFKSDir found - must be ECMs first start\n" );

            /* create CFKSDir */
            if ( mkdir ( ecm->str.fileStorePath, 0775 ) < 0) {
                dbgPrint ( DBG_ERR, DBG_FLS
                    "Failed to create CFKSDir directory '%s' (%s)\n",
                    DBG_FL, ecm->str.fileStorePath, strerror ( errno ));

                failed = TRUE;
            }
        }
    }

    ifNotFailed {

         /* Setup alternative key store location if necessary */
         if ( access ( ecm->str.keyPath, F_OK ) != 0 ) {
            dbgPrint ( DBG_INFO, "No key directory found - must be ECMs first start\n" );

            /* create */
            if ( mkdir ( ecm->str.keyPath, 0775 ) < 0) {
                dbgPrint ( DBG_ERR, DBG_FLS
                    "Failed to create key directory '%s' (%s)\n",
                    DBG_FL, ecm->str.keyPath, strerror ( errno ));

                failed = TRUE;
            }
        }
    }

#ifdef CONFIG_ECM_USE_SDC

    /* Initialize SDC and get session */
    ifNotFailed {
        ret = sdc_open_session (&ecm->sdc.session);
        failed = checkSdcRet( ret );
    }

    ifNotFailed {
        ret = sdc_wrap_unwrap_type_alloc(&ecm->sdc.type);
        failed = checkSdcRet( ret );
    }

    ifNotFailed {
        ret= sdc_wrap_unwrap_type_set_algorithm(
            ecm->sdc.type, SDC_WRAP_ALG_AES );
        failed = checkSdcRet( ret );
    }

    ifNotFailed {
        ret = sdc_wrap_unwrap_type_set_block_mode(
            ecm->sdc.type, SDC_WRAP_BLK_CCM );
        failed = checkSdcRet( ret );
    }
#endif /* CONFIG_ECM_USE_SDC */

    ifNotFailed {

        ecm->currentTime = 0;
        ecm->latestCRLavailable = FALSE;

        /* Read UIN from given file */
        ifNotFailed {
            ecm->uniqueID = getUID( ecm );
            if(!ecm->uniqueID){
                dbgPrint(DBG_INFO, "UIN file is not available.\n" );
            }
        }
    }

    ifNotFailed {
        failed = addRootCert(ecm);
    }

    ifNotFailed {
        ecm->pkcs10Data.hashAlg = hash_sha1;
        ecm->pkcs10Data.uniqueIdSrc = NULL;
        ecm->pkcs10Data.setUniqueIdAsCn = TRUE;
        if ( ecm->uniqueID ) {

            s = strlen ( ecm->uniqueID );
            failed = escCheckSizetForOv( s );
            ifNotFailed {
                failed = alloc ((void **)&ecm->pkcs10Data.deviceUniqueId, s + 1);
            }
            ifNotFailed {
                strncpy(ecm->pkcs10Data.deviceUniqueId, ecm->uniqueID, s );
            }
        }
        if ( ecm->str.deviceCertURI ) {

            s = strlen ( ecm->str.deviceCertURI );
            failed = escCheckSizetForOv( s );
            ifNotFailed {
                failed = alloc ((void**)&ecm->pkcs10Data.devCertUri, s + 1);
            }
            ifNotFailed {
                strncpy(ecm->pkcs10Data.devCertUri, ecm->str.deviceCertURI, strlen(ecm->str.deviceCertURI));
            }
        }

        ecm->pkcs10Data.keySize = ecm->keySize;
    }

    ifNotFailed {

        UINT8* data = NULL;
        UINT32 size = 0;
        ecm->nv.flags = 0;

        ecm->sdc.execSdcOp = FALSE;
        failed = loadFromDiskToDER( &ecm->sdc, ecm->str.nvData, &data, &size);

        if (size != sizeof(ecm->nv.flags)) {
            escFreePtr( &data );
            failed = TRUE;
        }

        if (!failed) {
            memcpy(&ecm->nv.flags, data, size);
            ecm->nv.flags = ntohl(ecm->nv.flags);
        } else {
            // no nvData available --> create one
            size = sizeof(ecm->nv.flags);
            data = malloc(size);
            if ( NULL == data ) {
                failed = TRUE;
            } else {
                memcpy(data, &ecm->nv.flags, size);
                failed = writeByteStreamToFile( &ecm->sdc, ecm->str.nvData, data, size);
            }
        }

        escFreePtr ( (UINT8 **)&data);
    }

    ifNotFailed {
        /* check for CSR file */
        if ( !ecm->str.csrFileName ) {

            dbgPrint ( DBG_ERR, DBG_FLS "No CSR file path defined\n", DBG_FL );
            failed = TRUE;

        } else {

            /* try accessing the file */
            if (access(ecm->str.csrFileName, F_OK) != -1) {

                SDC_PARAMS_s sdc;
                UINT8 *data = NULL;
                UINT32 dataLen = 0;

                memset ( &sdc, 0x0, sizeof ( SDC_PARAMS_s ));
                sdc.execSdcOp = FALSE;

                failed = loadFromFileToBuffer( &sdc, ecm->str.csrFileName, &data, &dataLen );

                if (( TRUE == failed ) || ( dataLen == 0 )) {

                    dbgPrint( DBG_INFO, "Failed to read CSR file, seems that PKCS#10 CSR was not requested yet!\n");
                    /* reset 'failed' since this should not be handled as an error */
                    failed = FALSE;

                } else {

                    ecm->pkcs10Data.csrRequested = TRUE;
                }

                escFreePtr( &data );

            } else {
                dbgPrint( DBG_INFO, "Failed to read CSR file, seems that PKCS#10 CSR was not requested yet!\n");
            }
        }
    }

    ifNotFailed {
        // loadLocalCRLs - ignore return Value
        if ( loadLocalCRLs(ecm->str.fileStorePath, (const CR_LIST_t **)&ecm->crList, &ecm->crListHash) == FALSE ) {
            /* at least one CRL was loaded,
               means that we can consider that secure time module was
               already sourced successful, set flag */
            ecm->localCrlLoaded = TRUE;
        }
    }

    return failed;
}

BOOL setupFileNames (
    ECM_PARAMS_t *ecm )
{
    BOOL failed = FALSE;
    size_t len = 0;

    /* sanity checks */
    if ( !ecm ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        /* use ./cfksdir if no path is set */
        if ( ecm->str.fileStorePath == NULL ) {
            ecmCatFileName( FALSE, &ecm->str.fileStorePath, "./", "cfksdir" );
        }

        /* set the key storage location path */
        if ( NULL == ecm->str.keyPath ) {
            len = strlen ( ecm->str.fileStorePath );

            failed = escCheckSizetForOv( len );
            ifNotFailed {
                failed = escMalloc ((void **)&ecm->str.keyPath, len + 1 );
            }
            ifNotFailed {
                memcpy ( ecm->str.keyPath, ecm->str.fileStorePath, len );
            }
        }
    }

    ifNotFailed {

        /* is key_name set in configuration file?
         * If yes, just concatenate with keyPath.
         * If not, create the whole file name from default name and appropriate file extensions.
         */
        if ( ecm->str.devKeyName ) {

            ecmCatFileName ( TRUE, &ecm->str.devKey, ecm->str.keyPath, ecm->str.devKeyName );

        } else {

            if (ecm->keyFormat == ID_FORMAT_CONTAINER_PEM) {
                if ( ecm->keyContainer == ID_FORMAT_CONTAINER_PKCS8 ) {
                    ecmCatFileName ( TRUE, &ecm->str.devKey, ecm->str.keyPath,
                        KEY_FILE_NAME PKCS8_FILE_IDENTIFIER FILE_EXT_PEM );
                } else {
                    ecmCatFileName ( TRUE, &ecm->str.devKey, ecm->str.keyPath,
                        KEY_FILE_NAME FILE_EXT_PEM );
                }
            } else if (ecm->keyFormat == ID_FORMAT_CONTAINER_DER) {
                if ( ecm->keyContainer == ID_FORMAT_CONTAINER_PKCS8 ) {
                    ecmCatFileName ( TRUE, &ecm->str.devKey, ecm->str.keyPath,
                        KEY_FILE_NAME PKCS8_FILE_IDENTIFIER FILE_EXT_DER );
                } else {
                    ecmCatFileName ( TRUE, &ecm->str.devKey, ecm->str.keyPath,
                        KEY_FILE_NAME FILE_EXT_DER );
                }
            }

#ifdef CONFIG_ECM_USE_SDC

            if ( ecm->str.devKey ) {
                failed = appendSdcSuffix( (UINT8 **) &ecm->str.devKey );
            }

#endif /* CONFIG_ECM_USE_SDC */
        }
    }

    ifNotFailed {
        if(!ecm->str.nvData) {
            ecmCatFileName ( TRUE,
                &ecm->str.nvData, ecm->str.fileStorePath, NV_DATA_FILE_NAME );
        }
    }

    return ( failed );
}

BOOL deleteCertsinFolder(char* path){
    BOOL failed = FALSE;
    DIR *dir = NULL;
    struct dirent *entry = NULL;
    size_t s = 0;

    if ( !path) {
        return TRUE;
    }

    if( (dir = opendir(path)) != NULL){
        while ((entry = readdir(dir)) != NULL){
            if(strstr(entry->d_name, ".crt") != NULL){

                char* fpath = NULL;
                s = strlen(path)+strlen(entry->d_name)+1;

                failed = escCheckSizetForOv( s );
                ifNotFailed {
                    failed = escMalloc ((void **)&fpath, s + 1);
                }
                ifNotFailed {
                    strncpy(fpath, path, strlen (path) + strlen ( entry->d_name) + 2);
                    strcat(fpath, "/");
                    strcat(fpath, entry->d_name);
                    failed = remove(fpath) < 0 ? TRUE : FALSE;
                }
                escFreePtr((UINT8**)&fpath);
            }
        }
        closedir(dir);
    }
    return failed;
}

BOOL addRootCert(ECM_PARAMS_t* ecm){
    if(!ecm) {
        return TRUE;
    }
    BOOL failed = FALSE;

    UINT8* rootData = NULL;
    UINT32 rootSize = 0;
    size_t s = 0;

    dbgPrint ( DBG_INFO, "Loading root certificate\n");

#ifdef CONFIG_ECM_USE_SDC
    setupSdcSession( &ecm->sdc, ecm->sdc.certWrapKeyId);
#endif /* CONFIG_ECM_USE_SDC */

    failed = loadFromDiskToDER( &ecm->sdc, ecm->str.rootCrt, &rootData, &rootSize);

#ifdef CONFIG_ECM_USE_SDC
    releaseSdcSession( &ecm->sdc );
#endif /* CONFIG_ECM_USE_SDC */

    if(failed){
        free(rootData);
        return failed;
    }

    CERT_LIST_s* certs = NULL;
    failed = escMalloc ((void **)&certs, sizeof(CERT_LIST_s));
    if(failed){
        free(rootData);
        return failed;
    }

    if(ecm->certList == NULL){
        ecm->certList = certs;
    } else {
        CERT_LIST_s* tempcerts = ecm->certList;
        while(tempcerts->next)
            tempcerts = tempcerts->next;
        tempcerts->next = certs;
    }
    Certificate_t* crt = NULL;
    failed = alloc((void**)&crt, sizeof(Certificate_t));
    if(failed){
        free(rootData);
        return failed;
    }
    failed = asn1Decode( rootData, rootSize, &asn_DEF_Certificate, crt, FALSE);
    failed = cmpExtractCrtParamsFromTBSCertificate ( &crt->tbsCertificate, certs );
    if(failed){
        free(rootData);
        freeCertList(&ecm->certList);
        ASN_STRUCT_FREE(asn_DEF_Certificate, crt);
        return failed;
    }

    s = strlen ( ecm->str.rootCrt );
    failed = escCheckSizetForOv( s );
    ifNotFailed {
        failed = escMalloc ((void **)&ecm->certList->fileName, s + 1);
    }
    ifNotFailed {

        strncpy ( ecm->certList->fileName, ecm->str.rootCrt, s );
        ecm->certList->certPtr = rootData;
        ecm->certList->certSize = rootSize;
        ecm->certList->crt = crt;
        // mark as root
        ecm->certList->certId = cert_id_root;
        ecm->certList->devChain = TRUE;

        /* reset storing flag - not needed */
        ecm->certList->store = FALSE;
    }

    return failed;
}

BOOL removeInvalidCrlHashEntries (
    ECM_PARAMS_t *ecm )
{
    BOOL failed = FALSE;
    CRL_HASHES_t **crlPtr;
    CRL_HASHES_t *nextPtr;

    dbgIn (__func__ );


    /* sanity checks */
    if ( !ecm ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        crlPtr = &ecm->crListHash;
        while ( (*crlPtr) ) {

            /* remove CRLs marked as outdated or do not have a valid signature */
            if ( (TRUE == (*crlPtr)->outdated ) || ( FALSE == (*crlPtr)->validSignature )){

                /* free CR list entries as well */
                removeCrListEntryByIssuerCn ( (*crlPtr)->issuerCN, &ecm->crList );

                escFreePtr((UINT8 **)&(*crlPtr)->issuerCN );
                escFreePtr((UINT8**) &(*crlPtr)->digest );
                escFreePtr((UINT8**) &(*crlPtr)->certList );

                if ( (*crlPtr)->cl ) {
                    asnFreeStruct((void**) &(*crlPtr)->cl, asn_DEF_CertificateList);
                }

                nextPtr = (*crlPtr)->next;

                escFreePtr((UINT8**) &(*crlPtr) );

                (*crlPtr) = nextPtr;

            } else {
                crlPtr = &(*crlPtr)->next;
            }
        }
    }

    dbgOut (__func__ );

    return ( failed );
}


BOOL loadKeyCM(
    ECM_PARAMS_t* ecm)
{
    BOOL failed = FALSE;
    escFreePtr(&ecm->pkcs1PubKey);
    ecm->pkcs1PubKeySize = 0;

#ifdef ECM_PKCS10
    /* Do we already have memory allocated for keys e.g.
     because of a key re-newing? - Free it! */
    escFreePtr(&ecm->pkcs10Data.pkcs1PubKey);
    ecm->pkcs10Data.pkcs1PubKeySize = 0;
    ecm->pkcs10Data.keyPair = NULL;
#endif


    /* Load current Key File, if present */
    if (access(ecm->str.devKey, F_OK) != -1) {

#ifdef CONFIG_ECM_USE_SDC
    /* Don't use SDC here during unit testing (malloc(), free() re-direct issurs */
# ifndef UNIT_TESTING
        /* init SDC parameters */
        setupSdcSession ( &ecm->sdc, ecm->sdc.devKeyWrapKeyId);
# endif /* !UNIT_TESTING */

#endif /* CONFIG_ECM_USE_SDC */

        failed = EscCrypto_LoadPrivateKey(
            &ecm->sdc, ecm->str.devKey, &ecm->keyPair, ecm->keySize);

#ifdef CONFIG_ECM_USE_SDC
# ifndef UNIT_TESTING
        /* cleanup SDC parameters */
        releaseSdcSession( &ecm->sdc );
# endif /* !UNIT_TESTING */
#endif /* CONFIG_ECM_USE_SDC */

        ifNotFailed {

            /* allocate memory */
            failed = escMalloc ((void **)&ecm->pkcs1PubKey, MAX_ENC_RSA_PUBKEY_SIZE);
        }
        ifNotFailed {
            ecm->pkcs1PubKeySize = MAX_ENC_RSA_PUBKEY_SIZE;

            /* encode public key to PKCS#1 ASN1 DER */
            failed = EscCrypto_EncodePublicKey( &ecm->keyPair, ecm->pkcs1PubKey,
                &ecm->pkcs1PubKeySize, ecm->keySize);
        }

        ifNotFailed {

            ecm->pkcs1PubKey = realloc( ecm->pkcs1PubKey, ecm->pkcs1PubKeySize);

        } else {
            escFreePtr(&ecm->pkcs1PubKey);
            ecm->pkcs1PubKeySize = 0;
        }

#ifdef ECM_PKCS10
        ifNotFailed {
            ecm->pkcs10Data.keyPair = &ecm->keyPair;

            failed = escMalloc ((void**)&ecm->pkcs10Data.pkcs1PubKey, ecm->pkcs1PubKeySize);
        }
        ifNotFailed {
            memcpy(ecm->pkcs10Data.pkcs1PubKey, ecm->pkcs1PubKey, ecm->pkcs1PubKeySize);
            ecm->pkcs10Data.pkcs1PubKeySize = ecm->pkcs1PubKeySize;
        }
#endif /* ECM_PKCS10 */

    } /* Load current Key File, if present */
    else
        failed = TRUE;

    return failed;
}

BOOL createKeyFnc(
    ECM_PARAMS_t* ecm)
{
    BOOL failed = FALSE;
    const UINT32 pubKey = 65537U;
    UINT8 *keyBlob = NULL;
    UINT32 keyBlobSize = 0;

    /* generate key at ECM */
    dbgPrint( DBG_INFO, "Generate key at device\n");

    /* sanity checks */
    if ( !ecm ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    /* init RNG? */
    ifNotFailed {
        if (ecm->randCtx.counter == 0) {
            dbgPrint( DBG_INFO, "Starting RNG initialization\n");
            failed = EscCrypto_InitRandom(&ecm->randCtx);
            dbgPrint( DBG_INFO, "Finished RNG initialization\n");
        }
    }

    /* create key */
    ifNotFailed {
        dbgPrint( DBG_INFO, "Starting key generation\n");
        failed = EscCrypto_GenerateKeyPair(&ecm->randCtx, &ecm->keyPair, pubKey,
                &keyBlob, &keyBlobSize, ecm->keySize);
        dbgPrint( DBG_INFO, "Finished key generation\n");
    }


    ifNotFailed {
        /* is there already a key file? */
        if (access(ecm->str.devKey, F_OK) != -1) {
            /* we could not load the key file, so just remove it */
            if (remove(ecm->str.devKey) < 0) {
                dbgPrint( DBG_ERR, "Failed to remove old key file (%s)\n",
                        strerror(errno));
                failed = TRUE;
            }
        }
    }

    ifNotFailed {
        /* serialize key pair and write to file */

#ifdef CONFIG_ECM_USE_SDC
        setupSdcSession( &ecm->sdc, ecm->sdc.devKeyWrapKeyId );
#endif /* CONFIG_ECM_USE_SDC */

        failed = EscCrypto_StorePrivateKey2(
            &ecm->sdc, ecm->str.devKey, ecm->keyContainer,
            ecm->keyFormat, &ecm->keyPair, ecm->keySize);

#ifdef CONFIG_ECM_USE_SDC
        releaseSdcSession( &ecm->sdc );
#endif /* CONFIG_ECM_USE_SDC */

    }

    return failed;
}

/***************************************************************************
 * 6. END                                                                  *
 ***************************************************************************/
