/***************************************************************************
 *                                                                         *
 * Copyright                                                               *
 *     escrypt GmbH, Bochum, Germany                                       *
 *     Lise-Meitner-Allee 4                                                *
 *     D-44801 Bochum, Germany                                             *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        ecm_cm_main.c

   \brief       Main of ECM CM-Version

   $Rev: 830 $
 */
/***************************************************************************
$Author: mlange $
$Date: 2017-09-12 16:11:18 +0200 (Di, 12. Sep 2017) $
****************************************************************************/

/***************************************************************************
* 1. INCLUDES                                                              *
****************************************************************************/

#include <getopt.h>
#include <unistd.h>
#include <errno.h>
#include <dirent.h>
#include <pthread.h>
#include <signal.h>

#include "../inc/ecm.h"

#include "../../crl/inc/crl_utils.h"

#include "../../cmp/inc/asn1_utils.h"
#include "../inc/ecm_cm_tools.h"
#include "../inc/ecm_cm_EventHandler.h"

#include "../../common/inc/esc_debug.h"
#include "../inc/ecm_utils.h"
#include "../../common/inc/esc_utils.h"
#include "../../common/inc/esc_asn1_utils.h"
#include "../../cmp/inc/cmp_utils.h"
#include <base64.h>
//#include <ecm_cm_sdc_utils.h>

#ifdef UNIT_TESTING
#include "ecmUnitTests.h"
#endif

#include "../inc/ecm_cm_main.h"
#include "../inc/ecm_version.h"

#ifdef CONFIG_USE_LIBTIMER
#include <Timer.h>
#endif /* CONFIG_USE_LIBTIMER */

/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/***************************************************************************
 * 3. DEFINITIONS                                                          *
 ***************************************************************************/

static void readParams(int argc, char** argv, ECM_CONFIG_t* ecm_conf);
static void printCMUsage (char *name );
static int ecmCmLoadCerts( ECM_PARAMS_t* ecm, CERT_LIST_s** cert_List );
BOOL ecmCmIdentifyCerts ( ECM_PARAMS_t* ecm, CERT_LIST_s** cert_List );

ECM_PARAMS_t ecm;
ECM_CONFIG_t ecm_config;
pthread_t threadID = 0;

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/

/***************************************************************************
 * 5. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/
void printStructTm ( struct tm * t ) {
    printf ("%d sec, %d min, %d hour, %d mday, %d mon, %d year, %d wday, %d yday, %d flag\n",
        t->tm_sec, t->tm_min, t->tm_hour, t->tm_mday, t->tm_mon, t->tm_year, t->tm_wday, t->tm_yday, t->tm_isdst);
}

static char *constructFilePath (
    char *path,
    char *name,
    char *ext )
{
    char* filepath = NULL;
    char* filename = NULL;
    BOOL failed = FALSE;
    size_t s = 0;

    /* sanity checks */
    if ( !path || !name || !ext ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {

        s = strlen( name ) + strlen ( ext );
        failed = escCheckSizetForOv( s );
    }

    ifNotFailed {
        failed = alloc ((void **) &filename, s + 1 );
    }
    ifNotFailed {
        strncpy(filename, name, strlen ( name ));
        strncat(filename, ext, strlen ( ext ));

        ecmCatFileName ( TRUE, &filepath, path, filename );

    } else {
        escFreePtr (( UINT8 **) &filepath );
    }

    escFreePtr( (UINT8 **) &filename );

    return ( filepath );
}

BOOL ecmCmStoreCrls (
    ECM_PARAMS_t *ecm )
{
    BOOL failed = FALSE;

    dbgIn (__func__);

    /* sanity checks */
    if ( !ecm ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        CRL_HASHES_t* hashes = ecm->crListHash;
        char *path = NULL;
        while(hashes){
            if(hashes->certList && (hashes->store == TRUE )){
                // store CRLs (if necessary)
                path = constructFilePath(
                    ecm->str.fileStorePath, hashes->issuerCN, FILE_EXT_CRL );

                writeByteStreamToFile(
                    &ecm->sdc, path, hashes->certList, hashes->certListLen);

                escFreePtr( (UINT8 **) &path );
                hashes->store = FALSE;
            }
            hashes = hashes->next;
        }
    }

    dbgOut (__func__);

    return ( failed );
}

/** Signal handler for SIGINT, SIGSEGV, SIGTERM */
void ecmCmSignalHandler(
    int sig)
{
    void* threadRet = NULL;
    int ret = 0;

    if ( sig == SIGILL ) {
        return;
    }

    dbgPrint(DBG_INFO, "ECM signal handler received signal %d\n", sig );

    /* finish event handler thread */
    stopDBusService();

    /* CM target raises signal 11 if pthread_join() is called with threadID = 0! */
    if ( 0 != threadID ) {
        ret = pthread_join(threadID, threadRet);
        dbgPrint(DBG_INFO, "pthread_join returned %d\n", ret);
    }

    dbgPrint(DBG_INFO, "Shutting down ECM...\n");

    ecmCmCleanup( &ecm, &ecm_config );

    exit ( EXIT_FAILURE );
}

static void setUpCmSignalHandler( void )
{
    struct sigaction sig;
    sigset_t sigmask;

    /* setup signal handler */
    memset(&sig, 0x0, sizeof(struct sigaction));

    /* dont receive SIGALRM in main thread
     (this interrupts select() as well ) */
    sigemptyset(&sigmask);
    sigaddset(&sigmask, SIGALRM);
    pthread_sigmask( SIG_BLOCK, &sigmask, (sigset_t *) 0);

    /* receive SIGINT, SIGTERM, SIGSEGV in main thread */
    sigemptyset(&sigmask);
    sigaddset(&sigmask, SIGINT );
    sigaddset(&sigmask, SIGSEGV );
    sigaddset(&sigmask, SIGTERM);
    pthread_sigmask( SIG_UNBLOCK, &sigmask, (sigset_t *) 0);

    /* setup signal handling */
    sig.sa_flags = 0;
    sig.sa_handler = ecmCmSignalHandler;
    sigaction( SIGINT, &sig, (struct sigaction *) NULL);
    sigaction( SIGSEGV, &sig, (struct sigaction *) NULL);
    sigaction( SIGTERM, &sig, (struct sigaction *) NULL);
}

BOOL ecmCmCheckForErrors (
    ECM_PARAMS_t *ecm,
    BOOL ignoreDeviceCertCrl )
{
    BOOL failed = FALSE;
    ECMCM_DBUS_ERROR_t dbusRet;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !ecm ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {

        failed = ecmClearStatus( &ecm->status );
    }

    ifNotFailed {

        /* do we have UIN? */
        if ( NULL == ecm->pkcs10Data.deviceUniqueId ) {

            /* Try loading it */
            ecm->uniqueID = getUID( ecm );
            if(!ecm->uniqueID){

                /* if failed set error accordingly */
                dbgPrint(DBG_INFO, "UIN file is not available\n");
                failed = TRUE;
                ecmCmSetStatus( &ecm->status, EXT_IF_ERR_UIN_FILE_NOT_AVAILABLE, "null", NULL );
            }
        }

        /* build up hierarchy and verify signatures */
        if ( ecm->status.err == EXT_IF_ERR_NO_ERROR ) {
            failed = ecmCmIdentifyCerts(ecm, &ecm->certList);
        }

        /* do we have all CRLs? */
        if ( (ecm->status.err == EXT_IF_ERR_NO_ERROR) && (ecm->disableCRLcheck != TRUE) ) {

            CRL_HASHES_t *crlHash = ecm->crListHash;
	
			dbgPrint ( DBG_INFO, "CRL Check not disabling so verifying that we have them all\n" );
            /* Do we have CRLs at all? */
            if ( crlHash == NULL ) {

                /* No CRL available, this implies that Root CRL is missing as well, set errors */

                /* get root cert subject CN to indicate missing root CRL (issued by root) */
                CERT_LIST_s *tmp = NULL;
                failed = checkCertListForCertByLocalId (ecm->certList, cert_id_root, &tmp );

                ifNotFailed {
                    /* *tmp is set if checkCertListForCertByLocalId() returns successful! */
                    ecmCmSetStatus( &ecm->status, EXT_IF_ERR_CRL_NOT_PRESENT, tmp->subjectCn, "all" );
                    dbgPrint ( DBG_INFO, "Could not find root CRL.\n" );
                } else {
                    ecmCmSetStatus( &ecm->status, EXT_IF_ERR_INTERNAL_ERROR, "Root certificate missing.", NULL );
                    dbgPrint ( DBG_ERR, "Could not find root certificate for CN lookup.\n" );
                }

                failed = TRUE;
            }

            ifNotFailed {
                CERT_LIST_s *cert = ecm->certList;
                CRL_HASHES_t *crl = NULL;

                /* go through complete certificate list */
                while ( cert && (FALSE == failed)) {

                    crl = NULL;

                    /* is a crl of the certificates issuer available? */
                    ecmCmGetCrlHashByIssuerCn ( cert->issuerCn, ecm->crListHash, &crl );

                    /* identify the CRL (CRL issuer CN matches the current certificates issuer CN) */
                    if ( NULL == crl ) {
                        switch ( cert->certId ) {

                            case ( cert_id_issuer ):
                            case ( cert_id_endpoint ):

                                failed = TRUE;
                                ecmCmSetStatus( &ecm->status, EXT_IF_ERR_CRL_NOT_PRESENT, cert->issuerCn, cert->crlUri );
                                dbgPrint ( DBG_INFO, "CRL not available 'issuer CN=%s'.\n", cert->issuerCn );
                                break;

                            case ( cert_id_device ):
                                if ( ignoreDeviceCertCrl == FALSE ) {
                                    failed = TRUE;
                                    ecmCmSetStatus( &ecm->status, EXT_IF_ERR_CRL_NOT_PRESENT, cert->issuerCn, cert->crlUri );
                                    dbgPrint ( DBG_INFO, "CRL not available 'issuer CN=%s'.\n", cert->issuerCn );
                                }
                                break;

                            default:
                                break;
                        }
                    }

                    cert = cert->next;
                }
            }
        }

        /* verify CRL signatures and check if they are still valid
         * (current time < "CRL next update" */
        if ( ecm->status.err == EXT_IF_ERR_NO_ERROR ) {

            /* get time from STC */
            dbusRet = ecmCmGetTimeFromStm ( &ecm->currentTime, FALSE );
            if ( dbusRet != DBUS_SUCCESS ) {
                dbgPrint ( DBG_INFO, "Could not acquire time from STC\n");
                ecmCmSetStatus( &ecm->status, EXT_IF_ERR_SEC_TIME_NOT_AVAILABLE, NULL, NULL );
            }
        }

        if ( (ecm->status.err == EXT_IF_ERR_NO_ERROR) && (ecm->disableCRLcheck != TRUE) ) {

            CRL_HASHES_t *crlHash = ecm->crListHash;

            while (crlHash && (ecm->status.err == EXT_IF_ERR_NO_ERROR)) {

                /* CRLs valid? */
                failed = verifyCRLSignature(crlHash->cl, ecm->certList, &ecm->status);
                ifNotFailed {
                    crlHash->validSignature = TRUE;

                    /* only check CRL validity if no CRL was locally loaded,
                     * means STC was never sourced with CRL creation time,
                     * e.g. at initial boot
                     * note: localCrlLoaded flag is set in src/ecm_cm_tools.c
                     */
                    if ( ecm->localCrlLoaded == TRUE ) {

                        CERT_LIST_s *issuerCert = NULL;

                        /* ignore EC CRL if parameter ignoreDeviceCertCrl is TRUE
                         * means for offline use case to prevent the following error:
                         * - EC CRL was downloaded once when device was online
                         * - If device is never online again and CRL is expired,
                         *   this will result in an error for the offline use cases as well
                         */
                        if (( crlHash->id == crl_id_ec ) && ( ignoreDeviceCertCrl == TRUE )) {

                            /* ignore that case */

                        } else {

                            if (( ecm->currentTime > crlHash->nextUpdate )) {

                                /* look for a certificate that has as issuer CN the same as the CRL.
                                 * This certificate contains the URL for CRL download.
                                 * If we do not have it, we cannot download it in the online case.
                                 * */
                                cmpGetCrtFromListBySbjOrIssuerOid( issuer, oid_commonName, crlHash->issuerCN, ecm->certList, &issuerCert, TRUE);

                                if ( ecm->currentTime > crlHash->nextUpdate ) {
                                    crlHash->outdated = TRUE;
                                    failed = TRUE;
                                    ecmCmSetStatus( &ecm->status, EXT_IF_ERR_CRL_EXPIRED, crlHash->issuerCN, (( NULL != issuerCert ) ? issuerCert->crlUri : NULL));
                                    dbgPrint ( DBG_INFO, "CRL expired 'issuer CN=%s'.\n", crlHash->issuerCN );
                                    break;
                                }

                                /* 2017-06-19:
                                 * New requirement ECM shall never check for
                                 * Effective date or Valid from of CRL and
                                 * Crt from ECM. */
                            }
                        }
                    }
                }
                crlHash = crlHash->next;

            }
        }

        /* verify certificate time stamps,
         * latest time is requested just a few lines above */
        if ( ecm->status.err == EXT_IF_ERR_NO_ERROR ) {
            failed = checkForTimeValidity(ecm->certList, &ecm->status, ecm->currentTime);
        }

        /* verify certificates against CRL */
        if ( (ecm->status.err == EXT_IF_ERR_NO_ERROR) && (ecm->disableCRLcheck != TRUE) ) {
            failed = verifyCertListAgainstCRL(ecm->certList, ecm->crList, &ecm->status );
        }

        if ( ecm->status.err == EXT_IF_ERR_NO_ERROR ) {
            ecm->certConfigValid = TRUE;
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}

/* If this is being built for a unit test. */
#ifdef UNIT_TESTING

int example_main(int argc, char *argv[]);
/* main is defined in the unit test so redefine name of the the main function
 * here. */
#define main example_main

#endif /* UNIT_TESTING */

int main(
    int argc,
    char *argv[])
{
    BOOL failed = FALSE;

    memset(&ecm, 0, sizeof(ECM_PARAMS_t));
    memset(&ecm.str, 0, sizeof(FILE_PATH_NAME_STRINGS_s));
    memset(&ecm_config, 0, sizeof(ECM_CONFIG_t));

    // read Command Line Parameters
    readParams(argc, argv, &ecm_config);

    // Init CM - ECM
    if(cm_init(&ecm, &ecm_config)){
        dbgPrint(DBG_ERR, "Could not initialize ECM\n");

        ecmCmCleanup( &ecm, &ecm_config );
        return EXIT_FAILURE;
    }

    /* set up Signal Handler */
    setUpCmSignalHandler();

    /* daemonize? */
    if (ecm_config.noDaemonFlag == FALSE) {

        /* if we daemonize, be quiet on stdout */
        ecm_config.quietFlag = TRUE;
        escDaemonize();
    }

	if ( TRUE == ecm.disableCRLcheck ) {
		dbgPrint( DBG_INFO, "Disabling CRL Check\n");
	} else {
		dbgPrint( DBG_INFO, "CRL Check Not disabled\n");
	}
	
	if ( TRUE == ecm.disableDeviceCertificateCheck) {
		dbgPrint( DBG_INFO, "Disabling Device Cert Check\n");
	} else {
		dbgPrint( DBG_INFO, "Device Cert Check Not disabled\n");
	}
	
    /* try loading key, might fail */
    if( loadKeyCM(&ecm) ) {

        if ( TRUE == ecm.genDevKeyAtStartup ) {

            if ( createKeyFnc(&ecm)) {
                dbgPrint(DBG_ERR, "Should generate key at startup but could not create Key. Cannot recover at this point...\n");
                ecmCmCleanup( &ecm, &ecm_config );
                return EXIT_FAILURE;
            } else {
                /* Try to load Key again */
                if(loadKeyCM( &ecm)){
                    dbgPrint(DBG_ERR, "Generated key but could not load it. Cannot recover at this point...\n");
                    ecmCmCleanup( &ecm, &ecm_config );
                    return EXIT_FAILURE;
                }
            }

        } else {
            dbgPrint( DBG_INFO, "Could not load device key.\n");
        }
    }

    /* Initialize DBus interface */
    failed = ecmCmDbusInit();

    ifFailed {
        dbgPrint ( DBG_ERR, "DBus initialization failed. Cannot recover at this point...\n");
        ecmCmSignalHandler(0);
    }

    /* get time from SecureTimeModule
     * At first start this is either BSI module time or the system build time.
     * Value can be in the future or past but should be accurate enough to
     * verify the root certificate which is valid until 2060 */
    ECMCM_DBUS_ERROR_t dbusRet = DBUS_UNSUPPORTED_ERROR_CODE;
    UINT8 loopCnt = 0;

    while ( dbusRet != DBUS_SUCCESS ) {
        dbusRet = ecmCmGetTimeFromStm ( &ecm.currentTime, TRUE );

        if ( dbusRet != DBUS_SUCCESS && (loopCnt == 0)) {
            loopCnt++;
            dbgPrint ( DBG_ERR,
                "Could not get time from Secure Time Module. "\
                "Keep on trying...\n");
        }

        sleep ( 1 );
    }

    /* verify root certificate validity (signature and time);
     * root certificate is the only certificate in the list at this point */
    failed = cmpVerifyCertSignatures(ecm.certList, &ecm.status, TRUE, TRUE );
    failed |= checkForTimeValidity(ecm.certList, &ecm.status, ecm.currentTime);

    ifFailed {
        dbgPrint ( DBG_ERR,
            "Verification of root certificate failed. "\
            "Cannot recover at this point...\n");
        ecmCmSignalHandler(0);
    }

    /* load all certificates */
    ecmCmLoadCerts(&ecm, &ecm.certList);

    ecmCmCheckForErrors ( &ecm, TRUE );

    /* at this point we shall be able to identify the CRLs */
    determineCrlTypeByIssuerCn ( &ecm.crListHash, ecm.certList, &ecm.status );

    // Start CM Event Handler
    threadID = startEventHandler(&ecm);

    dbgPrint ( DBG_INFO, "ECM v%d.%d.%s (%s %s) initialized, waiting for events ... \n",
        ECM_MAJOR_VERSION, ECM_MINOR_VERSION, ECM_PATCH_VERSION, __DATE__, __TIME__);

    for ( ;; ) {
        usleep(100000);
    }

    /* never reached */
    return 0;
}

BOOL ecmCmRemoveCertFromList (
    char *subject,
    CERT_LIST_s *lst )
{
    BOOL failed = FALSE;
    CERT_LIST_s *ptr = lst;
    CERT_LIST_s *prev = NULL;
    CERT_LIST_s *free = NULL;

    dbgIn (__func__ );

    /* sanity checks */
    if ( !subject || !lst ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        ptr = lst;
        prev = lst;

        while ( ptr ) {

            if ( strncmp ( ptr->subjectCn, subject, strlen ( ptr->subjectCn )) == 0 ) {

                /* mark to free this item */
                free = ptr;
                /* remove this element from list by directly assigning
                 * next element to next element of previous element */
                prev->next = ptr->next;
            }

            /* adjust pointer of previous certificate */
            if ( ptr != prev ) {
                prev = ptr;
            }

            /* next element */
            ptr = ptr->next;

            if ( free ) {
                /* avoid freeing remaining chain */
                free->next = NULL;
                /* free chain containing only this element */
                freeCertList( &free );
                free = NULL;
            }
        }
    }

    dbgOut (__func__ );

    return ( failed );
}

BOOL ecmCmRemoveCertsByLocalId (
    CERT_ID_e id,
    CERT_LIST_s *lst )
{
    BOOL failed = FALSE;
    CERT_LIST_s *ptr = lst;
    CERT_LIST_s *next = lst;
    char *subj = NULL;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !lst ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        while ( ptr ) {

            if ( ptr->certId == id ) {

                next = ptr->next;
                subj = strdup ( ptr->subjectCn );
                ecmCmRemoveCertFromList( subj, lst );
                escFreePtr( (UINT8 **) &subj );
                ptr = next;

            } else {
                ptr = ptr->next;
            }
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}

static void readParams(int argc, char** argv, ECM_CONFIG_t* ecm_conf) {
    static struct option longOpts[] = {
            { "config-file", 1, 0, 'c' },
            { "help", 0, 0, 'h' },
            { "no-daemon", 0, 0, 'n' },
            { "quiet", 0, 0, 'q' },
    };

    if ( !ecm_conf ) {
        return;
    }

    while (1) {

        int opt = 0;
        int optIdx = 0;
        opt = getopt_long(argc, argv, "c:hnq:", longOpts, &optIdx);

        /* no more options? */
        if (opt == -1) {
            break;
        }

        switch (opt) {

        /* config file name */
        case ('c'):
            ecm_conf->confFiles[0].name = optarg;
            break;

            /* help */
        case ('h'):
            printCMUsage(strrchr(argv[0], '/'));
            exit(EXIT_SUCCESS);

            /* no-daemon */
        case ('n'):
            ecm_conf->noDaemonFlag = TRUE;
            break;

        case ('q'):
            ecm_conf->quietFlag = TRUE;
            break;

        default:
            printCMUsage(strrchr(argv[0], '/'));
            exit(EXIT_FAILURE);
        }
    } /* process command line arguments */
}

static void printCMUsage (char *name ) {

    if ( !name ) {
        return;
    }
    printf("\n%s %d.%d.%s (%s %s). Usage:\n\n",
        name, ECM_MAJOR_VERSION, ECM_MINOR_VERSION,
        ECM_PATCH_VERSION, __DATE__, __TIME__ );

    printf("%s [options]\n", name);
    printf(" --config-file   (-c) ... "\
        "Path to configuration file.\n");
    printf(" --help          (-h) ... "\
        "Print this help.\n");
    printf(" --no-daemon     (-n) ... "\
        "Do not run application as daemon.\n");
    printf(" --quiet         (-q) ... "\
        "Suppresses the ECM terminal output.\n");
    printf("\n");
}

BOOL ecmCmSetStatus (
    STATUS_s *stat,
    EXT_IF_ERROR_CODES_t err,
    char* str,
    char* url )
{
    BOOL failed = FALSE;
    size_t s = 0;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !stat ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        stat->err = err;
        if ( stat->str ) {
            free ( stat->str );
            stat->str = NULL;
        }
        if ( str ) {
            s = strlen ( str );
            failed = escCheckSizetForOv( s );
            ifNotFailed {
                failed = escMalloc ((void **)&stat->str, s + 1 );
            }
            ifNotFailed {
                memcpy ( stat->str, str, s );
            }
        }
    }

    ifNotFailed {

        if ( stat->url ) {
            escFreePtr((UINT8**)&stat->url);
        }
        if ( url ) {
            s = strlen ( url );
            failed = escCheckSizetForOv( s );
            ifNotFailed {
                failed = alloc ((void **)&stat->url, s + 1 );
            }
            ifNotFailed {
                memcpy ( stat->url, url, s );
            }
        }
    }

    dbgOut ( __func__ );

    return (failed);
}

BOOL ecmClearStatus (
    STATUS_s *stat )
{
    BOOL failed = FALSE;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !stat ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        stat->err = EXT_IF_ERR_NO_ERROR;
        escFreePtr( (UINT8 **) &stat->str );
        escFreePtr( (UINT8 **) &stat->url );
    }

    dbgOut ( __func__ );

    return ( failed );
}

static int ecmCmLoadCerts(
    ECM_PARAMS_t* ecm,
    CERT_LIST_s** cert_List )
{
    int count = 0;
    DIR *dir = NULL;
    struct dirent *entry = NULL;
    CERT_LIST_s* certList = NULL;
    BOOL loadFailed = FALSE;

    dbgIn ( __func__ );

    if(!ecm || !cert_List) {
        return -1;
    }

    certList = *cert_List;

    /* find all Certs in CFKS directory */
    if(certList != NULL){
        certList = findCertListLastItem(certList);
    }

    if( (dir = opendir(ecm->str.fileStorePath)) != NULL){
        while ((entry = readdir(dir)) != NULL){
            if(strstr(entry->d_name, FILE_EXT_CRT ) != NULL){
                CERT_LIST_s* certEntry = NULL;
                certEntry = calloc(1, sizeof(CERT_LIST_s));
                if ( NULL == certEntry ) {
                    loadFailed = TRUE;
                    return -1;
                }

                char *path = NULL;
                ecmCatFileName (
                    TRUE, &path, ecm->str.fileStorePath, entry->d_name );

#ifdef CONFIG_ECM_USE_SDC
                setupSdcSession( &ecm->sdc, ecm->sdc.certWrapKeyId );
#endif /* CONFIG_ECM_USE_SDC */

                loadFailed = loadFromDiskToDER(
                    &ecm->sdc, path, &certEntry->certPtr, &certEntry->certSize);

#ifdef CONFIG_ECM_USE_SDC
                releaseSdcSession( &ecm->sdc );
#endif /* CONFIG_ECM_USE_SDC */

                escFreePtr( (UINT8 **) &path);

                if(!loadFailed){
                    Certificate_t* crt = NULL;
                    loadFailed = alloc((void**)&crt, sizeof(Certificate_t));
                    if(!loadFailed){
                        loadFailed = asn1Decode( certEntry->certPtr,
                                certEntry->certSize, &asn_DEF_Certificate,
                                crt, FALSE);
                    }
                    if(!loadFailed){
                        loadFailed = cmpExtractCrtParamsFromTBSCertificate (
                            &crt->tbsCertificate, certEntry );
                        certEntry->store = FALSE;
                    }
                    if(!loadFailed){
                        certEntry->crlUri = getCRLURIfromCertificate(crt);
                        certEntry->crt = crt;
                    }
					if ( TRUE == loadFailed ) {
                        if ( NULL != crt ) {
                            escFreePtr ((UINT8 **)&crt );
                        }
                    }
                }
                if(!loadFailed){
                    if(certList){
                        certList->next = certEntry;
                        count++;
                        certList = certList->next;
                    }
                    else{
                        certList = certEntry;
                        // if given CERT_LIST_s is NULL - assign the newly created entry
                        *cert_List=certList;
                        count++;
                    }
                }
                else{
                    freeCertList(&certEntry);
                }
            }
        }
        closedir(dir);
    }

    return ( count );
}


BOOL ecmCmIdentifyCerts (
    ECM_PARAMS_t* ecm,
    CERT_LIST_s** cert_List )
{
    BOOL failed = FALSE;
    CERT_LIST_s* certList = NULL;
    CERT_LIST_s *tmp;

    dbgIn (__func__ );

    /* sanity checks */
    if ( !ecm || !cert_List ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    /* create certificate hierarchy, identify certificates, etc... */
    ifNotFailed {
        certList = *cert_List;

        failed = cmpDetermineCrtTypesAndHierarchy2 (
            *cert_List, ecm->pkcs1PubKey, ecm->pkcs1PubKeySize );
    }

    ifNotFailed {
        failed = checkCertListForCertByLocalId ( *cert_List, cert_id_device, &tmp );

        if(failed && (ecm->disableDeviceCertificateCheck != TRUE)){

            dbgPrint(DBG_INFO, "Could not find device certificate 'subject CN=%s'\n", ecm->uniqueID);
            ecmCmSetStatus( &ecm->status, EXT_IF_ERR_CERT_NOT_PRESENT, ecm->uniqueID, NULL );
        }
    }

    ifNotFailed {

        // verify CertChain and Signatures
        certList = *cert_List;
        failed = checkCertListForCompleteChain(certList, &ecm->status, TRUE );
    }

    ifNotFailed {
        certList = *cert_List;
        failed = cmpVerifyCertSignatures(certList, &ecm->status, FALSE, TRUE );
        // verify Time validity with acquired Time from CRL
    }

    /* at this point certificate signature verification may have failed,
     * (chain incomplete or signature invalid) go on and do additional tests */

    ifNotFailed {
        /* is the device certificate chain complete? */
        failed = checkCertListForCompleteDeviceCertChain ( certList );
    }

    dbgOut (__func__ );

    return ( failed );
}

int loadCRLandUpdateTime(
    CERT_LIST_s* certList,
    CR_LIST_t** cr_List,
    ull_time_t* time,
    CRL_HASHES_t** crlHashes,
    STATUS_s *stat )
{
    if (!certList || !cr_List || !time) {
        return -1;
    }

    // downloadCRLs
    CERT_LIST_s* certptr = certList;
    CR_LIST_t* crList = NULL;

    ull_time_t crlCreateTime = 0;
    ull_time_t crlTempTime = 0;
    ull_time_t crlNextUpdTime = 0;
    while (certptr) {
        if (certptr->crlUri) {
            int temp = addCRLfromURI(certptr->crlUri, &crList, certList,
                    &crlTempTime, &crlNextUpdTime, crlHashes, stat);
            crlCreateTime = (crlTempTime > crlCreateTime) ? crlTempTime : crlCreateTime;
            if (temp < -1) {
                // download failed - try again
                // clean up
                freeCRL(crList);
                return -2;
            } else if (temp < 0) {
                // other error
                // clean up
                freeCRL(crList);
                return -1;
            }
        }
        certptr = certptr->next;
    }

    /* adjust current time */
    if ( crlNextUpdTime < *time ){
        /* CRL next update time is also in the past - thats not expected */
        freeCRL(crList);
        return -1;
//    } else if ( crlNextUpdTime > *time && *time < crlCreateTime ) {
    } else {
        /* CRL next update will be in the future
           and our current time is lower that CRL creation time - adjust */

        /* push CLR time to STM and receive latest time */
#ifdef UNIT_TESTING
        *time = crlCreateTime;
#else /* UNIT_TESTING */
        ecmCmPushTimeToStm( stc_interface_provide_cea_time, crlCreateTime );
#endif /* UNIT_TESTING */

        if(*cr_List == NULL){
            *cr_List = crList;
        } else{
            CR_LIST_t* crtemp = *cr_List;
            while(crtemp->next){
                crtemp = crtemp->next;
            }
            // Append loaded CRLs
            crtemp->next = crList;
        }
        return 0;
    }
    return 0;
}

BOOL ecmCmStoreCerts (
    ECM_PARAMS_t *ecm )
{
    BOOL failed = FALSE;
    char *name = NULL;
    char *filepath = NULL;
    CERT_LIST_s *ptr = NULL;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !ecm ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        ptr = ecm->certList;

        while ( ptr ) {

            /* store certs from cert list, marked for storing */
            if ( ptr->store ) {

                if ( ptr->certId == cert_id_device ) {
                    if ( ecm->str.deviceCertName ) {
                        name = ecm->str.deviceCertName;
                    } else {
                        /* at this point uniqueID shall be available.
                         * otherwise this error was detected before */
                        name = ecm->uniqueID;
                    }
                } else if ( ptr->certId == cert_id_root ) {
                    ptr = ptr->next;
                    continue;
                } else {
                    name = ptr->subjectCn;
                }

                filepath = constructFilePath (
                    ecm->str.fileStorePath, name, FILE_EXT_CRT );

#ifdef CONFIG_ECM_USE_SDC

                failed = appendSdcSuffix( (UINT8 **) &filepath );
                setupSdcSession( &ecm->sdc, ecm->sdc.certWrapKeyId );
#endif /* CONFIG_ECM_USE_SDC */

                /* convert to PEM first? */
                if ( ID_FORMAT_CONTAINER_PEM == ecm->certFormat ) {

                    char *start_tag = PEM_CERT_START_TAG;
                    char *end_tag = PEM_CERT_END_TAG;
                    char *enc = NULL;
                    char *result_with_linebreak = NULL;
                    char *pem_encoded = NULL;
                    char *result = NULL;
                    size_t s = 0;

                    UINT32 outsize = 0;
                    enc = escBase64Encode(ptr->certPtr, ptr->certSize, &outsize);

                    if (!enc || outsize < 1) {
                        failed = TRUE;
                        escFreePtr((UINT8 **) &enc);
                        break;
                    } else {

                        failed = escCheckSizetForOv( outsize );

                        ifNotFailed {
                            failed = alloc ((void **) &result, outsize + 1 );
                        }
                        ifNotFailed {

                            snprintf(result, outsize + 1, "%s", enc);
                            result_with_linebreak = ecmAddPemNl( (unsigned char *) result);

                            s = strlen(start_tag) + strlen(result_with_linebreak) + strlen(end_tag);

                            failed = escCheckSizetForOv( s );
                        }
                        ifNotFailed {
                            failed = alloc ((void **)&pem_encoded, s + 1 );
                        }
                        ifNotFailed {
                            memcpy(pem_encoded, start_tag, strlen(start_tag));
                            memcpy(pem_encoded + strlen(start_tag), result_with_linebreak, strlen(result_with_linebreak));
                            memcpy( pem_encoded + strlen(start_tag) + strlen(result_with_linebreak), end_tag, strlen(end_tag) + 1);

                            failed = writeByteStreamToFile( &ecm->sdc, filepath, (UINT8*) pem_encoded, strlen(pem_encoded));
                        }
                    }

                    escFreePtr((UINT8 **) &pem_encoded);
                    escFreePtr((UINT8 **) &result_with_linebreak);
                    escFreePtr((UINT8 **) &enc);
                    escFreePtr((UINT8 **) &result);

                } else {

                    writeByteStreamToFile( &ecm->sdc, filepath, ptr->certPtr, ptr->certSize);
                }

#ifdef CONFIG_ECM_USE_SDC
                releaseSdcSession( &ecm->sdc );
#endif /* CONFIG_ECM_USE_SDC */

                escFreePtr( (UINT8 **)&filepath );

                /* reset store flag */
                ptr->store = FALSE;
            }

            ptr = ptr->next;
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}


/***************************************************************************
 * 6. END                                                                  *
 ***************************************************************************/
