/***************************************************************************
 *                                                                         *
 * Copyright                                                               *
 *     escrypt GmbH, Bochum, Germany                                       *
 *     Lise-Meitner-Allee 4                                                *
 *     D-44801 Bochum, Germany                                             *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        ecm_utils.h

   \brief       Utilities module of Embedded Certificate Manager

   ***

   $Rev: 811 $
 */
/***************************************************************************
$Author: mlange $
$Date: 2017-08-25 15:34:26 +0200 (Fr, 25. Aug 2017) $
****************************************************************************/

#ifndef __ECM_UTILS_H__
#define __ECM_UTILS_H__

#ifdef __cplusplus
extern "C" {
#endif

/***************************************************************************
 * 1. INCLUDES                                                             *
 ***************************************************************************/

#include "../inc/ecm.h"
#include "../../common/inc/esc_common.h"
#include "../../cmp/inc/asn1_utils.h"

/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/** Enumeration of certificate OID sources */
typedef enum {
    oid_issuer = 0,
    oid_subject,
} CRT_OID_SRC_E;


/***************************************************************************
 * 3. DECLARATIONS                                                         *
 ***************************************************************************/

/** File identifier enumeration */
typedef enum {
    /** ID none */
    file_id_none = 0,
    /** PKCS1 encoded private key file ID */
    file_id_pkcs1PrivKey,
    /** Non-volatile data file ID */
    file_id_nv_data,
    file_id_last_entry,
} FILE_ID_e;

/** Struct representing a Buffer */
typedef struct BufferStruct {
    /** Buffer */
    char* buf;
    /** Buffer Size */
    size_t size;
} BufferStruct_t;

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/

/***************************************************************************
 * 5. FUNCTION PROTOTYPES                                                  *
 ***************************************************************************/

/**
This function is used to initialize the timer

\param[in] params 
    ECM parameters structure

\return ECM_RET_t
\retval EcmRetOk on success

*/
ECM_RET_t
initTimer (
    ECM_PARAMS_t *params );

/**
This function is used to print the usage information

\param[in] name 
    Application name 

*/
void 
printUsage (
    char *name );

/**
This function is used to lookup the string representation of a
state machine state 

\param[in] state 
    State machine state variable

\return char*
\retval String representation of the state

*/
char *
state2Str ( 
    UINT32 state );


/**
This function is used to add newline characters to a
Base64 encoded string.

\param[in] data
    Base64 encoded string without newline characters

\return char*
\retval Base64 encoded string with newline characters

*/
char *ecmAddPemNl (
    const UINT8 *data );

/**
This function is used to load the non volatile parameters 

\param[in] params 
    ECM parameters structure

\param[out] params 
    Updated ECM parameters structure

\return ECM_RET_t
\retval EcmRetOk on success

*/
ECM_RET_t
loadNvParams (
    ECM_PARAMS_t *params );

/**
This function is used to convert a BOOL value to ECM_RET_t

\param[in] failed
    BOOL value to convert

\return ECM_RET_t
\retval EcmRetOk if failed == FALSE
\retval EcmRetFailed if failed == TRUE

*/
ECM_RET_t
bool2EcmRet (
    BOOL failed );

/**
This function is used to free ECM specific data from the global structure

\param[in] ecm
    Pointer to ECM parameters structure

\param[out] ecm
    Pointer to updated ECM parameters structure

\return none

*/
void
freeEcmData (
    ECM_PARAMS_t *ecm );

/**
This function is used to check if all mandatory parameters where set
by the configuration file.

\param[in] params
    Pointer to ECM parameters structure

\param[out] params
    Pointer to updated ECM parameters structure

\return BOOL
\retval FALSE on success
\retval TRUE on failure

*/
BOOL
checkMandatoryOptions (
    ECM_PARAMS_t *params );

/**
This function is used to handle a given configuration file line,
ie. checks for the relevant parameters and stores them in the global structure.

\param[in] nameBuf
    Pointer to the name of the configuration parameter
\param[in] linePtr
    Pointer to the value of the configuration parameter
\param[in] params
    Pointer to ECM parameters structure

\param[out] params
    Pointer to updated ECM parameters structure

\return BOOL
\retval FALSE on success
\retval TRUE on failure

*/
BOOL
handleConfigFileLine (
    char *nameBuf,
    char *linePtr,
    ECM_PARAMS_t *params );

/**
This function is used to set a flag. The flag structure is stored afterwards.

\param[in] ecm
    Pointer to ECM parameters structure

\param[in] flag
    Flags to set

\return ECM_RET_t
\retval EcmRetOk on success

*/
ECM_RET_t
ecmSetNvFlag (
    ECM_PARAMS_t *ecm,
    UINT32 flag );

/**
This function is used to clear a flag. The flag structure is stored afterwards.

\param[in] ecm
    Pointer to ECM parameters structure

\param[in] flag
    Flags to clear

\return ECM_RET_t
\retval EcmRetOk on success

*/
ECM_RET_t
ecmClearNvFlag (
    ECM_PARAMS_t *ecm,
    UINT32 flag );

/**
This function is used to delete a given Directory

\param[in] path
    Path of directory which shall be deleted

\return BOOL
\retval FALSE on success
\retval TRUE on failure
*/
BOOL
ecmDeleteDir (
    char *path );

/**
This function is used to construct a filepath

\param[in] sep
    BOOL if Separator should be inserted

\param[out] dst
    constructed filepath

\param[in] pre
    Prefix

\param[in] suf
    Suffix

\return BOOL
\retval FALSE on success
\retval TRUE on failure
*/
size_t ecmCatFileName (
    BOOL sep,
    char **dst,
    char *pre,
    char *suf );

/**
This function is used to load all certificates set in the configuration file,
e.g. root certificate or other TLS trust certificates.

\param[in] ecm
    ECM parameters structure

\param[out] ecm
    Updated ECM parameters structure

\return ECM_RET_t
\retval EcmRetOk on success
*/
ECM_RET_t
loadStaticCerts (
    ECM_PARAMS_t *ecm );


/**
This function is used to load the device certificate.

\param[in] uin
    Devices UIN
\param[in] uribase
    URI base for download from ecm.conf

\param[out] data
    Pointer to pointer for Buffer - The Memory is allocated by the function and
    needs to be free'd by the caller

\param[out] dataSize
    Size of the DER encoded Certificate in the data Buffer

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure
*/
BOOL
downloadDeviceCertificate2 (
    char *uin,
    char *uribase,
    UINT8 **data,
    UINT32 *dataSize );

/**
This function is used to find the last item of the certificate list.

\param[in] lst
    Pointer to certificate list

\return #BOOL
\retval Pointer to last certificate list item
\retval NULL if list is empty

*/
CERT_LIST_s
*findCertListLastItem (
    CERT_LIST_s *lst );

/**
This function checks if a Certificate is a Root Certificate by comparing the
Issuer and Subject Name

\param[in] crtEntry
    Pointer to Certificate_t

\return #BOOL
\retval #FALSE on Certificate is NOT a root Certificate
\retval #TRUE on Certificate is a root Certificate
*/
BOOL
isRootCert(
    CERT_LIST_s* crtEntry);

/**
This function is used to load all certificates of the Certificate Chain based
on a given Certificate
The function utilizes the downloadIssuersCertificate() Function

\param[in] crt
    Pointer to Certificate_t

\param[in] certList
    Pointer where Certificate will be stored

\retval Amount of loaded Certificates
*/
int
downloadCertificateChain(
    Certificate_t *crt,
    CERT_LIST_s** certList);

/**
This function is used to download the Issuers Certificate of a given Certificate
If the downloaded Certificate is PEM encoded, the function will try to convert
the Certificate to DER.

\param[in] crt
    Pointer to Certificate_t

\param[out] issuerCrt
    Pointer to Buffer with the Issuers Certificate DER encoded - The function
    allocates Memory accordingly. The Memory needs to be free'd by the caller.

\param[out] issuerCrtSize
    Size of the Issuers Certificate

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure
*/
BOOL
downloadIssuerCertificate(
    Certificate_t *crt,
    UINT8** issuerCrt,
    UINT32* issuerCrtSize);

/**
This function is used to download any file from URI to Buffer

\param[in] uri
    char Array specifing URI

\param[out] buf
    Pointer to char Pointer - Memory for the Buffer will be allocated and
    needs to be free'd by the caller

\param[out] size
    Pointer to UINT32 - returning the size of the Buffer


\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure
*/
BOOL
downloadFromURItoBuffer(
    char* uri,
    char** buf,
    UINT32* size);

/**
This function is used to download any file from URI to Disk

\param[in] uri
    char Array specifing URI

\param[in] dest
    Destination of File

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure
*/
BOOL
downloadFromURItoDisk(
    char* uri,
    char* dest);

/**
This function is used to compare 2 unsigned integer arrays

\param[in] array1
    unsigned int Array

\param[in] array2
    unsigned int Array

\param[in] length
    Amount of integers to be compared

\return #BOOL
\retval #TRUE if Arrays match
\retval #FALSE if Arrays do NOT match
*/
BOOL
matchUIntArrays(
    unsigned int* array1,
    unsigned int* array2,
    unsigned int length);

/**
This function is used to check a Buffer for pem encoding
If the Buffer starts with "-----BEGIN " the data is most likely pem encoded (rfc1421)
This function does not check for data consistency, etc.

\param[in] data
    Pointer to Buffer with Data which shall be checked

\return #BOOL
\retval #TRUE if Data is PEM encoded
\retval #FALSE if Data is probably NOT PEM encoded
*/
BOOL
checkForPEM(
    char* data);

/**
This function is used to check if data is a PEM encoded
This check is based on the common prefix for PEM encoded data: -----BEGIN CERTIFICATE

\param[in] data
    Pointer to Buffer with Data which shall be checked

\return #BOOL
\retval #TRUE if Data is PEM encoded
\retval #FALSE if Data is probably NOT PEM encoded
*/
BOOL
checkForPEMCertificate(
    char* data);

/**
This function is used to check if data is a PEM RSA
This check is based on the common prefix for PEM encoded data: -----BEGIN RSA PRIVATE KEY-----

\param[in] data
    Pointer to Buffer with Data which shall be checked

\return #BOOL
\retval #TRUE if Data is PEM encoded Certificate
\retval #FALSE if Data is probably NOT PEM encoded Certificate
*/
BOOL
checkForPEMRSAKey(
    char* data);

/**
This function is used to check if data is a PEM RSA
This check is based on the common prefix for PEM encoded data: -----BEGIN PRIVATE KEY-----

\param[in] data
    Pointer to Buffer with Data which shall be checked

\return #BOOL
\retval #TRUE if Data is PEM encoded Certificate
\retval #FALSE if Data is probably NOT PEM encoded Certificate
*/
BOOL
checkForPEMPKCS8Key(
    char* data);


/**
 * This function is used to write a Bytestram (e.g. DER encoded ASN1 Message to a File
 *
 * \param[in] sdc
 *      Pointer to SDC parameter structure
 * \param[in] fileName
 *      File name to write the Bytestream to
 * \param[in] byteStream
 *      Pointer to the ByteStream
 * \param[in] length
 *      Amount of Bytes which shall be written to File
 *
 * \return #BOOL
 * \retval #FALSE on success (file could be written)
 * \retval #TRUE on failure (file coud not be written)
 *
 */
BOOL
writeByteStreamToFile(
    SDC_PARAMS_s *sdc,
    const char* fileName,
    UINT8* byteStream,
    UINT32 length);

/**
 * This function is used to load a Bytestream from a file into a buffer
 * The buffer is allocated accordingly by the function and needs to be free'd by
 * the caller
 *
 * \param[in] sdc
 *      Pointer to SDC parameter structure
 * \param[in] path
 *      File name to write to
 * \param[in] byteStream
 *      Pointer to the ByteStream which shall be stored
 * \param[out] length
 *      Amount of Bytes to write
 *
 * \return #BOOL
 * \retval #FALSE on success (file could be written)
 * \retval #TRUE on failure (file coud not be written)
 *
 */
BOOL
loadFromFileToBuffer(
    SDC_PARAMS_s *sdc,
    const char* path,
    UINT8** byteStream,
    UINT32* length);

/**
 * This function is used to acquire Certificate from a List of Certificates by
 * Subject ID or Issuer ID
 *
 * \param[in] src
 *      Enum Subject or Issuer
 * \param[in] oid
 *      OID - common name or title
 * \param[in] ref
 *      Name which is searched for
 * \param[in] crtLst
 *      Certificate List in which to search for the Certificate
 * \param[out] crt
 *      Found Certificate
 * \param[in] quiet
 *      Flag to signal if Debug Output shall be printed
 *
 * \return #BOOL
 * \retval #FALSE on success
 * \retval #TRUE on failure
 *
 */
BOOL
getCrtFromListBySbjOrIssuerOid (
    CRT_OID_SRC_E src,
    OID_t oid,
    char *ref,
    CERT_LIST_s *crtLst,
    CERT_LIST_s **crt,
    BOOL quiet );

/**
 * This function is used to load a Bytestream from a file into a buffer and
 * convert it to DER if the bytestream is PEM encoded.
 * If it is not PEM encoded any file will be loaded as it is.
 * The buffer is allocated accordingly by the function and needs to be free'd by
 * the caller
 *
 * \param[in] sdc
 *      Pointer to SDC parameter structure
 * \param[in] fileName
 *      File name to read from
 * \param[out] derData
 *      Pointer to Pointer where ByteStream will be stored
 * \param[out] derLength
 *      Amount of Bytes which are loaded
 *
 * \return #BOOL
 * \retval #FALSE on success (file could be written)
 * \retval #TRUE on failure (file coud not be written)
 *
 */
BOOL
loadFromDiskToDER (
    SDC_PARAMS_s *sdc,
    char *fileName,
    UINT8** derData,
    UINT32* derLength);

/**
 * This function is used to  check a given List of Certificates for their
 * validity concerning time
 *
 * \param[in] certList
 *      Pointer to List of Certs
 * \param[in] stat
 *      Pointer to status structure to record errors
 * \param[in] time
 *      Current known time
 *
 * \return #BOOL
 * \retval #FALSE on success (all Certs a valid)
 * \retval #TRUE on failure (one or more files are not valid)
 *
 */
BOOL
checkForTimeValidity(
    CERT_LIST_s* certList,
    STATUS_s *stat,
    ull_time_t time);

/**
This function is used to set default values to parameters if they are not set
within the configuration file.

\param[in] params
    Pointer to ECM parameters structure

\param[out] params
    Pointer to updated ECM parameters structure

\return BOOL
\retval FALSE on success
\retval TRUE on failure

*/
BOOL
setDefaultOptions (
    ECM_PARAMS_t *ecm );

/***************************************************************************
 * 6. MACRO FUNCTIONS                                                      *
 ***************************************************************************/

/***************************************************************************
 * 7. END                                                                  *
 ***************************************************************************/

#ifdef __cplusplus
}
#endif

#endif /* __ECM_UTILS_H__ */
