/***************************************************************************
 *                                                                         *
 * Copyright                                                               *
 *     escrypt GmbH, Bochum, Germany                                       *
 *     Lise-Meitner-Allee 4                                                *
 *     D-44801 Bochum, Germany                                             *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        ecm_cm_EventHandler.h

   \brief       Embedded Certificate Manager Event Handler include

   ***

   $Rev: 937 $
 */
/***************************************************************************
$Author: mlange $
$Date: 2019-02-20 16:27:46 +0100 (Mi, 20. Feb 2019) $
****************************************************************************/

#ifndef ECM_INC_ECM_CM_EVENTHANDLER_H_
#define ECM_INC_ECM_CM_EVENTHANDLER_H_

#ifdef __cplusplus
extern "C" {
#endif

/***************************************************************************
 * 1. INCLUDES                                                             *
 ***************************************************************************/
#include <pthread.h>
#include "esc_common.h"
#include "ecm.h"

/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/***************************************************************************
 * 3. DECLARATIONS                                                         *
 ***************************************************************************/
/** Flag if Event Handler shall stop */
volatile BOOL exitCMEventHandler;

/** Parameters used by the Event Handler */
typedef struct {
    /** Pointer to ECM parameters structure */
    ECM_PARAMS_t *ecm;
} EventHandlerData;

/** Enumeration of DBus related errors */
typedef enum {
    /** No error */
    DBUS_SUCCESS = 0,
    /** Message creation failed */
    DBUS_MESSAGE_CREATION_FAILED,
    /** Send message with reply failed */
    DBUS_SEND_MESSAGE_WITH_REPLY_FAILED,
    /** 'Pending' is NULL */
    DBUS_PENDING_IS_NULL,
    /** 'Reply' is NULL */
    DBUS_REPLY_IS_NULL,
    /** Reading reply arguments failed */
    DBUS_READ_REPLY_ARGS_FAILED,
    /** Appending arguments failed */
    DBUS_APPEND_ARGS_FAILED,
    /** Arguments invalid */
    DBUS_ARGS_INVALID,
    /** Unsupported error code */
    DBUS_UNSUPPORTED_ERROR_CODE,
} ECMCM_DBUS_ERROR_t;

/** Enumeration for DBus reply types */
typedef enum {
    /** Generic DBus reply (UINT32 || String) */
    dbusRep_generic = 0,
    /** getCrlNextUpd DBus reply (UINT32 || String || UINT64) */
    dbusRep_getCrlNextUpd,
    /** getCSR DBus reply (UINT32 || String || byte array ) */
    dbusRep_getCsr,
} DBUS_REPLY_e;

/** Enumeration of STC interface */
typedef enum {
    /** Invalid / unused interface */
    stc_interface_none = 0,
    /** SetTime interface */
    stc_interface_set_time,
    /** ProvideCEAtime interface */
    stc_interface_provide_cea_time,
    /** Invalid / unused interface */
    stc_interface_max
} STC_INTERFACE_e;

/** DBUS get CSR reply structure */
typedef struct {
    /** CSR data */
    UINT8 *csr;
    /** CSR data length */
    UINT32 csrLen;
} DBUS_GET_CSR_REPLY_s;

/** String 'ROOT' */
#define ROOT_STR "ROOT"
/** String 'IC' */
#define IC_STR "IC"

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/

/***************************************************************************
 * 5. FUNCTION PROTOTYPES                                                  *
 ***************************************************************************/

/**
This function is used to start the ECM Event Handler

\param[in] ecm
    ECM parameters structure

\return pthread_t
\retval Thread ID or 0 on Failure
*/
pthread_t
startEventHandler(
    ECM_PARAMS_t* ecm);

/**
This function is used to stop the ECM Event Handler

*/
void
stopDBusService();

/**
This function is used to initialize ECMs DBus interface

 * \return #BOOL
 * \retval #FALSE on success
 * \retval #TRUE on failure
*/
BOOL
ecmCmDbusInit();

/**
This function is used to push the given time to the secure time component STC.

\param[in] time
    time value to push
\param[in] i
    Interface of STC to use

 * \return #ECMCM_DBUS_ERROR_t
*/
ECMCM_DBUS_ERROR_t ecmCmPushTimeToStm (
    STC_INTERFACE_e i,
    ull_time_t time );

/**
This function is used to retrieve the current time from
the secure time component STC.

\param[in] currTime
    Pointer where current time is stored
\param[in] quiet
    Flag to signal if errors shall be reported (FALSE) or not (TRUE)

 * \return #ECMCM_DBUS_ERROR_t
*/
ECMCM_DBUS_ERROR_t ecmCmGetTimeFromStm (
    ull_time_t *currTime,
    BOOL quiet );

/**
This function is used to retrieve the CRL hash of a given issuer.

\param[in] issuer
    Issuer common name string
\param[in] crlHashes
    Pointer to local CRL hashes structure containing all CRL hashes
\param[in] ptr
    Pointer where pointer to CRL hash is stored, if found

 * \return #BOOL
 * \retval #FALSE on success
 * \retval #TRUE on failure
*/
BOOL ecmCmGetCrlHashByIssuerCn (
    char *issuer,
    CRL_HASHES_t *crlHashes,
    CRL_HASHES_t **ptr );


/***************************************************************************
 * 6. MACRO FUNCTIONS                                                      *
 ***************************************************************************/

/***************************************************************************
 * 7. END                                                                  *
 ***************************************************************************/

#ifdef __cplusplus
}
#endif

#endif /* ECM_INC_ECM_CM_EVENTHANDLER_H_ */
