/***************************************************************************
 * Copyright                                                               *
 *                                                                         *
 *     ESCRYPT GmbH - Embedded Security       ESCRYPT Inc.                 *
 *     Zentrum fuer IT-Sicherheit             315 E Eisenhower Parkway     *
 *     Lise-Meitner-Allee 4                   Suite 214                    *
 *     44801 Bochum                           Ann Arbor, MI 48108          *
 *     Germany                                USA                          *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        sha_1.c

   \brief       SHA-1 implementation (FIPS-180-2 resp. RFC 2202 compliant)

   $Rev: 937 $
 */
/***************************************************************************/

/***************************************************************************
* 1. INCLUDES                                                              *
****************************************************************************/

#include "../inc/sha_1.h"

/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/** Length of schedule in words. */
#define EscSha1MSched_NUM_WORDS 16U

/** Circular right rotation, as defined in function ROTL of section 3.2 point 4. */
#define EscSha1_ROTL( x, n ) ( ( ( x ) << ( n ) ) | ( ( x ) >> ( 32U - ( n ) ) ) )

/** Logical function Ch of sec. 4.1.1. */
#define EscSha1_FuncCh( x, y, z ) ( ( z ) ^ ( ( x ) & ( ( y ) ^ ( z ) ) ) )

/** Logical function Maj of sec. 4.1.1 */
#define EscSha1_FuncMaj( x, y, z ) ( ( ( x ) & ( y ) ) | ( ( z ) & ( ( x ) ^ ( y ) ) ) )

/** Logical function Parity of sec. 4.1.1 */
#define EscSha1_FuncParity( x, y, z ) ( ( x ) ^ ( y ) ^ ( z ) )

/***************************************************************************
 * 3. DEFINITIONS                                                          *
 ***************************************************************************/
/** Message schedule context according to sec. 6.1.2 step 1. */
typedef struct {
    /**
    W_t. The array keeps the last 16 Wt.
    In the beginning w[0] = W_0, w[1] = W_1, etc.
    For t= 16, w[0] = W_16; t=17, w[1] = W-17, etc.
    */
    UINT32 w[ EscSha1MSched_NUM_WORDS ];
    /* Current t. */
    UINT8 t;
} EscSha1MSched_ContextT;

static void
EscSha1MSched_Init(
    EscSha1MSched_ContextT* sched,
    const UINT8 block[] );

static UINT32
EscSha1MSched_NextW(
    EscSha1MSched_ContextT* sched );

static void
EscSha1_UpdateHash(
    EscSha1_ContextT* ctx );

static void
EscSha1_UpdateHashFast(
    EscSha1_ContextT* ctx,
    const UINT8 block[] );

static void
EscSha1_ApplyPadding(
    EscSha1_ContextT* ctx );

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/

/***************************************************************************
 * 5. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/

static void
EscSha1MSched_Init(
    EscSha1MSched_ContextT* sched,
    const UINT8 block[] )
{
    UINT8 i;

    sched->t = 0U;

    /* W_0 .. W_15 is M_0..M-15, see sec. 6.1.2 step 1 */
    for ( i = 0U; i < EscSha1MSched_NUM_WORDS; i++ ) {
        UINT8 offset;

        offset = (UINT8)( 4U * i );
        sched->w[ i ] = ( ( (UINT32)block[ offset ] ) << 24 ) |
            ( ( (UINT32)block[ offset + 1U ] ) << 16 ) |
            ( ( (UINT32)block[ offset + 2U ] ) << 8 ) |
            ( (UINT32)block[ offset + 3U ] );
    }
}

static UINT32
EscSha1MSched_NextW(
    EscSha1MSched_ContextT* sched )
{
    UINT32 ret;
    UINT8 curT;

    /* W_t 6.1.2 step 1 */
    curT = sched->t;

    if ( curT < 16U ) {
        ret = sched->w[ curT ];
    } else {                    /* sched->t >= 16U */
        /* we calculate the indexes. The number -x is congruent to +(16-x) in |F_16 */
        UINT8 iw_3 = ( curT + ( 16U - 3U ) ) & 0x0fU;     /* Index of W_t-3 */
        UINT8 iw_8 = ( curT + ( 16U - 8U ) ) & 0x0fU;     /* Index of W_t-8 */
        UINT8 iw_14 = ( curT + ( 16U - 14U ) ) & 0x0fU;     /* Index of W_t-14 */
        UINT8 iw_16 = curT & 0x0fU; /* Index of W_t-16, and also t%16 */
        UINT32* W;
        UINT32 rotArg;

        W = sched->w;

        /*lint -save -esym(960,17.4) W is an array*/
        rotArg = W[ iw_3 ] ^ W[ iw_8 ] ^ W[ iw_14 ] ^ W[ iw_16 ];
        /*lint -restore W is an array*/

        ret = EscSha1_ROTL( rotArg, 1U );

        sched->w[ iw_16 ] = ret;  /* the next W_t-1 */
    }

    sched->t++;                 /* point to next t */

    return ret;
}

/** Padding according to section 5.1.1 */
static void
EscSha1_ApplyPadding(
    EscSha1_ContextT* ctx )
{
    const UINT8 PAD_LENGTH_SIZE = 8U;   /* Size of the length in the padding in byte. (We keep the first 3 byte always zero) */
    UINT32 lengthBits;

    Esc_ASSERT( ctx->blockLen < EscSha1_BLOCK_BYTES );

    /* Append a 0x10. */
    ctx->block[ ctx->blockLen ] = 0x80U;
    ctx->blockLen++;
    if ( ctx->blockLen == EscSha1_BLOCK_BYTES ) {
        EscSha1_UpdateHash( ctx );
    }

    /* Check if there is enough space for the length. If not, fill the
       current block with zeros and hash it. */
    if ( ctx->blockLen > ( EscSha1_BLOCK_BYTES - PAD_LENGTH_SIZE ) ) {
        while ( ctx->blockLen < EscSha1_BLOCK_BYTES ) {
            ctx->block[ ctx->blockLen ] = 0U;
            ctx->blockLen++;
        }

        EscSha1_UpdateHash( ctx );
    }

    /* Fill the block with zeros, except the last bytes for the length. */
    while ( ctx->blockLen < ( EscSha1_BLOCK_BYTES - PAD_LENGTH_SIZE ) ) {
        ctx->block[ ctx->blockLen ] = 0U;
        ctx->blockLen++;
    }

    /* Append the length in bits. */
    lengthBits = ctx->totalLen;
    /* The length in the format 0 | 0 | 0 | b31..b29 | b28..b21 | b20..b13 | b12..b5 | b4..b0 000 */
    /* b4..b0 */
    ctx->block[ EscSha1_BLOCK_BYTES - 1U ] = (UINT8)( ( (UINT8)( lengthBits & 0x1fU ) ) << 3 );
    lengthBits >>= 5;

    /* b12..b5 */
    ctx->block[ ( EscSha1_BLOCK_BYTES - 1U ) - 1U ] = (UINT8)( lengthBits & 0xffU );
    lengthBits >>= 8;

    /* b20..b13 */
    ctx->block[ ( EscSha1_BLOCK_BYTES - 1U ) - 2U ] = (UINT8)( lengthBits & 0xffU );
    lengthBits >>= 8;

    /* b28..b21 */
    ctx->block[ ( EscSha1_BLOCK_BYTES - 1U ) - 3U ] = (UINT8)( lengthBits & 0xffU );
    lengthBits >>= 8;

    /* b31..b29 */
    ctx->block[ ( EscSha1_BLOCK_BYTES - 1U ) - 4U ] = (UINT8)( lengthBits & 0xffU );
    ctx->block[ ( EscSha1_BLOCK_BYTES - 1U ) - 5U ] = 0U;
    ctx->block[ ( EscSha1_BLOCK_BYTES - 1U ) - 6U ] = 0U;
    ctx->block[ ( EscSha1_BLOCK_BYTES - 1U ) - 7U ] = 0U;

    ctx->blockLen += PAD_LENGTH_SIZE;

    Esc_ASSERT( ctx->blockLen == EscSha1_BLOCK_BYTES );
}

/**
Hashes this single hash block.  The blockLen must be EscSha1_BLOCK_BYTES;
Does the inner loop of section 6.1.2.
*/
static void
EscSha1_UpdateHashFast(
    EscSha1_ContextT* ctx,
    const UINT8 block[] )
{
    /* Constants from section 4.2.1 */
    static const UINT32 K[] = { 0x5a827999U, 0x6ed9eba1U, 0x8f1bbcdcU, 0xca62c1d6U };
    const UINT8* msg;
    UINT32* H;
    EscSha1MSched_ContextT sched;
    UINT8 t;
    UINT32 a, b, c, d, e;
    UINT32 T;

    msg = block;
    H = ctx->hash;

    /* Step 1 */
    EscSha1MSched_Init( &sched, msg );

    /* Step 2 */
    /*lint -save -esym(960,17.4) H is an array*/
    a = H[ 0 ];
    b = H[ 1 ];
    c = H[ 2 ];
    d = H[ 3 ];
    e = H[ 4 ];
    /*lint -restore H is an array*/

    /* Step 3 */
    /* 0<=t<=19 */
    for ( t = 0U; t <= 19U; t++ ) {
        T = EscSha1_ROTL( a, 5U ) + EscSha1_FuncCh( b, c, d ) + e + K[ 0 ] + EscSha1MSched_NextW( &sched );

        e = d;
        d = c;
        c = EscSha1_ROTL( b, 30U );
        b = a;
        a = T;
    }

    /* 20<=t<=39 */
    for (; t <= 39U; t++ ) {
        T = EscSha1_ROTL( a, 5U ) + EscSha1_FuncParity( b, c, d ) + e + K[ 1 ] + EscSha1MSched_NextW( &sched );

        e = d;
        d = c;
        c = EscSha1_ROTL( b, 30U );
        b = a;
        a = T;
    }

    /* 40<=t<=59 */
    for (; t <= 59U; t++ ) {
        T = EscSha1_ROTL( a, 5U ) + EscSha1_FuncMaj( b, c, d ) + e + K[ 2 ] + EscSha1MSched_NextW( &sched );

        e = d;
        d = c;
        c = EscSha1_ROTL( b, 30U );
        b = a;
        a = T;
    }

    /* 60<=t<=79 */
    for (; t <= 79U; t++ ) {
        T = EscSha1_ROTL( a, 5U ) + EscSha1_FuncParity( b, c, d ) + e + K[ 3 ] + EscSha1MSched_NextW( &sched );

        e = d;
        d = c;
        c = EscSha1_ROTL( b, 30U );
        b = a;
        a = T;
    }

    /* Step 4 */
    /*lint -save -esym(960,17.4) H is an array*/
    H[ 0 ] += a;
    H[ 1 ] += b;
    H[ 2 ] += c;
    H[ 3 ] += d;
    H[ 4 ] += e;
    /*lint -restore H is an array*/

    /* Reset blocklen */
    ctx->blockLen = 0U;
}

/**
Fast hashing of a single hash block.  The ctx->blockLen must be EscSha256_BLOCK_BYTES;
Does the inner loop of section 6.2.2.
*/
static void
EscSha1_UpdateHash(
    EscSha1_ContextT* ctx )
{
    Esc_ASSERT( ctx->blockLen == EscSha1_BLOCK_BYTES );

    EscSha1_UpdateHashFast( ctx, ctx->block );

    ctx->blockLen = 0U;
}

/***********************************************************************
 ***********************************************************************
 **** Exported functions
 ***********************************************************************
 ***********************************************************************/
BOOL
EscSha1_Init(
    EscSha1_ContextT* ctx )
{
    BOOL hasFailed = TRUE;
    UINT8 i;
    const UINT8 HASH_SIZE = 5U;
    static const UINT32 INITIAL_HASH[ 5 ] = {
        0x67452301U, 0xefcdab89U, 0x98badcfeU, 0x10325476U, 0xc3d2e1f0U
    };

    if ( ctx != 0 ) {
        /* Set initial hash value Sec. 5.3.2 */
        for ( i = 0U; i < HASH_SIZE; i++ ) {
            ctx->hash[ i ] = INITIAL_HASH[ i ];
        }

        ctx->blockLen = 0U;
        ctx->totalLen = 0U;

        hasFailed = FALSE;
    }

    return hasFailed;
}

BOOL
EscSha1_Update(
    EscSha1_ContextT* ctx,
    const UINT8 payload[],
    UINT32 dataLen )
{
    BOOL hasFailed = TRUE;

    if ( ( ctx != 0 ) && ( payload != 0 ) ) {
        UINT32 bytesLeft = dataLen;
        UINT32 dataPos = 0U;

        while ( bytesLeft > 0U ) {
            if ( ( ctx->blockLen == 0U ) && ( bytesLeft >= EscSha1_BLOCK_BYTES ) ) {
                UINT32 numFullBlocks = bytesLeft / EscSha1_BLOCK_BYTES;
                /* The number of bytes we process with UpdateHashFast() */
                UINT32 fullBlockBytes = numFullBlocks * EscSha1_BLOCK_BYTES;

                bytesLeft -= fullBlockBytes;
                ctx->totalLen += fullBlockBytes;

                while ( dataPos <= ( dataLen - EscSha1_BLOCK_BYTES ) ) {
                    EscSha1_UpdateHashFast( ctx, &payload[ dataPos ] );
                    dataPos += EscSha1_BLOCK_BYTES;
                }

                /* blockLen remains unchanged */
                Esc_ASSERT( ctx->blockLen == 0U );
            } else {
                UINT8 bytesToFill;
                /* fill block with remaining bytes, blockLen is now smaller than EscSha1_BLOCK_BYTES */
                if ( bytesLeft >= ( (UINT32)EscSha1_BLOCK_BYTES - (UINT32)ctx->blockLen ) ) {
                    bytesToFill = EscSha1_BLOCK_BYTES - ctx->blockLen;
                } else {
                    bytesToFill = (UINT8)bytesLeft;
                }

                ctx->totalLen += bytesToFill;
                bytesLeft -= bytesToFill;
                /* Copy the new bytes to the current block */
                while ( bytesToFill > 0U ) {
                    ctx->block[ ctx->blockLen ] = payload[ dataPos ];

                    ctx->blockLen++;
                    dataPos++;
                    bytesToFill--;
                }

                /* if block is complete - hash */
                if ( ctx->blockLen == EscSha1_BLOCK_BYTES ) {
                    EscSha1_UpdateHash( ctx );
                }
            }
        }

        Esc_ASSERT( ctx->blockLen < EscSha1_BLOCK_BYTES );

        hasFailed = FALSE;
    }

    return hasFailed;
}

BOOL
EscSha1_Finish(
    EscSha1_ContextT* ctx,
    UINT8 digest[] )
{
    BOOL hasFailed = TRUE;
    UINT8 i;

    if ( ( ctx != 0 ) && ( digest != 0 ) ) {
        /* Pad data according to Sec. 5.1.1 */
        EscSha1_ApplyPadding( ctx );
        /* Hash last full block */
        EscSha1_UpdateHash( ctx );

        /* Copy digest back */
        for ( i = 0U; i < 5U; i++ ) {
            UINT8 offset = (UINT8)( i * 4U );
            UINT32 h = ctx->hash[ i ];

            digest[ offset + 3U ] = (UINT8)( h & 0xffU );
            h >>= 8;
            digest[ offset + 2U ] = (UINT8)( h & 0xffU );
            h >>= 8;
            digest[ offset + 1U ] = (UINT8)( h & 0xffU );
            h >>= 8;
            digest[ offset ] = (UINT8)( h & 0xffU );
        }

        hasFailed = FALSE;
    }

    return hasFailed;
}

BOOL
EscSha1_Calc(
    const UINT8 payload[],
    UINT32 dataLen,
    UINT8 digest[] )
{
    BOOL hasFailed = TRUE;
    EscSha1_ContextT ctx;

    if ( ( payload != 0 ) && ( digest != 0 ) ) {
        hasFailed = EscSha1_Init( &ctx );

        if ( hasFailed == FALSE ) {
            hasFailed = EscSha1_Update( &ctx, payload, dataLen );

            if ( hasFailed == FALSE ) {
                hasFailed = EscSha1_Finish( &ctx, digest );
            }
        }
    }

    return hasFailed;
}

/***************************************************************************
 * 6. END                                                                  *
 ***************************************************************************/
