/***************************************************************************
 * Copyright                                                               *
 *                                                                         *
 *     ESCRYPT GmbH - Embedded Security       ESCRYPT Inc.                 *
 *     Zentrum fuer IT-Sicherheit             315 E Eisenhower Parkway     *
 *     Lise-Meitner-Allee 4                   Suite 214                    *
 *     44801 Bochum                           Ann Arbor, MI 48108          *
 *     Germany                                USA                          *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        rsa_4096.c

   \brief       RSA 4096 encryption and decryption

   $Rev: 937 $
 */
/***************************************************************************/
/***************************************************************************
 * 1. INCLUDES                                                             *
 ***************************************************************************/
#include "../inc/rsa_4096.h"
#include <stdlib.h>
/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/** Length of single multiplication results in words */
#define EscRsa4096_RSA_SIZE_WORDS_LONG ( ( 2U * EscRsa4096_RSA_SIZE_WORDS ) )

#ifdef EscRsa4096_ENABLE_GENRSA
/** Number of small primes that fit into a UINT8 (used for RSA key pair generation) */
#    define EscRsa4096_NUM_PRIMES 53U
#endif

/***************************************************************
 Distance between p and q: ceil(2^0.1 << (EscRsa4096_RSA_BASE - 1))
 ***************************************************************/
#ifdef EscRsa4096_ENABLE_GENRSA
#    ifdef Esc_HAS_INT64
#        define EscRsa4096_PQ_LOWER_BOUND   2301615986U
#    else
#        define EscRsa4096_PQ_LOWER_BOUND   35120U
#    endif
#endif

/***************************************************************
Extracts the high part of FWORD w and returns it as a FWORD
 ***************************************************************/
#define EscRsa4096_HI_FWORD( w ) ( ( ( w ) >> EscRsa4096_RSA_BASE ) & EscRsa4096_MAX_VAL )

/***************************************************************
Extracts the high part of FWORD w and returns it as a HWORD
 ***************************************************************/
#define EscRsa4096_HI_HWORD( w ) ( (EscRsa4096_HWORD)( ( ( w ) >> EscRsa4096_RSA_BASE ) & EscRsa4096_MAX_VAL ) )

/***************************************************************
Extracts the low part of FWORD w and returns it as a FWORD
 ***************************************************************/
#define EscRsa4096_LO_FWORD( w ) ( ( w ) & EscRsa4096_MAX_VAL )

/***************************************************************
Extracts the low part of FWORD w and returns it as a HWORD
 ***************************************************************/
#define EscRsa4096_LO_HWORD( w ) ( (EscRsa4096_HWORD)( ( w ) & EscRsa4096_MAX_VAL ) )

/***************************************************************
return min(x,y)
 ***************************************************************/
#define EscRsa4096_MIN( x, y ) ( ( ( x ) < ( y ) ) ? ( x ) : ( y ) )

/***************************************************************************
 * 3. DEFINITIONS                                                          *
 ***************************************************************************/

/* Multiplication result field element. */
typedef struct {
    EscRsa4096_HWORD words[ EscRsa4096_RSA_SIZE_WORDS_LONG ];
    /* No carry needed */
} EscRsa4096_FieldElementLongT;

typedef struct {
    /** Data to calculate c = x^e mod m */

    /** Multiplication result */
    EscRsa4096_FieldElementLongT* c;

    /** Modulus */
    const EscRsa4096_FieldElementT* m;

    /** x */
    EscRsa4096_FieldElementT* x;
} EscRsa4096_MultiplyDataT;

#ifdef EscRsa4096_USE_MONTGOM_MUL
typedef struct {
    /** Modulus */
    EscRsa4096_FieldElementT mod;
    /** negative inverse of mod[0] */
    EscRsa4096_HWORD m;
} EscRsa4096_MontGomElementT;
#endif

/* Field element functions */
#ifndef EscRsa4096_ENABLE_STACK_SAVING_INTERFACE
static void
EscRsa4096Fe_ToBytesBE(
    UINT8 dest[],
    const EscRsa4096_FieldElementT* source );

#endif

#if ( !defined( EscRsa4096_USE_MONTGOM_MUL ) ) || defined( EscRsa4096_NO_MONTGOM_SHORT_EXP ) || defined( EscRsa4096_USE_SHORT_EXPONENT ) || defined( EscRsa4096_ENABLE_GENRSA ) || defined( EscRsa4096_ENABLE_CRT )
static void
EscRsa4096Fe_ToLongFe(
    EscRsa4096_FieldElementLongT* dst,
    const EscRsa4096_FieldElementT* src );
#endif

static void
EscRsa4096Fe_FromLongFe(
    EscRsa4096_FieldElementT* dst,
    const EscRsa4096_FieldElementLongT* src );

#if defined( EscRsa4096_USE_SLIDING_WINDOW ) || defined( EscRsa4096_USE_MONTGOM_MUL ) || defined( EscRsa4096_ENABLE_GENRSA ) || defined ( EscRsa4096_ENABLE_CRT )
static void
EscRsa4096Fe_Assign(
    EscRsa4096_FieldElementT* dst,
    const EscRsa4096_FieldElementT* src );
#endif

static BOOL
EscRsa4096Fe_GreaterThan(
    const EscRsa4096_FieldElementT* a,
    const EscRsa4096_FieldElementT* b );

#ifdef EscRsa4096_USE_MONTGOM_MUL
static void
EscRsa4096Fe_MontGom(
    EscRsa4096_FieldElementT* x,
    const EscRsa4096_FieldElementT* y,
    const EscRsa4096_MontGomElementT* mg );

static void
EscRsa4096_MontGom_Init(
    const EscRsa4096_MultiplyDataT* pm,
    EscRsa4096_MontGomElementT* mg );

#endif

static void
EscRsa4096Fe_ModularReduction(
    const EscRsa4096_MultiplyDataT* pm );

static void
EscRsa4096Fe_MultiplyClassically(
    const EscRsa4096_MultiplyDataT* pm );

#if ( !defined( EscRsa4096_USE_MONTGOM_MUL ) ) || defined( EscRsa4096_NO_MONTGOM_SHORT_EXP ) || defined( EscRsa4096_USE_SHORT_EXPONENT ) || defined( EscRsa4096_ENABLE_GENRSA )
static void
EscRsa4096Fe_SquareClassically(
    const EscRsa4096_MultiplyDataT* pm );

static void
EscRsa4096Fe_Square(
    const EscRsa4096_MultiplyDataT* pm );
#endif

static void
EscRsa4096Fe_Multiply(
    const EscRsa4096_MultiplyDataT* pm );

static BOOL
EscRsa4096Fe_IsBitSet(
    const EscRsa4096_FieldElementT* e,
    UINT32 thebit );

static void
EscRsa4096Fe_BigPow(
    const EscRsa4096_MultiplyDataT* pm,
    const EscRsa4096_FieldElementT* e );

#if defined( EscRsa4096_NO_MONTGOM_SHORT_EXP ) || defined( EscRsa4096_USE_SHORT_EXPONENT ) || defined( EscRsa4096_ENABLE_STACK_SAVING_INTERFACE )
#    ifndef EscRsa4096_ENABLE_STACK_SAVING_INTERFACE
static BOOL
EscRsa4096_ShortPow(
    const UINT8 message[],
    const UINT8 modulus[],
    const UINT32 exponent,
    UINT8 result[] );
#    else

static BOOL
EscRsa4096_ShortPow(
    EscRsa4096_FieldElementT* message,
    const EscRsa4096_FieldElementT* modulus,
    const UINT32 exponent );
#    endif
#endif

#if defined ( EscRsa4096_ENABLE_GENRSA ) || defined ( EscRsa4096_ENABLE_CRT )
static SINT8
EscRsa4096Fe_AbsoluteCompare(
    const EscRsa4096_FieldElementT* a,
    const EscRsa4096_FieldElementT* b );

static void
EscRsa4096Fe_Subtract(
    EscRsa4096_FieldElementT* a,
    const EscRsa4096_FieldElementT* b );
#endif

#ifdef EscRsa4096_ENABLE_GENRSA
static void
EscRsa4096Fe_ShiftRight(
    EscRsa4096_FieldElementT* a );

static void
EscRsa4096Fe_SignedShiftRight(
    EscRsa4096_FieldElementT* a,
    EscRsa4096_HWORD* aSign );

static BOOL
EscRsa4096Fe_IsZero(
    const EscRsa4096_FieldElementT* a );

static BOOL
EscRsa4096Fe_IsOne(
    const EscRsa4096_FieldElementT* a );

static void
EscRsa4096Fe_SignedAdd(
    EscRsa4096_FieldElementT* a,
    EscRsa4096_HWORD* aSign,
    const EscRsa4096_FieldElementT* b,
    const EscRsa4096_HWORD bSign );

static void
EscRsa4096Fe_SignedSubtract(
    EscRsa4096_FieldElementT* a,
    EscRsa4096_HWORD* aSign,
    const EscRsa4096_FieldElementT* b,
    const EscRsa4096_HWORD bSign );

static void
EscRsa4096Fe_AddWord(
    EscRsa4096_FieldElementT* c,
    const EscRsa4096_HWORD a );

static EscRsa4096_HWORD
EscRsa4096Fe_ModWord(
    const EscRsa4096_FieldElementT* a,
    EscRsa4096_HWORD w );

static BOOL
EscRsa4096Fe_IsPositive(
    EscRsa4096_HWORD sign,
    const EscRsa4096_FieldElementT* fe);

static BOOL
EscRsa4096Fe_ModInv(
    const EscRsa4096_FieldElementT* x,
    const EscRsa4096_FieldElementT* y,
    EscRsa4096_FieldElementT* result );

static BOOL
EscRsa4096_MillerRabin(
    EscRandom_ContextT* randCtx,
    const EscRsa4096_FieldElementT* n,
    const UINT32 iter,
    BOOL* prime );

static BOOL
EscRsa4096_GeneratePrime(
    EscRandom_ContextT* randCtx,
    EscRsa4096_FieldElementT* n );
#endif

#ifdef EscRsa4096_ENABLE_CRT
static void
EscRsa4096Fe_Add(
    EscRsa4096_FieldElementT* a,
    const EscRsa4096_FieldElementT* b );
#endif

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/

/***************************************************************************
 * 5. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/

#if ( !defined( EscRsa4096_ENABLE_STACK_SAVING_INTERFACE ) ) || defined( EscRsa4096_ENABLE_GENRSA ) || defined( EscRsa4096_ENABLE_CRT )
/***************************************************************
 * Converts a byte array into a word array (big endian)        *
 ***************************************************************/
void
EscRsa4096Fe_FromBytesBE(
    EscRsa4096_FieldElementT* dest,
    const UINT8 source[],
    UINT16 len )
{
    UINT16 i;
    EscRsa4096_HWORD* dstArray;

    dstArray = dest->words;

    for ( i = 0U; i < ( len / EscRsa4096_WORD_SIZE ); i++ ) {
        EscRsa4096_HWORD d;
        d = ( (EscRsa4096_HWORD)source[ ( len - 1U ) - ( i * EscRsa4096_WORD_SIZE ) ] ) |
            ( (EscRsa4096_HWORD)( ( (EscRsa4096_HWORD)source[ ( len - 1U ) - ( ( i * EscRsa4096_WORD_SIZE ) + 1U ) ] ) << 8 ) );

#    if EscRsa4096_WORD_SIZE == 4U
        d |= ( (EscRsa4096_HWORD)( ( (EscRsa4096_HWORD)source[ ( len - 1U ) - ( ( i * EscRsa4096_WORD_SIZE ) + 2U ) ] ) << 16 ) ) |
            ( (EscRsa4096_HWORD)( ( (EscRsa4096_HWORD)source[ ( len - 1U ) - ( ( i * EscRsa4096_WORD_SIZE ) + 3U ) ] ) << 24 ) );
#    endif

        /*lint -save -esym(960,17.4) dstArray is an array*/
        dstArray[ i ] = d;
        /*lint -restore dstArray is an array*/
    }
    for ( i = ( len / EscRsa4096_WORD_SIZE ); i < EscRsa4096_RSA_SIZE_WORDS; i++ ) {
        /*lint -e960 dstArray is an array*/
        dstArray[ i ] = 0U;
        /*lint +e960 dstArray is an array*/
    }
}

#endif

/***************************************************************
 * Converts a word array into a byte array (big endian)        *
 ***************************************************************/
#ifndef EscRsa4096_ENABLE_STACK_SAVING_INTERFACE
static void
EscRsa4096Fe_ToBytesBE(
    UINT8 dest[],
    const EscRsa4096_FieldElementT* source )
{
    UINT16 i;
    EscRsa4096_HWORD v;

    for ( i = 0U; i < EscRsa4096_KEY_BYTES; i++ ) {
        v = source->words[ i >> ( EscRsa4096_WORD_SIZE / 2U ) ];
        v >>= ( (UINT16)( ( i & ( EscRsa4096_WORD_SIZE - 1U ) ) << 3 ) ) & 0xFFU;
        dest[ ( EscRsa4096_KEY_BYTES - 1U ) - i ] = (UINT8)v;
    }
}

#endif

/************************************************************************
 Assigns element src to long element dst
 dst := src
************************************************************************/
#if ( !defined( EscRsa4096_USE_MONTGOM_MUL ) ) || defined( EscRsa4096_NO_MONTGOM_SHORT_EXP ) || defined( EscRsa4096_USE_SHORT_EXPONENT ) || defined( EscRsa4096_ENABLE_GENRSA ) || defined( EscRsa4096_ENABLE_CRT )
static void
EscRsa4096Fe_ToLongFe(
    EscRsa4096_FieldElementLongT* dst,
    const EscRsa4096_FieldElementT* src )
{
    UINT16 i;

    /* copy src */
    for ( i = 0U; i < EscRsa4096_RSA_SIZE_WORDS; i++ ) {
        dst->words[ i ] = src->words[ i ];
    }

    /* fill remaining values with zero */
    for ( i = EscRsa4096_RSA_SIZE_WORDS; i < EscRsa4096_RSA_SIZE_WORDS_LONG; i++ ) {
        dst->words[ i ] = 0U;
    }
}

#endif
/************************************************************************
 Assigns long element src to element dst
 dst := src
************************************************************************/
static void
EscRsa4096Fe_FromLongFe(
    EscRsa4096_FieldElementT* dst,
    const EscRsa4096_FieldElementLongT* src )
{
    UINT16 i;

    /** The value fits into dst */
    Esc_ASSERT( src->words[ EscRsa4096_RSA_SIZE_WORDS ] == 0U );

    /* copy src */
    for ( i = 0U; i < EscRsa4096_RSA_SIZE_WORDS; i++ ) {
        dst->words[ i ] = src->words[ i ];
    }
}

/************************************************************************
 Assigns element src to element dst
 dst := src
************************************************************************/
#if defined( EscRsa4096_USE_SLIDING_WINDOW ) || defined( EscRsa4096_USE_MONTGOM_MUL ) || defined( EscRsa4096_ENABLE_GENRSA ) || defined ( EscRsa4096_ENABLE_CRT )
static void
EscRsa4096Fe_Assign(
    EscRsa4096_FieldElementT* dst,
    const EscRsa4096_FieldElementT* src )
{
    UINT16 i;
    /* copy src */
    for ( i = 0U; i < EscRsa4096_RSA_SIZE_WORDS; i++ ) {
        dst->words[ i ] = src->words[ i ];
    }
}

#endif

/************************************************************************
Main reduction loop (HAC, Algorithm 14.20)
c := c mod m
m must be != 0
************************************************************************/
static void
EscRsa4096Fe_ModularReduction(
    const EscRsa4096_MultiplyDataT* pm )
{
    /*lint --e{661} --e{662} It is confirmed that the array out of bounds warning is a false positive */
    SINT16 mwords_msb;
    SINT16 i;
    UINT16 j;
    EscRsa4096_HWORD q;
    EscRsa4096_HWORD carry;
    EscRsa4096_FWORD ciAndCmin1;
    EscRsa4096_HWORD* cwords;
    const EscRsa4096_HWORD* mwords;
    BOOL finished;

    cwords = pm->c->words;
    mwords = pm->m->words;

    mwords_msb = (SINT16)EscRsa4096_RSA_SIZE_WORDS - 1;
    /*lint -save -esym(960,17.4) cwords, mwords are arrays */
    while ( ( mwords_msb >= 0 ) && ( mwords[ mwords_msb ] == 0U ) ) {
        mwords_msb--;
    }
    Esc_ASSERT( mwords_msb >= 0 );

    for ( i = ( (SINT16)EscRsa4096_RSA_SIZE_WORDS_LONG - 1 ); i > mwords_msb; i-- ) {
        /* if x_i == m_k-1 then q=b-1 else q=(x_i*b + x_i-1) div m_k-1 */
        /* Calc (x_i*b + x_i-1) already, we need it later too */
        ciAndCmin1 = ( (EscRsa4096_FWORD)cwords[ i ] << EscRsa4096_RSA_BASE ) | ( (EscRsa4096_FWORD)cwords[ i - 1 ] );

        if ( cwords[ i ] == mwords[ mwords_msb ] ) {
            q = (EscRsa4096_HWORD)EscRsa4096_MAX_VAL;
        } else {
            q = (EscRsa4096_HWORD)( ( ciAndCmin1 ) / (EscRsa4096_FWORD)mwords[  mwords_msb ] );
        }

        /* while q*(m[k-1] * b + m[k-2]) > x[i]*b^2 + x[i-1] *b + x[i-2] do q--; */
        finished = FALSE;
        while ( finished == FALSE ) {
            EscRsa4096_FWORD qcHi;
            EscRsa4096_FWORD qcLo;

            EscWatchdog_Call( EscWatchdog_STATIC_WATCHDOG_TRIGGER );

            qcHi = (EscRsa4096_FWORD) q * (EscRsa4096_FWORD) mwords[ mwords_msb ];
            if ( mwords_msb > 0 ) {
                qcLo = (EscRsa4096_FWORD) q * (EscRsa4096_FWORD) mwords[ mwords_msb - 1 ];
            } else {
                qcLo = 0U;
            }
            qcHi += EscRsa4096_HI_FWORD( qcLo );

            /* qcHi contains the upper two bytes of q*(m[k-1] * b + m[k-2]),
               ciAndCmin1 the upper two bytes of x[i]*b^2 + x[i-1] */
            if ( qcHi < ciAndCmin1 ) {
                /* q*(m[k-1] * b + m[k-2]) < x[i]*b^2 + x[i-1] *b + x[i-2] */
                finished = TRUE;
            } else if ( ( qcHi == ciAndCmin1 ) && ( EscRsa4096_LO_HWORD( qcLo ) <= cwords[ i - 2 ] ) ) {
                /* q*(m[k-1] * b + m[k-2]) <= x[i]*b^2 + x[i-1] *b + x[i-2] */
                finished = TRUE;
            } else {
                q--;
            }
        }

        /* x = x-q*m*b^(i-k) */
        carry = 0U;
        for ( j = 0U; j <= (UINT16)mwords_msb; j++ ) {
            EscRsa4096_FWORD xhi;
            UINT16 cIdx;

            EscWatchdog_Call( EscWatchdog_STATIC_WATCHDOG_TRIGGER );

            xhi = ( (EscRsa4096_FWORD)q * (EscRsa4096_FWORD)mwords[ j ] ) + (EscRsa4096_FWORD)carry;
            carry = EscRsa4096_HI_HWORD( xhi );
            cIdx = (UINT16)( ( ( (UINT16) i - (UINT16) mwords_msb ) - 1U ) + j );
            xhi = (EscRsa4096_FWORD)cwords[ cIdx ] - EscRsa4096_LO_FWORD( xhi );
            cwords[ cIdx ] = EscRsa4096_LO_HWORD( xhi );

            if ( EscRsa4096_HI_FWORD( xhi ) != 0U ) {
                carry++;
            }
        }

        /* if (x<0) then x=x+m*b^(i-k) */
        if ( cwords[ i ] < carry ) {
            carry = 0U;

            for ( j = 0U; j <= (UINT16) mwords_msb; j++ ) {
                EscRsa4096_FWORD xhi;
                UINT16 cIdx;

                EscWatchdog_Call( EscWatchdog_STATIC_WATCHDOG_TRIGGER );

                cIdx = (UINT16)( ( ( (UINT16) i - (UINT16) mwords_msb ) - 1U ) + j );
                xhi = (EscRsa4096_FWORD)cwords[ cIdx ] + ( (EscRsa4096_FWORD)mwords[ j ] ) + (EscRsa4096_FWORD)carry;
                carry = EscRsa4096_HI_HWORD( xhi );
                cwords[ cIdx ] = EscRsa4096_LO_HWORD( xhi );
            }
        }
        cwords[ i ] = 0U;
        /*lint -restore cwords is an array; out of bounds pointer */
    }
}

/*********************************
  Multiplies two field elements c := c * x.
  We use the integer multiplication algorithm from Menzenes, et. al.
  "Elliptic Curve Cryptography". Algorithm 2.9.
  The algorithm is endianness independent.
 *********************************/
static void
EscRsa4096Fe_MultiplyClassically(
    const EscRsa4096_MultiplyDataT* pm )
{
    EscRsa4096_HWORD i;
    EscRsa4096_HWORD j;
    EscRsa4096_HWORD min;
    EscRsa4096_HWORD temp;
    EscRsa4096_FWORD t;
    EscRsa4096_HWORD b_0, a_0;
    EscRsa4096_FWORD uv_t;
    EscRsa4096_FWORD uv;

    EscRsa4096_FieldElementT inC;
    EscRsa4096_HWORD* awords;
    EscRsa4096_HWORD* bwords;
    EscRsa4096_HWORD* cwords;

    awords = inC.words;
    bwords = pm->x->words;
    cwords = pm->c->words;

    uv_t = 0U;
    t = 0U;

    /* 1.) make a copy of c, clear c. inC:= c; c := 0 */
    Esc_ASSERT( cwords[ EscRsa4096_RSA_SIZE_WORDS ] == 0U );    /* single precision */
    EscRsa4096Fe_FromLongFe( &inC, pm->c );

    for ( i = 0U; i < ( 2U * EscRsa4096_RSA_SIZE_WORDS ); i++ ) {
        b_0 = EscRsa4096_MIN( ( EscRsa4096_RSA_SIZE_WORDS - 1U ), i );
        a_0 = i - b_0;
        min = EscRsa4096_MIN( ( EscRsa4096_RSA_SIZE_WORDS - a_0 ), ( b_0 + 1U ) );
        for ( j = 0U; j < min; j++ ) {
            /* a[a_o +j] * b[b_o - j] */
            /*lint -save -esym(960,17.4) awords and bwords are arrays*/
            uv = (EscRsa4096_FWORD)awords[ a_0 + j ] * (EscRsa4096_FWORD)bwords[ b_0 - j ];
            /*lint -restore awords and bwords are arrays*/
            /* sum on uv_t */
            uv_t += uv;
            /* if overflow save carry to t */
            if ( uv_t < uv ) {
                t++;
            }
        }

        /* c[i] = v, v = u, u = t, t = 0 */
        /*lint -save -esym(960,17.4) cwords is an array*/
        cwords[ i ] = EscRsa4096_LO_HWORD( uv_t );
        /*lint -restore cwords is an array*/
        temp = EscRsa4096_HI_HWORD( uv_t );
        uv_t = temp;
        uv_t |= t << EscRsa4096_RSA_BASE;
        t = 0U;
    }
}

/*********************************
  Squares two field elements c := c * c
  The input c has the maximum length of RSA_SIZE_WORDS.
  By maintaining another copy of c in RAM,
  the multiplication algorithm can be used for this task.
 *********************************/
#if ( !defined( EscRsa4096_USE_MONTGOM_MUL ) ) || defined( EscRsa4096_NO_MONTGOM_SHORT_EXP ) || defined( EscRsa4096_USE_SHORT_EXPONENT ) || defined( EscRsa4096_ENABLE_GENRSA )
static void
EscRsa4096Fe_SquareClassically(
    const EscRsa4096_MultiplyDataT* pm )
{
    /* declarations */
#    if ( !defined( EscRsa4096_NO_MONTGOM_SHORT_EXP ) ) && ( !defined( EscRsa4096_USE_SLIDING_WINDOW ) )
    EscRsa4096_HWORD min;
    EscRsa4096_HWORD b_0, a_0;
#    endif
    EscRsa4096_HWORD i;
    EscRsa4096_HWORD j;
    EscRsa4096_FWORD uv;
    EscRsa4096_FWORD t = 0U;
    EscRsa4096_FWORD uv_t = 0U;

    EscRsa4096_FieldElementT inC;
    EscRsa4096_HWORD* cwords;

    cwords = pm->c->words;

    /* 1.) inC := c; c := 0 */
    Esc_ASSERT( cwords[ EscRsa4096_RSA_SIZE_WORDS ] == 0U );    /* single precision */
    EscRsa4096Fe_FromLongFe( &inC, pm->c );

#    if defined( EscRsa4096_NO_MONTGOM_SHORT_EXP ) || defined( EscRsa4096_USE_SLIDING_WINDOW )
    for ( i = 0U; i < EscRsa4096_RSA_SIZE_WORDS; i++ ) {
        const EscRsa4096_HWORD max = ( i + 1U ) / 2U;
        const EscRsa4096_HWORD half = i / 2U;

        for ( j = 0U; j <= half; j++ ) {
            /* a[a_o + j] * b[b_o - j] */
            uv = (EscRsa4096_FWORD)inC.words[ j ] * (EscRsa4096_FWORD)inC.words[ i - j ];
            /* sum on uv_t */
            uv_t += uv;
            /* if overflow save carry to t */
            if ( uv_t < uv ) {
                t++;
            }
            if ( j < max ) {
                /* sum on uv_t */
                uv_t += uv;
                /* if overflow save carry to t */
                if ( uv_t < uv ) {
                    t++;
                }
            }
        }

        /* c[i] = v, v = u, u = t, t = 0 */
        /*lint -save -esym(960,17.4) cwords is an array*/
        cwords[ i ] = EscRsa4096_LO_HWORD( uv_t );
        /*lint -restore cwords is an array*/
        uv_t >>= EscRsa4096_RSA_BASE;
        uv_t |= t << EscRsa4096_RSA_BASE;
        t = 0U;
    }
    for (; i < ( EscRsa4096_RSA_SIZE_WORDS * 2U ); i++ ) {
        const EscRsa4096_HWORD max = ( i + 1U ) / 2U;
        const EscRsa4096_HWORD half = i / 2U;

        for ( j = ( ( i - EscRsa4096_RSA_SIZE_WORDS ) + 1U ); j <= half; j++ ) {
            /* a[a_o + j] * b[b_o - j] */
            uv = (EscRsa4096_FWORD)inC.words[ j ] * (EscRsa4096_FWORD)inC.words[ i - j ];
            /* sum on uv_t */
            uv_t += uv;
            /* if overflow save carry to t */
            if ( uv_t < uv ) {
                t++;
            }
            if ( j < max ) {
                /* sum on uv_t */
                uv_t += uv;
                /* if overflow save carry to t */
                if ( uv_t < uv ) {
                    t++;
                }
            }
        }

        /*lint -save -esym(960,17.4) cwords is an array*/
        cwords[ i ] = EscRsa4096_LO_HWORD( uv_t );
        /*lint -restore cwords is an array*/
        uv_t >>= EscRsa4096_RSA_BASE;
        uv_t |= t << EscRsa4096_RSA_BASE;
        t = 0U;
    }
#    else
    for ( i = 0U; i < ( 2U * EscRsa4096_RSA_SIZE_WORDS ); i++ ) {
        b_0 = EscRsa4096_MIN( ( EscRsa4096_RSA_SIZE_WORDS - 1U ), i );
        a_0 = i - b_0;
        min = EscRsa4096_MIN( ( EscRsa4096_RSA_SIZE_WORDS - a_0 ), ( b_0 + 1U ) );
        for ( j = 0U; j < min; j++ ) {
            /* a[a_o + j] * b[b_o - j] */
            uv = (EscRsa4096_FWORD)inC.words[ a_0 + j ] * (EscRsa4096_FWORD)inC.words[ b_0 - j ];
            /* sum on uv_t */
            uv_t += uv;
            /* if overflow save carry to t */
            if ( uv_t < uv ) {
                t++;
            }
        }

        /* c[i] = v, v = u, u = t, t = 0 */
        /*lint -save -esym(960,17.4) cwords is an array*/
        cwords[ i ] = EscRsa4096_LO_HWORD( uv_t );
        /*lint -restore cwords is an array*/
        uv_t >>= EscRsa4096_RSA_BASE;
        uv_t |= t << EscRsa4096_RSA_BASE;
        t = 0U;
    }
#    endif
}

/************************************************************************
 * c = c * c mod m                                                           *
 ************************************************************************/
static void
EscRsa4096Fe_Square(
    const EscRsa4096_MultiplyDataT* pm )
{
    EscRsa4096Fe_SquareClassically( pm );

    EscRsa4096Fe_ModularReduction( pm );
}

#endif

/************************************************************************
 * c = c * x mod m                                                           *
 ************************************************************************/
static void
EscRsa4096Fe_Multiply(
    const EscRsa4096_MultiplyDataT* pm )
{
    EscRsa4096Fe_MultiplyClassically( pm );

    /* classical modular reduction */
    EscRsa4096Fe_ModularReduction( pm );
}

/** returns whether or not the specified bit is set */
static BOOL
EscRsa4096Fe_IsBitSet(
    const EscRsa4096_FieldElementT* e,
    UINT32 thebit )
{
    BOOL isSet;
    if ( ( e->words[ ( EscRsa4096_RSA_SIZE_WORDS - 1U ) - ( thebit / EscRsa4096_RSA_BASE ) ] & (EscRsa4096_HWORD)( (EscRsa4096_HWORD)1UL << ( ( EscRsa4096_RSA_BASE - 1U ) - ( thebit % EscRsa4096_RSA_BASE ) ) ) ) != 0U ) {
        isSet = TRUE;
    } else {
        isSet = FALSE;
    }

    return isSet;
}

/*********************************************************************
 * computes the power of a field element by the given exponent       *
 * x = x^e mod m                                                     *
 *                                                                   *
 * the exponent must not be 0                                        *
 *********************************************************************/
#ifdef EscRsa4096_USE_MONTGOM_MUL
#    ifdef EscRsa4096_USE_SLIDING_WINDOW
static void
EscRsa4096Fe_BigPow(
    const EscRsa4096_MultiplyDataT* pm,
    const EscRsa4096_FieldElementT* e )
{
    EscRsa4096_FieldElementT x_, y_, one, x_2;
    EscRsa4096_MontGomElementT mg;
    UINT32 thebit = 0U;
    UINT32 i, n, windowsize, j;
    UINT16 exponent;
    /* array w/ size 2^k_m */
    EscRsa4096_FieldElementT fe_array[ (UINT16)( (UINT16)( (UINT16)2U << ( EscRsa4096_WINDOW_SIZE - 1U ) ) ) ];
    /* build word one */
    for ( i = 1U; i < EscRsa4096_RSA_SIZE_WORDS; i++ ) {
        one.words[ i ] = 0U;
    }
    one.words[ 0 ] = 1U;

    /* build R = 2^mod_bit_size */
    for ( i = 0U; i < EscRsa4096_RSA_SIZE_WORDS_LONG; i++ ) {
        pm->c->words[ i ] = 0U;
    }
    pm->c->words[ EscRsa4096_RSA_SIZE_WORDS ] = 1U;

    EscRsa4096Fe_ModularReduction( pm );
    /* x_ = R mod n */
    EscRsa4096Fe_FromLongFe( &x_, pm->c );
    /* pm.c = c * x mod m  =  R * x mod m */
    EscRsa4096Fe_Multiply( pm );
    /* y_ = x * R */
    EscRsa4096Fe_FromLongFe( &y_, pm->c );

    /* init Montgomery variables */
    EscRsa4096_MontGom_Init( pm, &mg );

    /* fe_array[0] will not be used!! */
    EscRsa4096Fe_Assign( &fe_array[ 0 ], &y_ );
    EscRsa4096Fe_MontGom( &y_, &y_, &mg );
    EscRsa4096Fe_Assign( &x_2, &fe_array[ 0 ] );
    /* precomputation - i < 2^k_m */
    for ( i = 1U; i < (UINT16)( (UINT16)( (UINT16)2U << ( EscRsa4096_WINDOW_SIZE - 1U ) ) ); i++ ) {
        EscRsa4096Fe_MontGom( &x_2, &y_, &mg );
        EscRsa4096Fe_Assign( &fe_array[ i ], &x_2 );
    }
    EscRsa4096Fe_Assign( &x_, &fe_array[ 0 ] );
    while ( EscRsa4096Fe_IsBitSet( e, thebit ) == FALSE ) {
        thebit++;
    }
    thebit++;

    while ( thebit < EscRsa4096_KEY_BITS ) {
        /* if Exponent = 0 double only */
        if ( EscRsa4096Fe_IsBitSet( e, thebit ) == FALSE ) {
            EscRsa4096Fe_MontGom( &x_, &x_, &mg );
            thebit++;
        } else {
            /* build window */
            exponent = 0U;
            n = EscRsa4096_WINDOW_SIZE - 1U;
            windowsize = EscRsa4096_WINDOW_SIZE;
            for ( j = thebit; ( j < ( thebit + EscRsa4096_WINDOW_SIZE ) ) && ( j < EscRsa4096_KEY_BITS ); j++ ) {
                /* build temporary exponent */
                if ( EscRsa4096Fe_IsBitSet( e, j ) ) {
                    /*lint -save -e701 n is unsigned*/
                    exponent |= (UINT16)( (UINT16)1U << n );
                    /*lint -restore */
                }
                n--;
            }
            thebit += EscRsa4096_WINDOW_SIZE;
            /* make window odd and adjust i, reduce window size */
            while ( ( exponent % 2U ) == 0U ) {
                exponent = exponent >> 1U;
                thebit--;
                windowsize--;
            }

            /* double */
            for ( j = 0U; j < windowsize; j++ ) {
                EscRsa4096Fe_MontGom( &x_, &x_, &mg );
            }
            /* add */
            EscRsa4096Fe_MontGom( &x_, &fe_array[ (UINT16)( exponent / 2U ) ], &mg );
        }
    }
    EscRsa4096Fe_MontGom( &x_, &one, &mg );
    EscRsa4096Fe_Assign( pm->x, &x_ );
}

#    else /* EscRsa4096_USE_SLIDING_WINDOW */
static void
EscRsa4096Fe_BigPow(
    const EscRsa4096_MultiplyDataT* pm,
    const EscRsa4096_FieldElementT* e )
{
    EscRsa4096_FieldElementT x_, y_, one;
    EscRsa4096_MontGomElementT mg;
    UINT32 thebit = 0U;
    UINT16 i;
    /* build word one */
    for ( i = 1U; i < EscRsa4096_RSA_SIZE_WORDS; i++ ) {
        one.words[ i ] = 0U;
    }
    one.words[ 0 ] = 1U;

    /* build R = 2^mod_bit_size */
    for ( i = 0U; i < EscRsa4096_RSA_SIZE_WORDS_LONG; i++ ) {
        pm->c->words[ i ] = 0U;
    }
    pm->c->words[ EscRsa4096_RSA_SIZE_WORDS ] = 1U;

    EscRsa4096Fe_ModularReduction( pm );
    /* x_ = R mod n */
    EscRsa4096Fe_FromLongFe( &x_, pm->c );
    /* pm.c = c * x mod m  =  R * x mod m */
    EscRsa4096Fe_Multiply( pm );
    /* y_ = x * R */
    EscRsa4096Fe_FromLongFe( &y_, pm->c );

    /* init Montgomery variables */
    EscRsa4096_MontGom_Init( pm, &mg );

    /* find first bit in exponent != 0 */
    while ( EscRsa4096Fe_IsBitSet( e, thebit ) == FALSE ) {
        thebit++;
    }

    /* square and multiply */
    while ( thebit < EscRsa4096_KEY_BITS ) {
        /* c := c * c mod m */
        EscRsa4096Fe_MontGom( &x_, &x_, &mg );

        /* if e_i = 1 then c := c * x mod m */
        if ( EscRsa4096Fe_IsBitSet( e, thebit ) ) {
            /* c := c * x mod m */
            EscRsa4096Fe_MontGom( &x_, &y_, &mg );
        }
        thebit++;
    }
    EscRsa4096Fe_MontGom( &x_, &one, &mg );
    /* x := c. We could actually copy the result directly into the
       output byte array, but stick to this for maintainability. */
    EscRsa4096Fe_Assign( pm->x, &x_ );
}

#    endif

#else /* EscRsa4096_USE_MONTGOM_MUL */
#    ifdef EscRsa4096_USE_SLIDING_WINDOW
/* sliding window */
static void
EscRsa4096Fe_BigPow(
    const EscRsa4096_MultiplyDataT* pm,
    const EscRsa4096_FieldElementT* e )
{
    UINT32 thebit = 0U;
    UINT32 i, n, windowsize, j;
    UINT16 exponent;
    /* array w/ size 2^k_m */
    EscRsa4096_FieldElementT fe_array[ (UINT16)( (UINT16)( (UINT16)2U << ( EscRsa4096_WINDOW_SIZE - 1U ) ) ) ];
    EscRsa4096Fe_ToLongFe( pm->c, pm->x );
    EscRsa4096Fe_FromLongFe( &fe_array[ 0 ], pm->c );
    EscRsa4096Fe_Square( pm );
    EscRsa4096Fe_FromLongFe( pm->x, pm->c );
    EscRsa4096Fe_ToLongFe( pm->c, &fe_array[ 0 ] );

    /* precomputation - i < 2^k_m */
    for ( i = 1U; i < (UINT16)( (UINT16)( (UINT16)2U << ( EscRsa4096_WINDOW_SIZE - 1U ) ) ); i++ ) {
        EscRsa4096Fe_Multiply( pm );
        EscRsa4096Fe_FromLongFe( &fe_array[ i ], pm->c );
    }
    EscRsa4096Fe_ToLongFe( pm->c, &fe_array[ 0 ] );

    while ( EscRsa4096Fe_IsBitSet( e, thebit ) == FALSE ) {
        thebit++;
    }

    /* c == x => MSB has already been processed */
    thebit++;

    while ( thebit < EscRsa4096_KEY_BITS ) {
        /* if Exponent = 0 double only */
        if ( EscRsa4096Fe_IsBitSet( e, thebit ) == FALSE ) {
            EscRsa4096Fe_Square( pm );
            thebit++;
        } else {
            /* build window */
            exponent = 0U;
            n = EscRsa4096_WINDOW_SIZE - 1U;
            windowsize = EscRsa4096_WINDOW_SIZE;
            for ( j = thebit; ( j < ( thebit + EscRsa4096_WINDOW_SIZE ) ) && ( j < EscRsa4096_KEY_BITS ); j++ ) {
                /* build temporary exponent */
                if ( EscRsa4096Fe_IsBitSet( e, j ) ) {
                    /*lint -save -e701 n is unsigned*/
                    exponent |= (UINT16)( (UINT16)1U << n );
                    /*lint -restore */
                }
                n--;
            }
            thebit += EscRsa4096_WINDOW_SIZE;
            /* make window odd and adjust i, reduce window size */
            while ( ( exponent & 1U ) == 0U ) {
                exponent >>= 1U;
                thebit--;
                windowsize--;
            }

            /* square */
            for ( j = 0U; j < windowsize; j++ ) {
                EscRsa4096Fe_Square( pm );
            }
            /* multiply */
            EscRsa4096Fe_Assign( pm->x, &fe_array[ (UINT16)( exponent / 2U ) ] );
            EscRsa4096Fe_Multiply( pm );
        }
    }
    EscRsa4096Fe_FromLongFe( pm->x, pm->c );
}

#    else /* EscRsa4096_USE_SLIDING_WINDOW */
/* square and multiply */
static void
EscRsa4096Fe_BigPow(
    const EscRsa4096_MultiplyDataT* pm,
    const EscRsa4096_FieldElementT* e )
{
    /* Perform square and multiply. We left-shift the exponent
       and examine the most significant bit */
    UINT32 thebit = 0U;

    /* c := x */
    EscRsa4096Fe_ToLongFe( pm->c, pm->x );

    /* find first bit in exponent != 0 */
    while ( EscRsa4096Fe_IsBitSet( e, thebit ) == FALSE ) {
        thebit++;
    }

    /* c == x => MSB has already been processed */
    thebit++;

    /* square and multiply */
    while ( thebit < EscRsa4096_KEY_BITS ) {
        /* c := c * c mod m */
        EscRsa4096Fe_Square( pm );

        /* if e_i = 1 then c := c * x mod m */
        if ( EscRsa4096Fe_IsBitSet( e, thebit ) ) {
            /* c := c * x mod m */
            EscRsa4096Fe_Multiply( pm );
        }
        thebit++;
    }

    /* x := c. We could actually copy the result directly into the
       output byte array, but stick to this for maintainability. */
    EscRsa4096Fe_FromLongFe( pm->x, pm->c );
}

#    endif
#endif

/***********************************************
 * check if value of field element a is larger *
 * than the value of field element b           *
 ***********************************************/
static BOOL
EscRsa4096Fe_GreaterThan(
    const EscRsa4096_FieldElementT* a,
    const EscRsa4096_FieldElementT* b )
{
    SINT16 i;
    BOOL isGreater = FALSE;

    for ( i = ( (SINT16)EscRsa4096_RSA_SIZE_WORDS - 1 ); i >= 0; i-- ) {
        if ( a->words[ i ] != b->words[ i ] ) {
            if ( a->words[ i ] > b->words[ i ] ) {
                isGreater = TRUE;
            } else {
                isGreater = FALSE;
            }
            break;
        }
    }

    return isGreater;
}

#ifdef EscRsa4096_USE_MONTGOM_MUL
static void
EscRsa4096_MontGom_Init(
    const EscRsa4096_MultiplyDataT* pm,
    EscRsa4096_MontGomElementT* mg )
{
    UINT8 i;
    EscRsa4096_HWORD y[ EscRsa4096_RSA_BASE + 1U ];
    /* assign modulus */
    EscRsa4096Fe_Assign( &mg->mod, pm->m );
    mg->m = pm->m->words[ 0U ];
    /* calculate negative inverse of least significant word of the modulus mod 2^word_size
       see EUROCRYPT 90, Lecture Notes in ComputerScience, No. 473, pages 230-244 */
    y[ 1 ] = 1U;
    for ( i = 2U; i <= EscRsa4096_RSA_BASE; i++ ) {
        /*lint -save -e701 (i-1) is unsigned*/
        if ( i != EscRsa4096_RSA_BASE ) {
            y[ 0U ] = (EscRsa4096_HWORD)( ( mg->m * y[ i - 1U ] ) % ( (EscRsa4096_HWORD)( (EscRsa4096_HWORD)( 1U ) << i ) ) );
        } else {
            y[ 0U ] = (EscRsa4096_HWORD)( mg->m * y[ i - 1U ] );
        }
        if ( y[ 0U ] < (EscRsa4096_HWORD)( ( (EscRsa4096_HWORD)1U << ( i - 1U ) ) ) ) {
            y[ i ] = y[ i - 1U ];
        } else {
            y[ i ] = y[ i - 1U ] + ( (EscRsa4096_HWORD)( (EscRsa4096_HWORD)1U << ( i - 1U ) ) );
        }
        /*lint -restore */
    }
    /* calculate -y[RSA_BASE] mod 2^WORD_SIZE */
    mg->m = ( EscRsa4096_HWORD ) ~y[ EscRsa4096_RSA_BASE ];
    mg->m++;
}

/** Montgomery Multiplication w/ CIOS algorithm, x = MontGom(x, y)
 *  see Analyzing and Comparing Montgomery Multiplication Algorithms, Cetin Kaya Koc and Tolga Acar
 **/
static void
EscRsa4096Fe_MontGom(
    EscRsa4096_FieldElementT* x,
    const EscRsa4096_FieldElementT* y,
    const EscRsa4096_MontGomElementT* mg )
{
    /* Initialization */
    EscRsa4096_HWORD i, j, n, uv_tmp;
    EscRsa4096_FWORD uv;
    EscRsa4096_HWORD c[ EscRsa4096_RSA_SIZE_WORDS + 2U ];
    EscRsa4096_HWORD t[ EscRsa4096_RSA_SIZE_WORDS + 2U ];

    /* zero c array */
    for ( i = 0U; i < ( EscRsa4096_RSA_SIZE_WORDS + 2U ); i++ ) {
        c[ i ] = 0U;
    }

    /* CIOS algorithm */
    for ( i = 0U; i < EscRsa4096_RSA_SIZE_WORDS; i++ ) {
        uv = 0U;

        for ( j = 0U; j < EscRsa4096_RSA_SIZE_WORDS; j++ ) {
            uv_tmp = EscRsa4096_HI_HWORD( uv );
            uv = (EscRsa4096_FWORD)uv_tmp + ( (EscRsa4096_FWORD)x->words[ j ] * (EscRsa4096_FWORD)y->words[ i ] ) + (EscRsa4096_FWORD)c[ j ];
            c[ j ] = EscRsa4096_LO_HWORD( uv );
        }

        uv_tmp = EscRsa4096_HI_HWORD( uv );
        uv = (EscRsa4096_FWORD)c[ EscRsa4096_RSA_SIZE_WORDS ] + (EscRsa4096_FWORD)uv_tmp;
        c[ EscRsa4096_RSA_SIZE_WORDS ] = EscRsa4096_LO_HWORD( uv );
        c[ EscRsa4096_RSA_SIZE_WORDS + 1U ] = EscRsa4096_HI_HWORD( uv );
        n = (EscRsa4096_HWORD)( c[ 0U ] * mg->m );

        uv = (EscRsa4096_FWORD)c[ 0U ] + ( (EscRsa4096_FWORD)n * (EscRsa4096_FWORD)mg->mod.words[ 0U ] );
        for ( j = 1U; j < EscRsa4096_RSA_SIZE_WORDS; j++ ) {
            uv_tmp = EscRsa4096_HI_HWORD( uv );
            uv = (EscRsa4096_FWORD)uv_tmp + ( (EscRsa4096_FWORD)n * (EscRsa4096_FWORD)mg->mod.words[ j ] ) + (EscRsa4096_FWORD)c[ j ];
            c[ j - 1U ] = EscRsa4096_LO_HWORD( uv );
        }

        uv_tmp = EscRsa4096_HI_HWORD( uv );
        uv = (EscRsa4096_FWORD)c[ EscRsa4096_RSA_SIZE_WORDS ] + (EscRsa4096_FWORD)uv_tmp;
        c[ EscRsa4096_RSA_SIZE_WORDS - 1U ] = EscRsa4096_LO_HWORD( uv );
        c[ EscRsa4096_RSA_SIZE_WORDS ] = c[ EscRsa4096_RSA_SIZE_WORDS + 1U ] + EscRsa4096_HI_HWORD( uv );
    }

    /* third step of Montgomery, subtract modulus if necessary */
    uv = 0U;
    for ( i = 0U; i < EscRsa4096_RSA_SIZE_WORDS; i++ ) {
        /* subtract mod and save carry in highword of uv (tricky one ;) ) */
        uv_tmp = (EscRsa4096_HWORD)( (EscRsa4096_HWORD)( ~EscRsa4096_HI_HWORD( uv ) ) + 1U );
        uv = ( (EscRsa4096_FWORD)c[ i ] - (EscRsa4096_FWORD)mg->mod.words[ i ] ) - (EscRsa4096_FWORD)uv_tmp;
        t[ i ] = EscRsa4096_LO_HWORD( uv );
    }
    /* last carry */
    uv_tmp = (EscRsa4096_HWORD)( (EscRsa4096_HWORD)( ~EscRsa4096_HI_HWORD( uv ) ) + 1U );
    uv = (EscRsa4096_FWORD)c[ EscRsa4096_RSA_SIZE_WORDS ] - (EscRsa4096_FWORD)uv_tmp;
    t[ EscRsa4096_RSA_SIZE_WORDS ] = EscRsa4096_LO_HWORD( uv );

    /* c > mod ? return t:return c */
    if ( EscRsa4096_HI_HWORD( uv ) == 0U ) {
        for ( i = 0U; i < EscRsa4096_RSA_SIZE_WORDS; i++ ) {
            x->words[ i ] = t[ i ];
        }
    } else {
        for ( i = 0U; i < EscRsa4096_RSA_SIZE_WORDS; i++ ) {
            x->words[ i ] = c[ i ];
        }
    }
}

#endif

#if defined( EscRsa4096_NO_MONTGOM_SHORT_EXP ) || defined( EscRsa4096_USE_SHORT_EXPONENT ) || defined( EscRsa4096_ENABLE_STACK_SAVING_INTERFACE )
/***************************************************************************
 * Modular exponentiation for very small exponents. c = m^e mod n           *
 * (expects big endian input)                                              *
 * maximum exponent = (2^32)-1                                             *
 ***************************************************************************/
#    ifndef EscRsa4096_ENABLE_STACK_SAVING_INTERFACE
static BOOL
EscRsa4096_ShortPow(
    const UINT8 message[],
    const UINT8 modulus[],
    const UINT32 exponent,
    UINT8 result[] )
#    else
static BOOL
EscRsa4096_ShortPow(
    EscRsa4096_FieldElementT* message,
    const EscRsa4096_FieldElementT* modulus,
    const UINT32 exponent )
#    endif
{
    /* declarations */
    BOOL hasFailed = TRUE;
    SINT8 thebit = 31;
    EscRsa4096_FieldElementLongT multiplication_result;
    EscRsa4096_MultiplyDataT mult;
#    ifndef EscRsa4096_ENABLE_STACK_SAVING_INTERFACE
    EscRsa4096_FieldElementT x, m;
#    endif

    mult.c = &multiplication_result;
#    ifdef EscRsa4096_ENABLE_STACK_SAVING_INTERFACE
    mult.m = modulus;
    mult.x = message;
#    else
    mult.m = &m;
    mult.x = &x;
#    endif

    if ( ( message != 0 ) &&
         ( modulus != 0 )
#    ifndef EscRsa4096_ENABLE_STACK_SAVING_INTERFACE
         && ( result != 0 )
#    endif
         && ( exponent != 0U ) )
    {
#    ifndef EscRsa4096_ENABLE_STACK_SAVING_INTERFACE
        EscRsa4096Fe_FromBytesBE( &x, message, EscRsa4096_KEY_BYTES );

        /* We never calculate with the modulus directly, only the
            normalized version. */
        EscRsa4096Fe_FromBytesBE( &m, modulus, EscRsa4096_KEY_BYTES );
#    endif

        if ( mult.m->words[ EscRsa4096_RSA_SIZE_WORDS - 1U ] != 0U ) {
            /* perform parameter and plausibility check   */
            /* make sure modulus > signature */
            if ( EscRsa4096Fe_GreaterThan( mult.m, mult.x ) == TRUE ) {
                EscWatchdog_Call( EscWatchdog_STATIC_WATCHDOG_TRIGGER );

                /* c := x */
                EscRsa4096Fe_ToLongFe( mult.c, mult.x );

                /* find first bit in exponent != 0 */
                while ( ( exponent & (UINT32)( 1UL << thebit ) ) == 0U ) {
                    thebit--;
                }

                /* c == x => MSB has already been processed */
                thebit--;

                /* square and multiply */
                while ( thebit >= 0 ) {
                    /* c := c * c mod m */
                    EscRsa4096Fe_Square( &mult );

                    /* if e_i = 1 then c := c * x mod m */
                    if ( ( exponent & (UINT32)( 1UL << thebit ) ) != 0U ) {
                        /* c := c * x mod m */
                        EscRsa4096Fe_Multiply( &mult );
                    }
                    thebit--;
                }

                /* x := c. We could actually copy the result directly into the
                    output byte array, but stick to this for maintainability. */
                EscRsa4096Fe_FromLongFe( mult.x, mult.c );

                EscWatchdog_Call( EscWatchdog_STATIC_WATCHDOG_TRIGGER );

#    ifndef EscRsa4096_ENABLE_STACK_SAVING_INTERFACE
                EscRsa4096Fe_ToBytesBE( result, mult.x );
#    endif
                hasFailed = FALSE;
            }
        }
    }
    return hasFailed;
}

#endif

/***************************************************************************
 * Modular exponentiation c = m^e mod n (expects big endian input)          *
 ***************************************************************************/
#ifndef EscRsa4096_ENABLE_STACK_SAVING_INTERFACE
BOOL
EscRsa4096_ModExpLong(
    const UINT8 message[],
    const UINT8 modulus[],
    const UINT8 exponent[],
    UINT8 result[] )
#else
BOOL
EscRsa4096_ModExpLongSaveStack(
    EscRsa4096_FieldElementT* message,
    const EscRsa4096_FieldElementT* modulus,
    const EscRsa4096_FieldElementT* exponent_field )
#endif
{
    /* declarations */
    BOOL hasFailed = TRUE;
    EscRsa4096_FieldElementLongT multiplication_result;
    EscRsa4096_MultiplyDataT mult;
#ifndef EscRsa4096_ENABLE_STACK_SAVING_INTERFACE
    EscRsa4096_FieldElementT x, m;
    EscRsa4096_FieldElementT exponent_field;
#endif

    EscRsa4096_HWORD sum = 0U;
    UINT32 i;

    mult.c = &multiplication_result;
#ifdef EscRsa4096_ENABLE_STACK_SAVING_INTERFACE
    mult.m = modulus;
    mult.x = message;
#else
    mult.m = &m;
    mult.x = &x;
#endif

    if ( ( message != 0 ) &&
         ( modulus != 0 )
#ifndef EscRsa4096_ENABLE_STACK_SAVING_INTERFACE
         && ( result != 0 )
         && ( exponent != 0 ) )
#else
         && ( exponent_field != 0 ) )
#endif
    {
#ifndef EscRsa4096_ENABLE_STACK_SAVING_INTERFACE
        EscRsa4096Fe_FromBytesBE( &exponent_field, exponent, EscRsa4096_KEY_BYTES );
#endif
        /* the exponent has to be at least 1 */
        for ( i = 0U; i < EscRsa4096_RSA_SIZE_WORDS; i++ ) {
#ifdef EscRsa4096_ENABLE_STACK_SAVING_INTERFACE
            sum |= exponent_field->words[ i ];
#else
            sum |= exponent_field.words[ i ];
#endif
        }
        if ( sum != 0U ) {
#ifndef EscRsa4096_ENABLE_STACK_SAVING_INTERFACE
            EscRsa4096Fe_FromBytesBE( &x, message, EscRsa4096_KEY_BYTES );

            /* We never calculate with the modulus directly, only the
               normalized version. */
            EscRsa4096Fe_FromBytesBE( &m, modulus, EscRsa4096_KEY_BYTES );
#endif

            if ( mult.m->words[ EscRsa4096_RSA_SIZE_WORDS - 1U ] != 0U ) {
                /* perform parameter and plausibility check   */
                /* make sure modulus > signature */
                if ( EscRsa4096Fe_GreaterThan( mult.m, mult.x ) == TRUE ) {
                    EscWatchdog_Call( EscWatchdog_STATIC_WATCHDOG_TRIGGER );

#ifdef EscRsa4096_ENABLE_STACK_SAVING_INTERFACE
                    EscRsa4096Fe_BigPow( &mult, exponent_field );
#else
                    EscRsa4096Fe_BigPow( &mult, &exponent_field );
#endif
                    EscWatchdog_Call( EscWatchdog_STATIC_WATCHDOG_TRIGGER );

#ifndef EscRsa4096_ENABLE_STACK_SAVING_INTERFACE
                    EscRsa4096Fe_ToBytesBE( result, mult.x );
#endif
                    hasFailed = FALSE;
                }
            }
        }
    }
    /* parameter check */
    return hasFailed;
}

/***************************************************************************
 * Modular exponentiation for small exponents. c = m^e mod n                *
 * (expects big endian input)                                              *
 ***************************************************************************/
#ifndef EscRsa4096_ENABLE_STACK_SAVING_INTERFACE
BOOL
EscRsa4096_ModExp(
    const UINT8 message[],
    const UINT8 modulus[],
    const UINT32 exponent,
    UINT8 result[] )
#else
BOOL
EscRsa4096_ModExpSaveStack(
    EscRsa4096_FieldElementT* message,
    const EscRsa4096_FieldElementT* modulus,
    const UINT32 exponent )
#endif
{
    BOOL hasFailed;

#if defined( EscRsa4096_ENABLE_STACK_SAVING_INTERFACE )
    hasFailed = EscRsa4096_ShortPow( message, modulus, exponent );
#elif defined( EscRsa4096_USE_SHORT_EXPONENT )
    hasFailed = EscRsa4096_ShortPow( message, modulus, exponent, result );
#else /* EscRsa4096_USE_SHORT_EXPONENT / EscRsa4096_ENABLE_STACK_SAVING_INTERFACE */
#    ifdef EscRsa4096_NO_MONTGOM_SHORT_EXP
    if ( exponent == 3U ) {
        hasFailed = EscRsa4096_ShortPow( message, modulus, exponent, result );
    } else {
#    endif /* EscRsa4096_NO_MONTGOM_SHORT_EXP */

        UINT8 longexp[ EscRsa4096_KEY_BYTES ];
        UINT32 i;

        for ( i = 0U; i < ( EscRsa4096_KEY_BYTES - 4U ); i++ ) {
            longexp[ i ] = 0U;
        }
        longexp[ EscRsa4096_KEY_BYTES - 1U ] = (UINT8)( ( exponent ) & 0xFFU );
        longexp[ EscRsa4096_KEY_BYTES - 2U ] = (UINT8)( ( exponent >> 8 ) & 0xFFU );
        longexp[ EscRsa4096_KEY_BYTES - 3U ] = (UINT8)( ( exponent >> 16 ) & 0xFFU );
        longexp[ EscRsa4096_KEY_BYTES - 4U ] = (UINT8)( ( exponent >> 24 ) & 0xFFU );

        hasFailed = EscRsa4096_ModExpLong( message, modulus, longexp, result );

#    ifdef EscRsa4096_NO_MONTGOM_SHORT_EXP
    }
#    endif /* EscRsa4096_NO_MONTGOM_SHORT_EXP */

#endif /* EscRsa4096_USE_SHORT_EXPONENT */
    return hasFailed;
}

#if defined ( EscRsa4096_ENABLE_GENRSA ) || defined ( EscRsa4096_ENABLE_CRT )
/*******************************************
 * compares two field elements `a' and `b' *
  +1: a>b
  -1: a<b
   0: a==b
 *******************************************/
static SINT8
EscRsa4096Fe_AbsoluteCompare(
    const EscRsa4096_FieldElementT* a,
    const EscRsa4096_FieldElementT* b )
{
    SINT16 i;
    SINT8 compResult = 0;

    /* compare the corresponding words until a different word pair is found */
    for ( i = (SINT16)( (SINT16)EscRsa4096_RSA_SIZE_WORDS - 1 ); ( compResult == 0 ) && ( i >= 0 ); i-- ) {
        /* next lower pair */
        if ( a->words[ i ] > b->words[ i ] ) {
            /* a > b */
            compResult = 1;
        } else if ( a->words[ i ] < b->words[ i ] ) {
            /* a < b */
            compResult = -1;
        } else {
            /* do nothing */
        }
    }

    return compResult;
}

/************************
 * Calculate a := a - b *
 ************************/
static void
EscRsa4096Fe_Subtract(
    EscRsa4096_FieldElementT* a,
    const EscRsa4096_FieldElementT* b )
{
    EscRsa4096_FWORD carry = 1U;
    EscRsa4096_FWORD diff;
    UINT32 i;

    for ( i = 0U; i < EscRsa4096_RSA_SIZE_WORDS; i++) {
        diff = ( (EscRsa4096_FWORD) a->words[ i ] ) + ( (EscRsa4096_FWORD) ( b->words[ i ] ^ EscRsa4096_MAX_VAL ) ) + carry;
        carry = EscRsa4096_HI_FWORD( diff );
        a->words[ i ] = (EscRsa4096_HWORD) EscRsa4096_LO_FWORD( diff );
    }
}
#endif

#ifdef EscRsa4096_ENABLE_GENRSA
/*****************************************
 * shifts field element right by one bit *
 *****************************************/
static void
EscRsa4096Fe_ShiftRight(
    EscRsa4096_FieldElementT* a )
{
    UINT16 i;

    /* shift first words */
    for ( i = 0U; i < ( EscRsa4096_RSA_SIZE_WORDS - 1U ); i++ ) {
        EscRsa4096_HWORD v;
        v = a->words[ i ] >> 1;
        v |= (EscRsa4096_HWORD)( a->words[ i + 1U ] << ( EscRsa4096_RSA_BASE - 1U ) );

        a->words[ i ] = v;
    }

    /* shift last word */
    a->words[ EscRsa4096_RSA_SIZE_WORDS - 1U ] >>= 1;   /* most significant word */
}

/*****************************************
 * shifts field element right by one bit *
 *****************************************/
static void
EscRsa4096Fe_SignedShiftRight(
    EscRsa4096_FieldElementT* a,
    EscRsa4096_HWORD* aSign )
{
    UINT16 i;
    EscRsa4096_HWORD v;

    /* shift first words */
    for ( i = 0U; i < ( EscRsa4096_RSA_SIZE_WORDS - 1U ); i++ ) {
        v = a->words[ i ] >> 1;
        v |= (EscRsa4096_HWORD)( a->words[ i + 1U ] << ( EscRsa4096_RSA_BASE - 1U ) );
        a->words[ i ] = v;
    }

    /* shift last word */
    v = a->words[ EscRsa4096_RSA_SIZE_WORDS - 1U ] >> 1;
    v |= (EscRsa4096_HWORD) ( *aSign << ( EscRsa4096_RSA_BASE - 1U ) );
    a->words[ EscRsa4096_RSA_SIZE_WORDS - 1U ] = v;
    v = *aSign & ( (EscRsa4096_HWORD) ( (EscRsa4096_HWORD) 1U << ( EscRsa4096_RSA_BASE - 1U ) ) );
    *aSign >>= 1;
    *aSign |= v;
}

/**********************************
 * check if field element is zero *
 * note: sign irrelevant          *
 **********************************/
static BOOL
EscRsa4096Fe_IsZero(
    const EscRsa4096_FieldElementT* a )
{
    UINT16 i;
    BOOL isZero = TRUE;

    /* for i from 0 to t do a[i] =? 0 */
    for ( i = 0U; i < EscRsa4096_RSA_SIZE_WORDS; i++ ) {
        if ( a->words[ i ] != 0U ) {
            /* not zero */
            isZero = FALSE;
            break;
        }
    }

    return isZero;
}

/**********************************
 * checks if field element is one *
 **********************************/
static BOOL
EscRsa4096Fe_IsOne(
    const EscRsa4096_FieldElementT* a )
{
    /* declarations */
    BOOL isOne = TRUE;

    /* a[0] =? 1 */
    if ( a->words[ 0 ] != 1U ) {
        isOne = FALSE;
    } else {
        UINT16 i;
        /* for i from 1 to t-1 do a[i] =? 0 */
        for ( i = 1U; i < EscRsa4096_RSA_SIZE_WORDS; i++ ) {
            if ( a->words[ i ] != 0U ) {
                /* not one */
                isOne = FALSE;
            }
        }
    }

    return isOne;
}

/************************
 * Calculate a := a + b *
 ************************/
static void
EscRsa4096Fe_SignedAdd(
    EscRsa4096_FieldElementT* a,
    EscRsa4096_HWORD* aSign,
    const EscRsa4096_FieldElementT* b,
    const EscRsa4096_HWORD bSign )
{
    EscRsa4096_FWORD carry = 0U;
    EscRsa4096_FWORD sum;
    UINT32 i;

    for ( i = 0U; i < EscRsa4096_RSA_SIZE_WORDS; i++) {
        sum = ( (EscRsa4096_FWORD) a->words[ i ] ) + ( (EscRsa4096_FWORD) b->words[ i ] ) + carry;
        carry = EscRsa4096_HI_FWORD( sum );
        a->words[ i ] = (EscRsa4096_HWORD) EscRsa4096_LO_FWORD( sum );
    }
    sum = ( (EscRsa4096_FWORD) *aSign ) + ( (EscRsa4096_FWORD) ( bSign ) ) + carry;
    *aSign = (EscRsa4096_HWORD) EscRsa4096_LO_FWORD( sum );
}

/************************
 * Calculate a := a - b *
 ************************/
static void
EscRsa4096Fe_SignedSubtract(
    EscRsa4096_FieldElementT* a,
    EscRsa4096_HWORD* aSign,
    const EscRsa4096_FieldElementT* b,
    const EscRsa4096_HWORD bSign )
{
    EscRsa4096_FWORD carry = 1U;
    EscRsa4096_FWORD diff;
    UINT32 i;

    for ( i = 0U; i < EscRsa4096_RSA_SIZE_WORDS; i++) {
        diff = ( (EscRsa4096_FWORD) a->words[ i ] ) + ( (EscRsa4096_FWORD) (  b->words[ i ] ^ EscRsa4096_MAX_VAL ) ) + carry;
        carry = EscRsa4096_HI_FWORD( diff );
        a->words[ i ] = (EscRsa4096_HWORD) EscRsa4096_LO_FWORD( diff );
    }
    diff = ( (EscRsa4096_FWORD) *aSign ) + ( (EscRsa4096_FWORD) ( bSign ^ EscRsa4096_MAX_VAL ) ) + carry;
    *aSign = (EscRsa4096_HWORD) EscRsa4096_LO_FWORD( diff );
}

/*******************************************
 * Add a 16/32 bit word to a field element *
 *******************************************/
static void
EscRsa4096Fe_AddWord(
    EscRsa4096_FieldElementT* c,
    const EscRsa4096_HWORD a )
{
    EscRsa4096_HWORD* cwords;
    UINT32 i;

    cwords = c->words;

    /*lint -save -esym(960,17.4) cwords is an array*/
    cwords[ 0U ] += a;
    if ( cwords[ 0U ] < a ) {
        for ( i = 1U; i < EscRsa4096_RSA_SIZE_WORDS; i++) {
            cwords[ i ] += 1U;
            if ( cwords[ i ] >= 1U ) {
                break;
            }
        }
    }
    /*lint -restore cwords is an array*/
}

/******************************************************
 * Compute modulo of field element and a small number *
 ******************************************************/
static EscRsa4096_HWORD
EscRsa4096Fe_ModWord(
    const EscRsa4096_FieldElementT* a,
    EscRsa4096_HWORD w )
{
    EscRsa4096_FWORD ret = 0U;
    SINT32 i;

    for ( i = (SINT32) EscRsa4096_RSA_SIZE_WORDS - 1; i >= 0; i--) {
        ret = ( ( ret << EscRsa4096_RSA_BASE ) | (EscRsa4096_FWORD) a->words[ i ] ) % (EscRsa4096_FWORD) w;
    }

    return (EscRsa4096_HWORD) ret;
}

/******************************************************
 * Test if a signed field element is positive (> 0)   *
 ******************************************************/
static BOOL
EscRsa4096Fe_IsPositive(
    EscRsa4096_HWORD sign,
    const EscRsa4096_FieldElementT* fe)
{
    BOOL ret = FALSE;

    /* If MSB not set ... */
    if ( (sign & ( (EscRsa4096_HWORD) ( (EscRsa4096_HWORD) 1U << ( EscRsa4096_RSA_BASE - 1U ) ) )) == 0U) {
        /* .. and the entire value is non-zero ... */
        if ( (sign > 0) || (EscRsa4096Fe_IsZero(fe) == FALSE) ) {
            /* ... then it's positive */
            ret = TRUE;
        }
    }

    return ret;
}

/*********************************************
 * Modular inverse using binary extended gcd *
 *********************************************/
static BOOL
EscRsa4096Fe_ModInv(
    const EscRsa4096_FieldElementT* x,
    const EscRsa4096_FieldElementT* y,
    EscRsa4096_FieldElementT* result )
{
    EscRsa4096_FieldElementT u;
    EscRsa4096_FieldElementT v;
    EscRsa4096_FieldElementT a;
    EscRsa4096_FieldElementT b;
    EscRsa4096_FieldElementT c;
    EscRsa4096_FieldElementT d;
    UINT32 i;
    EscRsa4096_HWORD aSign;
    EscRsa4096_HWORD bSign;
    EscRsa4096_HWORD cSign;
    EscRsa4096_HWORD dSign;
    BOOL hasFailed = FALSE;

    for ( i = 0U; i < EscRsa4096_RSA_SIZE_WORDS; i++ ) {
        u.words[ i ] = x->words[ i ];
        v.words[ i ] = y->words[ i ];
        a.words[ i ] = 0U;
        b.words[ i ] = 0U;
        c.words[ i ] = 0U;
        d.words[ i ] = 0U;
    }
    a.words[ 0U ] = 1U;
    d.words[ 0U ] = 1U;
    aSign = 0U;
    bSign = 0U;
    cSign = 0U;
    dSign = 0U;

    do {
        while ( ( u.words[ 0U ] & 1U ) == 0U ) {
            EscRsa4096Fe_ShiftRight( &u );
            if ( ( ( a.words[ 0U ] & 1U ) == 1U ) || ( ( b.words[ 0U ] & 1U ) == 1U ) ) {
                EscRsa4096Fe_SignedAdd( &a, &aSign, y, 0U );
                EscRsa4096Fe_SignedSubtract( &b, &bSign, x, 0U );
            }
            EscRsa4096Fe_SignedShiftRight( &a, &aSign );
            EscRsa4096Fe_SignedShiftRight( &b, &bSign );
        }
        while ( ( v.words[ 0U ] & 1U ) == 0U ) {
            EscRsa4096Fe_ShiftRight( &v );
            if ( ( ( c.words[ 0U ] & 1U ) == 1U ) || ( ( d.words[ 0U ] & 1U ) == 1U ) ) {
                EscRsa4096Fe_SignedAdd( &c, &cSign, y, 0U );
                EscRsa4096Fe_SignedSubtract( &d, &dSign, x, 0U );
            }
            EscRsa4096Fe_SignedShiftRight( &c, &cSign );
            EscRsa4096Fe_SignedShiftRight( &d, &dSign );
        }
        if ( EscRsa4096Fe_AbsoluteCompare( &u, &v ) > -1 ) {
            EscRsa4096Fe_Subtract( &u, &v );
            EscRsa4096Fe_SignedSubtract( &a, &aSign, &c, cSign );
            EscRsa4096Fe_SignedSubtract( &b, &bSign, &d, dSign );
            /* "If B>0 then A <-- A+y and B <-- B-x." */
            if ( EscRsa4096Fe_IsPositive(bSign, &b) != FALSE ) {
                EscRsa4096Fe_SignedAdd( &a, &aSign, y, 0U );
                EscRsa4096Fe_SignedSubtract( &b, &bSign, x, 0U );
            }
        } else {
            EscRsa4096Fe_Subtract( &v, &u );
            EscRsa4096Fe_SignedSubtract( &c, &cSign, &a, aSign );
            EscRsa4096Fe_SignedSubtract( &d, &dSign, &b, bSign );
            /* "If D>0 then C <-- C+y and D <-- D-x." */
            if ( EscRsa4096Fe_IsPositive(dSign, &d) != FALSE ) {
                EscRsa4096Fe_SignedAdd( &c, &cSign, y, 0U );
                EscRsa4096Fe_SignedSubtract( &d, &dSign, x, 0U );
            }
        }
    } while ( EscRsa4096Fe_IsZero( &u ) == FALSE );

    if ( EscRsa4096Fe_IsOne( &v ) == FALSE ) {
        hasFailed = TRUE;
    }

    if ( dSign ) {
        EscRsa4096Fe_SignedAdd( &d, &dSign, x, 0U );
    }

    EscRsa4096Fe_Assign( result, &d );

    return hasFailed;
}

/*******************************
 * Miller-Rabin primality test *
 *******************************/
static BOOL
EscRsa4096_MillerRabin(
    EscRandom_ContextT* randCtx,
    const EscRsa4096_FieldElementT* n,
    const UINT32 iter,
    BOOL* prime )
{
    EscRsa4096_FieldElementT n1;
    EscRsa4096_FieldElementT r;
    EscRsa4096_FieldElementT y;
    EscRsa4096_FieldElementLongT multiplication_result;
    EscRsa4096_MultiplyDataT mult;
    UINT8 a[ EscRsa4096_KEY_BYTES ];
    UINT32 s;
    UINT32 i;
    UINT32 j;
    BOOL hasFailed = FALSE;

    /* n1 = n - 1, requires that n is odd */
    EscRsa4096Fe_Assign( &n1, n );
    n1.words[ 0 ] &= (EscRsa4096_HWORD) ( ~(EscRsa4096_HWORD)1U );

    /* r = n */
    EscRsa4096Fe_Assign( &r, n );

    /* Decompose candidate n into r and s with n - 1 = 2^s * r */
    s = 0U;
    do {
        EscRsa4096Fe_ShiftRight( &r );
        s++;
    } while ( ( r.words[ 0 ] & 1U ) == 0U );

    for ( i = 0U; i < ( EscRsa4096_KEY_BYTES / 2U ); i++ ) {
        a[ i ] = 0U;
    }

    *prime = TRUE;
    i = 0U;
    while ( ( *prime == TRUE ) && ( i < iter ) ) {
        do {
            /* Get random witness a with 2 <= a <= n - 2 */
            hasFailed |= EscRandom_GetRandom( randCtx, &a[ EscRsa4096_KEY_BYTES / 2U ], EscRsa4096_KEY_BYTES / 2U );
            a[ EscRsa4096_KEY_BYTES / 2U ] &= 0x7fU;
            EscRsa4096Fe_FromBytesBE( &y, a, EscRsa4096_KEY_BYTES );
        } while ( ( EscRsa4096Fe_IsZero( &y ) == TRUE )
              || ( EscRsa4096Fe_IsOne( &y ) == TRUE )
              || ( EscRsa4096Fe_AbsoluteCompare( &y, &n1 ) != -1 ) );

        /* y = a^r mod n */
        mult.c = &multiplication_result;
        mult.m = n;
        mult.x = &y;
        EscRsa4096Fe_BigPow( &mult, &r );

        if ( ( EscRsa4096Fe_IsOne( &y ) == FALSE ) && ( EscRsa4096Fe_AbsoluteCompare( &y, &n1 ) != 0 ) ) {
            j = 1U;
            while ( ( j < s ) && ( EscRsa4096Fe_AbsoluteCompare( &y, &n1 ) != 0 ) ) {
                /* y = y^2 mod n */
                mult.c = &multiplication_result;
                mult.m = n;
                mult.x = &y;
                EscRsa4096Fe_Square( &mult );

                if ( EscRsa4096Fe_IsOne( &y ) == TRUE ) {
                    *prime = FALSE;
                    break;
                }
                j++;
            }
            if ( ( *prime == TRUE ) && ( EscRsa4096Fe_AbsoluteCompare( &y, &n1 ) != 0 ) ) {
                *prime = FALSE;
            }
        }
        i++;
    }

    return hasFailed;
}

/********************************
 * Generate random prime number *
 ********************************/
#ifdef CONFIG_KEYGEN_PROFILING
 #include <time.h>
 #include <stdio.h>
 inline long timevaldiff(struct timeval *starttime, struct timeval *finishtime)
{
  long msec;
  msec=(finishtime->tv_sec-starttime->tv_sec)*1000;
  msec+=(finishtime->tv_usec-starttime->tv_usec)/1000;
  return msec;
}
#endif /* CONFIG_KEYGEN_PROFILING */

static BOOL
EscRsa4096_GeneratePrime(
    EscRandom_ContextT* randCtx,
    EscRsa4096_FieldElementT* n )
{
    static const UINT8 primes[ EscRsa4096_NUM_PRIMES ] =
    {
          3U,   5U,   7U,  11U,  13U,  17U,  19U,  23U,  29U,  31U,  37U,  41U,  43U,  47U,  53U,  59U,
         61U,  67U,  71U,  73U,  79U,  83U,  89U,  97U, 101U, 103U, 107U, 109U, 113U, 127U, 131U, 137U,
        139U, 149U, 151U, 157U, 163U, 167U, 173U, 179U, 181U, 191U, 193U, 197U, 199U, 211U, 223U, 227U,
        229U, 233U, 239U, 241U, 251U
    };
    UINT8 candidate[ EscRsa4096_KEY_BYTES ];
    UINT8 mods[ EscRsa4096_NUM_PRIMES ];
    EscRsa4096_HWORD delta;
    UINT32 i;
    BOOL prime = FALSE;
    BOOL hasFailed;
#ifdef CONFIG_KEYGEN_PROFILING
    struct timeval funcS, funcE;
    UINT32 func = 0;

    gettimeofday(&funcS, NULL);
#endif /* CONFIG_KEYGEN_PROFILING*/

    /* Get random number */
    for ( i = 0U; i < ( EscRsa4096_KEY_BYTES / 2U ); i++ ) {
        candidate[ i ] = 0U;
    }
    hasFailed = EscRandom_GetRandom( randCtx, &candidate[ EscRsa4096_KEY_BYTES / 2U ], EscRsa4096_KEY_BYTES / 2U );

    /* Set two most significant bits */
    candidate[ EscRsa4096_KEY_BYTES / 2U ] |= 0xc0U;

    /* Set least significant bit */
    candidate[ EscRsa4096_KEY_BYTES - 1U ] |= 1U;

    /* Convert to field element */
    EscRsa4096Fe_FromBytesBE( n, candidate, EscRsa4096_KEY_BYTES );

#ifdef CONFIG_KEYGEN_PROFILING
    int abc = 0;
    int def = 0;
    int millRabs = 0;
    struct timeval intWhileS, intWhileE;
    struct timeval calcModS, calcModE;
    struct timeval addWordS, addWordE;
    struct timeval millS, millE;
    UINT32 intWhile = 0, calcMod = 0, addWord = 0, mill = 0;
#endif /* CONFIG_KEYGEN_PROFILING */
    while ( ( prime == FALSE ) && ( hasFailed == FALSE ) ) {
#ifdef CONFIG_KEYGEN_PROFILING
        abc++;
#endif /* CONFIG_KEYGEN_PROFILING */
        /* Calculate modulo */
#ifdef CONFIG_KEYGEN_PROFILING
        gettimeofday(&calcModS, NULL);
#endif /* CONFIG_KEYGEN_PROFILING */
        for ( i = 0U; i < EscRsa4096_NUM_PRIMES; i++ ) {
            mods[ i ] = (UINT8) EscRsa4096Fe_ModWord( n, (EscRsa4096_HWORD) primes[ i ] );
        }
#ifdef CONFIG_KEYGEN_PROFILING
        gettimeofday(&calcModE, NULL);
        calcMod += timevaldiff(&calcModS, &calcModE);
        gettimeofday(&intWhileS, NULL);
#endif /* CONFIG_KEYGEN_PROFILING */
        delta = 0U;
        while ( ( prime == FALSE ) && ( delta < ( EscRsa4096_MAX_VAL - primes[ EscRsa4096_NUM_PRIMES - 1U ] ) ) ) {
#ifdef CONFIG_KEYGEN_PROFILING
        def++;
#endif /* CONFIG_KEYGEN_PROFILING */
            EscWatchdog_Call( EscWatchdog_STATIC_WATCHDOG_TRIGGER );
            prime = TRUE;
            for ( i = 0U; i < EscRsa4096_NUM_PRIMES; i++ ) {
                if ( ( ( (EscRsa4096_HWORD) mods[ i ] + delta ) % (EscRsa4096_HWORD) primes[ i ] ) <= 1U ) {
                    prime = FALSE;
                    delta += 2U;
                    break;
                }
            }
        }
#ifdef CONFIG_KEYGEN_PROFILING
        gettimeofday(&intWhileE, NULL);
        intWhile += timevaldiff(&intWhileS, &intWhileE);
#endif /* CONFIG_KEYGEN_PROFILING */

        /* Add delta to candidate */
#ifdef CONFIG_KEYGEN_PROFILING
        gettimeofday(&addWordS, NULL);
#endif /* CONFIG_KEYGEN_PROFILING */
        EscRsa4096Fe_AddWord( n, delta );
#ifdef CONFIG_KEYGEN_PROFILING
        gettimeofday(&addWordE, NULL);
        addWord += timevaldiff(&addWordS, &addWordE);
#endif /* CONFIG_KEYGEN_PROFILING */

        if ( prime == TRUE ) {
            /* Miller-Rabin primality test */
#ifdef CONFIG_KEYGEN_PROFILING
            millRabs++;
            gettimeofday(&millS, NULL);
#endif /* CONFIG_KEYGEN_PROFILING */
            hasFailed |= EscRsa4096_MillerRabin( randCtx, n, EscRsa4096_PRIMETEST_ITER, &prime );
#ifdef CONFIG_KEYGEN_PROFILING
            gettimeofday(&millE, NULL);
            mill += timevaldiff(&millS, &millE);
#endif /* CONFIG_KEYGEN_PROFILING */

            if ( prime == FALSE ) {
                EscRsa4096Fe_AddWord( n, 2U );
            }
        }
    }
#ifdef CONFIG_KEYGEN_PROFILING
        gettimeofday(&funcE, NULL);
        func += timevaldiff(&funcS, &funcE);

    printf("outer while loop called %d times\n",abc);
    printf("inner while loop called %d times\n",def);
    printf("%s %ldms\n",__func__, func);
    printf("tot. calculate modulus %ldms\n", calcMod);
    printf("avg. calculate modulus %ldms\n", calcMod / abc);
    printf("tot. internal while loop %ldms\n", intWhile );
    printf("avg. internal while loop %ldms\n", intWhile / abc);
    printf("tot. addWord %ldms\n", addWord );
    printf("avg. addWord %ldms\n", addWord / abc);
    printf("tot. miller rabin %ldms\n", mill );
    printf("avg. miller rabin %ldms\n", mill / millRabs);
#endif /* CONFIG_KEYGEN_PROFILING */

    return hasFailed;
}

/*************************
 * Generate RSA key pair *
 *************************/
BOOL
EscRsa4096_GenerateKeyPair(
    EscRandom_ContextT* randCtx,
    EscRsa4096_KeyPairT* keyPair,
    UINT32 pubExp )
{
    EscRsa4096_FieldElementLongT multiplication_result;
    EscRsa4096_MultiplyDataT mult;
    EscRsa4096_FWORD dist;
    UINT32 i;
    BOOL hasFailed = FALSE;

#ifdef CONFIG_KEYGEN_PROFILING
    struct timeval funcS, funcE;
    UINT32 func = 0;

    gettimeofday(&funcS, NULL);
#endif /* CONFIG_KEYGEN_PROFILING */

    if ( ( randCtx != 0 ) && ( pubExp != 0U ) && ( keyPair != 0 ) ) {

        /* Set public exponent e */
        for ( i = 0U; i < EscRsa4096_RSA_SIZE_WORDS; i++ ) {
            keyPair->e.words[ i ] = 0U;
        }
#    if EscRsa4096_WORD_SIZE == 4U
        keyPair->e.words[ 0U ] = pubExp;
#    else
        keyPair->e.words[ 0U ] = (EscRsa4096_HWORD)( pubExp >> EscRsa4096_RSA_BASE );
        keyPair->e.words[ 1U ] = (EscRsa4096_HWORD)( pubExp & EscRsa4096_MAX_VAL );
#    endif

        /* Generate random prime number for p */
#ifdef CONFIG_KEYGEN_PROFILING
        printf("%d start generate prime for p\n", time(NULL));
#endif /* CONFIG_KEYGEN_PROFILING */
        hasFailed = EscRsa4096_GeneratePrime( randCtx, &keyPair->p );
#ifdef CONFIG_KEYGEN_PROFILING
        printf("%d done generate prime for p\n\n", time(NULL));
#endif /* CONFIG_KEYGEN_PROFILING */

        /* Check if p - 1 is coprime to e */
        EscRsa4096Fe_Assign( &keyPair->iqmp, &keyPair->p );
        keyPair->iqmp.words[ 0U ] &= (EscRsa4096_HWORD)( ~(EscRsa4096_HWORD)1U );
        hasFailed |= EscRsa4096Fe_ModInv( &keyPair->e, &keyPair->iqmp, &keyPair->iqmp );

        /* Generate random prime number for q */
#ifdef CONFIG_KEYGEN_PROFILING
        printf("%d start generate prime for q\n", time(NULL));
        int genPrimeForQ = 0;
#endif /* CONFIG_KEYGEN_PROFILING */
        do {
#ifdef CONFIG_KEYGEN_PROFILING
            genPrimeForQ++;
#endif /* CONFIG_KEYGEN_PROFILING */
            hasFailed |= EscRsa4096_GeneratePrime( randCtx, &keyPair->q );

            /* Check if q - 1 is coprime to e */
            EscRsa4096Fe_Assign( &keyPair->iqmp, &keyPair->q );
            keyPair->iqmp.words[ 0U ] &= (EscRsa4096_HWORD)( ~(EscRsa4096_HWORD)1U );
            hasFailed |= EscRsa4096Fe_ModInv( &keyPair->e, &keyPair->iqmp, &keyPair->iqmp );

            /* Ensure p > q */
            if ( EscRsa4096Fe_AbsoluteCompare( &keyPair->p, &keyPair->q ) <= 0 ) {
                /* Swap p and q (using xor swap to save memory) */
                for ( i = 0U; i < EscRsa4096_RSA_SIZE_WORDS; i++ ) {
                    keyPair->p.words[ i ] ^= keyPair->q.words[ i ];
                    keyPair->q.words[ i ] ^= keyPair->p.words[ i ];
                    keyPair->p.words[ i ] ^= keyPair->q.words[ i ];
                }
            }

            /* Check distance between p and q */
            dist = ( (EscRsa4096_FWORD) keyPair->p.words[ ( EscRsa4096_RSA_SIZE_WORDS / 2U ) - 1U ] ) << ( EscRsa4096_RSA_BASE - 1U );
            dist /= ( (EscRsa4096_FWORD) keyPair->q.words[ ( EscRsa4096_RSA_SIZE_WORDS / 2U ) - 1U ] );
        } while ( dist <= EscRsa4096_PQ_LOWER_BOUND );
#ifdef CONFIG_KEYGEN_PROFILING
        printf("%d done generate prime for q, loop called %d times\n\n",
            time(NULL), genPrimeForQ);
#endif /* CONFIG_KEYGEN_PROFILING */

        /* Calculate n = p * q */
        mult.c = &multiplication_result;
        mult.x = &keyPair->p;
        EscRsa4096Fe_ToLongFe( mult.c, &keyPair->q );
        EscRsa4096Fe_MultiplyClassically( &mult );
        EscRsa4096Fe_FromLongFe( &keyPair->n, mult.c );

        /* Calculate d = e^-1 mod phi(n) */
        /* phi(n) = (p - 1) * (q - 1) = p * q - p - q + 1 = n - p - q + 1 */
        EscRsa4096Fe_Assign( &keyPair->iqmp, &keyPair->n );
        EscRsa4096Fe_Subtract( &keyPair->iqmp, &keyPair->p );
        EscRsa4096Fe_Subtract( &keyPair->iqmp, &keyPair->q );
        EscRsa4096Fe_AddWord( &keyPair->iqmp, 1U );
        hasFailed |= EscRsa4096Fe_ModInv( &keyPair->iqmp, &keyPair->e, &keyPair->d );

        /* Calculate d mod (p - 1) */
        EscRsa4096Fe_Assign( &keyPair->dmp1, &keyPair->p );
        keyPair->dmp1.words[ 0U ] &= (EscRsa4096_HWORD)( ~(EscRsa4096_HWORD)1U );
        mult.c = &multiplication_result;
        mult.m = &keyPair->dmp1;
        EscRsa4096Fe_ToLongFe( mult.c, &keyPair->d );
        EscRsa4096Fe_ModularReduction( &mult );
        EscRsa4096Fe_FromLongFe( &keyPair->dmp1, mult.c );

        /* Calculate d mod (q - 1) */
        EscRsa4096Fe_Assign( &keyPair->dmq1, &keyPair->q );
        keyPair->dmq1.words[ 0U ] &= (EscRsa4096_HWORD)( ~(EscRsa4096_HWORD)1U );
        mult.c = &multiplication_result;
        mult.m = &keyPair->dmq1;
        EscRsa4096Fe_ToLongFe( mult.c, &keyPair->d );
        EscRsa4096Fe_ModularReduction( &mult );
        EscRsa4096Fe_FromLongFe( &keyPair->dmq1, mult.c );

        /* Calculate q^-1 mod p */
        for ( i = 0U; i < EscRsa4096_RSA_SIZE_WORDS; i++ ) {
            keyPair->iqmp.words[ i ] = 0U;
        }
        hasFailed |= EscRsa4096Fe_ModInv( &keyPair->p, &keyPair->q, &keyPair->iqmp );
    } else {
        hasFailed = TRUE;
    }

#ifdef CONFIG_KEYGEN_PROFILING
    gettimeofday(&funcE, NULL);
    func += timevaldiff(&funcS, &funcE);
    printf("%s %ldms\n",__func__, func);
#endif /* CONFIG_KEYGEN_PROFILING */

    return hasFailed;
}
#endif /* EscRsa4096_ENABLE_GENRSA */

#ifdef EscRsa4096_ENABLE_CRT
/************************
 * Calculate a := a + b *
 ************************/
static void
EscRsa4096Fe_Add(
    EscRsa4096_FieldElementT* a,
    const EscRsa4096_FieldElementT* b )
{
    EscRsa4096_FWORD carry = 0U;
    EscRsa4096_FWORD sum;
    UINT32 i;

    for ( i = 0U; i < EscRsa4096_RSA_SIZE_WORDS; i++) {
        sum = ( (EscRsa4096_FWORD) a->words[ i ] ) + ( (EscRsa4096_FWORD) b->words[ i ] ) + carry;
        carry = EscRsa4096_HI_FWORD( sum );
        a->words[ i ] = (EscRsa4096_HWORD) EscRsa4096_LO_FWORD( sum );
    }
}

#    ifndef EscRsa4096_ENABLE_STACK_SAVING_INTERFACE
/***************************************************************************
 * Modular exponentiation for small exponents. c = m^e mod n ( with CRT )   *
 ***************************************************************************/
BOOL
EscRsa4096_ModExpCrt(
    UINT8 result[],
    const UINT8 message[],
    const EscRsa4096_KeyPairT* keyPair )
#    else /* EscRsa4096_ENABLE_STACK_SAVING_INTERFACE */
BOOL
EscRsa4096_ModExpCrtSaveStack(
    EscRsa4096_FieldElementT* result,
    const EscRsa4096_FieldElementT* message,
    const EscRsa4096_KeyPairT* keyPair )
#    endif /* EscRsa4096_ENABLE_STACK_SAVING_INTERFACE */
{
    EscRsa4096_FieldElementLongT multiplication_result;
    EscRsa4096_MultiplyDataT mult;
    EscRsa4096_FieldElementT mp;
    EscRsa4096_FieldElementT mq;
    EscRsa4096_FieldElementT temp;
    EscRsa4096_FieldElementT s;
    BOOL hasFailed = FALSE;

    if ( ( message != 0 ) && ( result != 0 ) && ( keyPair != 0 ) ) {

        /* mp = m^dmp1 mod p */
        mult.c = &multiplication_result;
        mult.x = &temp;
        mult.m = &keyPair->p;
#ifdef EscRsa4096_ENABLE_STACK_SAVING_INTERFACE
        EscRsa4096Fe_Assign( mult.x, message );
#else
        EscRsa4096Fe_FromBytesBE( mult.x, message, EscRsa4096_KEY_BYTES );
#endif
        EscRsa4096Fe_BigPow( &mult, &keyPair->dmp1 );
        EscRsa4096Fe_Assign( &mp, mult.x );

        /* mq = m^dmq1 mod q */
        mult.m = &keyPair->q;
#ifdef EscRsa4096_ENABLE_STACK_SAVING_INTERFACE
        EscRsa4096Fe_Assign( mult.x, message );
#else
        EscRsa4096Fe_FromBytesBE( mult.x, message, EscRsa4096_KEY_BYTES );
#endif
        EscRsa4096Fe_BigPow( &mult, &keyPair->dmq1 );
        EscRsa4096Fe_Assign( &mq, mult.x );

        /* s = (iqmp * (mp - mq) mod p) * q + mq */
        EscRsa4096Fe_Assign( &s, &mp );
        if ( EscRsa4096Fe_AbsoluteCompare( &mp, &mq ) == -1 ) {
            /* if mp < mq, add p to mp to ensure that (mp - mq) is positive */
            EscRsa4096Fe_Add( &s, &keyPair->p );
        }
        EscRsa4096Fe_Subtract( &s, &mq );

        mult.m = &keyPair->p;
        EscRsa4096Fe_Assign( mult.x, &s );
        EscRsa4096Fe_ToLongFe( mult.c, &keyPair->iqmp );
        EscRsa4096Fe_Multiply( &mult );

        EscRsa4096Fe_Assign( mult.x, &keyPair->q );
        EscRsa4096Fe_MultiplyClassically( &mult );
        EscRsa4096Fe_FromLongFe( &s, mult.c );

        EscRsa4096Fe_Add( &s, &mq );

#ifdef EscRsa4096_ENABLE_STACK_SAVING_INTERFACE
        EscRsa4096Fe_Assign( result, &s );
#else
        EscRsa4096Fe_ToBytesBE( result, &s );
#endif
    } else {
        hasFailed = TRUE;
    }

    return hasFailed;
}
#endif /* EscRsa4096_ENABLE_CRT */

/***************************************************************************
 * 6. END                                                                  *
 ***************************************************************************/
