/***************************************************************************
 * Copyright                                                               *
 *                                                                         *
 *     ESCRYPT GmbH - Embedded Security       ESCRYPT Inc.                 *
 *     Zentrum fuer IT-Sicherheit             315 E Eisenhower Parkway     *
 *     Lise-Meitner-Allee 4                   Suite 214                    *
 *     44801 Bochum                           Ann Arbor, MI 48108          *
 *     Germany                                USA                          *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 ***************************************************************************/

/***************************************************************************/
/*!
 \file        random.c

 \brief       Random number generator using /dev/urandom
 \attention   Good random numbers are essential for key generation!

 $Rev: 626 $
 */
/***************************************************************************
 $Author: mlange $
 $Date: 2017-01-27 14:07:51 +0100 (Fr, 27. Jan 2017) $

 History:
 11-Feb-2013 DV ------------------------ Creation ---------------------------

 $ENDOFHISTORY$
 ****************************************************************************/

/***************************************************************************
 * 1. INCLUDES                                                             *
 ***************************************************************************/

#include "../../ecm/inc/ecm.h"
#include <esc_common.h>
#include <crypto.h>
#include <random.h>
#include <stdio.h>
#include <string.h>
#include <esc_debug.h>

/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/***************************************************************************
 * 3. DECLARATIONS                                                         *
 ***************************************************************************/

/** this function reads entropy from /dev/random */
static BOOL EscRandom_GetEntropy( UINT8 entropy[ EscHashDrbg_SEED_LEN ] );

/***************************************************************************
 * 4. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/

static BOOL EscRandom_GetEntropy( UINT8 entropy[ EscHashDrbg_SEED_LEN ] ){
    BOOL hasFailed = TRUE;
    FILE *fp = fopen( RANDOM_SOURCE, "r" );
    if ( fp != NULL ) {
        if ( EscHashDrbg_ENTROPY_LEN == fread( entropy, 1, EscHashDrbg_ENTROPY_LEN, fp ) ) {
            hasFailed = FALSE;
        }
        fclose( fp );
	}
    return hasFailed;
}

/** init random */
BOOL EscRandom_Init( EscRandom_ContextT* ctx, const UINT8 rnd[ ] ) {
    BOOL hasFailed = TRUE;
    UINT8 entropy[ EscHashDrbg_SEED_LEN ];

    hasFailed = EscRandom_GetEntropy( entropy );
    /* init Hash DRBG using entropy and nonce */
    hasFailed |= EscHashDrbg_Init( ctx, entropy, rnd );
    return hasFailed;
}

/** Returns an array of random data */
BOOL EscRandom_GetRandom( EscRandom_ContextT* ctx, UINT8 rnd[ ], UINT32 len ) {

    BOOL hasFailed = EscHashDrbg_GetRandom( ctx, rnd, len );
    if ( hasFailed == TRUE ) {
        /* reseed on error */
        UINT8 entropy[ EscHashDrbg_SEED_LEN ];
        hasFailed = EscRandom_GetEntropy( entropy );
        hasFailed |= EscHashDrbg_Reseed( ctx, entropy );
        hasFailed |= EscHashDrbg_GetRandom( ctx, rnd, len );
    }
    return hasFailed;
}

/***************************************************************************
 * 6. END                                                                  *
 ***************************************************************************/
