/***************************************************************************
 * Copyright                                                               *
 *                                                                         *
 *     ESCRYPT GmbH - Embedded Security       ESCRYPT Inc.                 *
 *     Zentrum fuer IT-Sicherheit             315 E Eisenhower Parkway     *
 *     Lise-Meitner-Allee 4                   Suite 214                    *
 *     44801 Bochum                           Ann Arbor, MI 48108          *
 *     Germany                                USA                          *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        crypto.h

   \brief       Crypto wrapper

   \attention   Good random numbers are essential for key generation!

   $Rev: 16 $
 */
/***************************************************************************
$Author: dvandersee $
$Date: 2013-02-28 12:23:09 +0100 (Do, 28. Feb 2013) $

History:
11-Feb-2013 DV ------------------------ Creation ---------------------------

$ENDOFHISTORY$
****************************************************************************/
#ifndef	_CRYPTO_H_
#define	_CRYPTO_H_

// TODO CM Replace random number generation!

/***************************************************************************
 * 1. INCLUDES                                                             *
 ***************************************************************************/
#include "../../cycurlib/lib/inc/rsa.h"
#include "../../cycurlib/lib/inc/rsa_4096.h"
#include "../../cycurlib/lib/inc/rsa_1024.h"
#include "../../cycurlib/lib/inc/random.h"
#include "../../common/inc/esc_common.h"
#include "../../ecm/inc/ecm_cm_sdc_utils.h"


/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/
/** Random source device */
#ifdef UNIT_TESTING
#define RANDOM_SOURCE "/dev/hwrng"
#else /* UNIT_TESTING */
#define RANDOM_SOURCE "/dev/hwrng"
#endif /* UNIT_TESTING */

/***************************************************************************
 * 3. DECLARATIONS                                                         *
 ***************************************************************************/
/** key is managed by libCrypto */
typedef EscRsa_KeyPairT EscCrypto_RsaKeyPairT;
/** key is managed by libCrypto */
typedef EscRsa4096_KeyPairT EscCrypto4096_RsaKeyPairT;
/** key is managed by libCrypto */
typedef EscRsa1024_KeyPairT EscCrypto1024_RsaKeyPairT;

/***************************************************************************
 * 4. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/

/**
 * This function initializes the random context.
 *
 * \param[out]  randCtx     Updated random context
 *
 * \return                  TRUE on error; FALSE else
 **/
extern BOOL
EscCrypto_InitRandom(
    EscRandom_ContextT* randCtx );



/**
 * This function generates a new RSA key pair.
 *
 * \attention Good random numbers are essential for key generation!
 *
 * \param[in]   randCtx          Random context for key pair generation
 * \param[out]  randCtx          Updated random context
 * \param[out]  keyPair          RSA key pair
 * \param[in]   pubKey           RSA short public exponent (e.g., 65537)
 * \param[out]  keyBlob          Pointer where key blob will be stored
 * \param[out]  keyBlobSize      Pointer where key blob size will be stored
 * \param[in]   keySize          Size of the RSA Key (1024, 2048, 4096)
 *
 * \return                  TRUE on error; FALSE else
 **/
BOOL EscCrypto_GenerateKeyPair (
    EscRandom_ContextT* randCtx,
    void* const keyPair,
    const UINT32 pubKey,
    UINT8 **keyBlob,
    UINT32 *keyBlobSize,
    UINT32 keySize );

/**
 * This function serializes a public key.
 *
 * \param[in]   keyPair          RSA-2048 key pair
 * \param[out]  pubKeyData       Pointer to the buffer to store the serialized public key data
 * \param[in]   pubKeyDataSize   Maximum size of the serialized public key data
 * \param[out]  pubKeyDataSize   Actual size of the serialized public key data
 * \param[in]   keySize          Size of RSA Key Pair
 *
 * \return                       TRUE on error; FALSE else
 **/
extern BOOL
EscCrypto_EncodePublicKey(
    const void* const keyPair,
    UINT8* const pubKeyData,
    UINT32* const pubKeyDataSize,
	UINT32 keySize );


/**
 * This function serializes a private key.
 *
 * \param[in]   keyPair         RSA key pair
 * \param[out]  privKeyData     Pointer to the buffer to store the serialized private key data
 * \param[in]   privKeyDataSize Maximum size of the serialized private key data
 * \param[out]  privKeyDataSize Actual size of the serialized private key data
 * \param[in]   keySize         Size of RSA Key Pair
 *
 * \return                      TRUE on error; FALSE else
 **/
extern BOOL
EscCrypto_EncodePrivateKey(
    const void* const keyPair,
    UINT8* const privKeyData,
    UINT32* const privKeyDataSize,
	UINT32 keySize);


/**
 * This function deserializes a BER-encoded RSA public key.
 *
 * \param[out]  keyPair         RSA-2048 key pair
 * \param[in]   pubKeyData      Pointer to the serialized public key data
 * \param[in]   pubKeyDataSize  Size of the serialized public key data
 * \param[in]   keySize         Size of RSA Key Pair
 *
 * \return                      TRUE on error; FALSE else
 **/
BOOL
EscCrypto_DecodePublicKey(
    void* const keyPair,
    UINT8* const pubKeyData,
    const UINT32 pubKeyDataSize,
	UINT32 keySize);


/**
 * This function deserializes a BER-encoded private key.
 *
 * \param[out]  keyPair         RSA key pair
 * \param[in]   privKeyData     Pointer to the serialized private key data
 * \param[in]   privKeyDataSize Size of the serialized private key data
 * \param[in]   keySize         Size of RSA Key Pair
 *
 * \return                      TRUE on error; FALSE else
 **/
extern BOOL
EscCrypto_DecodePrivateKey(
    void* const keyPair,
    const UINT8* const privKeyData,
    const UINT32 privKeyDataSize,
	UINT32 keySize);


/**
 * This function loads and decodes a private key from a file into a given
 * EscRsa_KeyPair
 *
 * \param[in]       sdc         Pointer to SDC parameter structure
 * \param[in]       file        file path and name
 * \param[out]      keyPair     Pointer to KeyPair to be filled
 * \param[in]       keySize     Size of KeyPair (1024, 2048, 4096)
 *
 * \return                  TRUE on error; FALSE else
 **/
BOOL
EscCrypto_LoadPrivateKey(
    SDC_PARAMS_s *sdc,
    const char* const file,
    void* const keyPair,
    int keySize);

/**
 * This function extracts a rsa/pkcs#1 key from a pkcs#8 container
 * it allocates the necessary memory - This memory needs to be free'd by the caller
 *
 * \param[in]   pkcs8Data       file path and name
 * \param[in]   pkcs8Size       Pointer to KeyPair to be filled
 * \param[out]  rsaPrivKeyData  Pointer to extracted Private Key
 * \param[out]  rsaPrivKeySize  Size of rsaPrivKey
 *
 * \return                  TRUE on error; FALSE else
 **/
BOOL extractPrivKeyFromPKCS8(
		const UINT8* pkcs8Data,
		UINT32 pkcs8Size,
		UINT8** rsaPrivKeyData,
		UINT32* rsaPrivKeySize);

// ToDo never used --> Keep?
/**
 * This function encodes (using DER) and stores a private key into a file
 *
 * \deprecated Use EscCyrpto_StorePrivateKey2() instead
 *
 * \param[in]   file        File name
 * \param[in]   keyPair     RSA-2048 key pair
 *
 * \return                  TRUE on error; FALSE else
 **/
extern BOOL
EscCrypto_StorePrivateKey(
    const char* const file,
    const EscCrypto_RsaKeyPairT* const keyPair );

/**
 * This function encodes and stores a private key into a file,
 * supporting the following formats and containers
 *
 * \li PEM / DER encoding
 * \li PKCS#1 / PKCS#8 container
 *
 * \param[in]   sdc         Pointer to SDC parameter structure
 * \param[in]   file        File name
 * \param[in]   containerid Key container identifier
 * \param[in]   formatid    Key format identifier
 * \param[in]   keyPair     RSA key pair
 * \param[in]   keySize     RSA key pair Size
 *
 * \return                  TRUE on error; FALSE else
 **/
BOOL EscCrypto_StorePrivateKey2(
    SDC_PARAMS_s *sdc,
    const char* const file,
    const UINT32 containerid,
    const UINT32 formatid,
    const void* const keyPair,
	UINT32 keySize);


/**
 * This function signs a message according to RSASSA-PKCS1-v1_5_Sign using SHA-1 and RSA-2048.
 *
 * \param[in]   keyPair          RSA key pair.
 * \param[in]   message          Message to be signed.
 * \param[in]   messageLen       Length of the message.
 * \param[out]  signature        Signature
 * \param[in]   hashType         Hash algorithm to use
 * \param[in]   keySize     RSA key pair Size
 *
 * \return TRUE on error
 */
extern BOOL
EscCrypto_SignMessage(
    const void* const keyPair,
    const UINT8 message[],
    const UINT32 messageLen,
    UINT8* signature,
    const HASH_t hashType,
    UINT32 keySize);

/**
 * This function verifies a signature according to RSASSA-PKCS1-v1_5_Verify using SHA-1 and RSA-2048.
 *
 * \param[in]   keyPair     RSA key pair.
 * \param[in]   message     Signed message.
 * \param[in]   messageLen  Length of the message.
 * \param[in]   signature   Signature to be verified
 * \param[in]   hashType    Hash algorithm to use
 * \param[in]   keySize     RSA key pair Size
 *
 * \return TRUE on error
 */
extern BOOL
EscCrypto_VerifyMessage(
    const void* const keyPair,
    const UINT8* message,
    const UINT32 messageLen,
    const UINT8* signature,
    const HASH_t hashType,
    UINT32 keySize);

/**
 * This function verifies a signature according to RSASSA-PKCS1-v1_5_Verify using SHA-1 and RSA-2048.
 *
 * \param[in]   keyPair     RSA key pair.
 * \param[in]   message     Signed message.
 * \param[in]   messageLen  Length of the message.
 * \param[in]   signature   Signature to be verified
 * \param[in]   hashType    Hash algorithm to use
 *
 * \return TRUE on error
 */
extern BOOL
EscCrypto2048_VerifyMessage(
    const EscRsa_KeyPairT* const keyPair,
    const UINT8 message[],
    const UINT32 messageLen,
    const UINT8 signature[ EscRsa_KEY_BYTES ],
    const HASH_t hashType );

/**
 * This function verifies a signature according to RSASSA-PKCS1-v1_5_Verify using SHA-1 and RSA-4096.
 *
 * \param[in]   keyPair     RSA key pair.
 * \param[in]   message     Signed message.
 * \param[in]   messageLen  Length of the message.
 * \param[in]   signature   Signature to be verified
 * \param[in]   hashType    Hash algorithm to use
 *
 * \return TRUE on error
 */
extern BOOL
EscCrypto4096_VerifyMessage(
    const EscRsa4096_KeyPairT* const keyPair,
    const UINT8 message[],
    const UINT32 messageLen,
    const UINT8 signature[ EscRsa4096_KEY_BYTES ],
    const HASH_t hashType );


/**
 * This function verifies a signature according to RSASSA-PKCS1-v1_5_Verify using SHA-1 and RSA-1024.
 *
 * \param[in]   keyPair     RSA key pair.
 * \param[in]   message     Signed message.
 * \param[in]   messageLen  Length of the message.
 * \param[in]   signature   Signature to be verified
 * \param[in]   hashType    Hash algorithm to use
 *
 * \return TRUE on error
 */
extern BOOL
EscCrypto1024_VerifyMessage(
    const EscRsa1024_KeyPairT* const keyPair,
    const UINT8 message[],
    const UINT32 messageLen,
    const UINT8 signature[ EscRsa4096_KEY_BYTES ],
    const HASH_t hashType );

/**
 * This function extracts the public modulus size of a RSA key
 *
 * \param[in]   srcKeyData Pointer to key data
 * \param[in]   srcKeySize Key data size
 *
 * \param[out]  keyModSize Size of keys public modulus
 *
 * \return TRUE on error
 */
BOOL
EscCrypto_GetPubKeyModSize (
    UINT8 *const srcKeyData,
    UINT32 const srcKeySize,
    UINT32 *keyModSize);

/**
 * This function returns the Size occupied by a KeyPair in Bytes for the Key Size in Bits
 *
 * \param[in]   keySize in Bits
 *
 * \return size in Bytes
 */
UINT32 getKeyPairSize(UINT32 keySize);

#endif	/* _CRYPTO_H_ */
