/***************************************************************************
 *                                                                         *
 * Copyright                                                               *
 *     escrypt GmbH, Bochum, Germany                                       *
 *     Lise-Meitner-Allee 4                                                *
 *     D-44801 Bochum, Germany                                             *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        crl_utils.h

   \brief       Utilities used for CRL Handling

   ***

   $Rev: 780 $
 */
/***************************************************************************
$Author: mlange $
$Date: 2017-07-28 15:40:59 +0200 (Fr, 28. Jul 2017) $
****************************************************************************/

#ifndef CRL_INC_CRL_UTILS_H_
#define CRL_INC_CRL_UTILS_H_

#ifdef __cplusplus
extern "C" {
#endif

/***************************************************************************
 * 1. INCLUDES                                                             *
 ***************************************************************************/

#include "../../cmp/asn1-2005-multi-file-os-asn1c/INTEGER.h"
#include "../../cmp/asn1-2005-multi-file-os-asn1c/CertificateList.h"
#include "../../cmp/inc/cmp.h"
#include "../../cmp/inc/cmp_utils.h"

/* ecm Includes */
#include "../../common/inc/esc_common.h"


/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/***************************************************************************
 * 3. DECLARATIONS                                                         *
 ***************************************************************************/

/** CRL ID enumeration */
typedef enum {
    /** ID none */
    crl_id_none = 0,
    crl_id_root,
    crl_id_ic,
    crl_id_ec,
    crl_id_unused
} CRL_ID_e;

typedef struct CR_LIST_t {
    INTEGER_t userID;
    char* issuerCN;
    struct CR_LIST_t *next;

} CR_LIST_t;

typedef struct CRL_HASHES_t {
    CRL_ID_e id;
    BOOL store;
    EXT_IF_ERROR_CODES_t err;
    char* issuerCN;
    UINT8* digest;
    UINT8* certList;
    UINT32 certListLen;
    BOOL validSignature;
    BOOL outdated;
    ull_time_t thisUpdate;
    ull_time_t nextUpdate;
    CertificateList_t *cl;
    struct CRL_HASHES_t *next;
} CRL_HASHES_t;

BOOL
getCrlFromListByLocalId (
    CRL_HASHES_t *lst,
    CRL_ID_e id,
    CRL_HASHES_t **crlPtr );

BOOL determineCrlTypeByIssuerCn (
    CRL_HASHES_t **crl,
    CERT_LIST_s *certList,
    STATUS_s *stat);

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/

/***************************************************************************
 * 5. FUNCTION PROTOTYPES                                                  *
 ***************************************************************************/

/**
This function is used to load a CRL from filesystem
!!! if certlist is NULL the CRLs are NOT checked !!!

\param[in] path
    path to a CRL on filesystem

\param[out] crList
    List of SerialNumber and Issuer CN of revoked Certificates
    List is allocated by this function and must be freed by the caller!

\param[in] certList
    List of Certificates used to verify Signature of CRL
    !!! if NULL the CRLs are NOT checked !!!

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL loadCRL(char* path, const CR_LIST_t** crList, CERT_LIST_s* certList,
        CRL_HASHES_t* digest);

/**
This function is used to add Certificate Revocation entries to a CR_LIST_t
The entries are parsed from the CRL specified by the URI

\param[in] uri
    URI to CRL

\param[in] certList
    List of Certificates to verify the CRL

\param[out] crList
    List of SerialNumber and Issuer CN of revoked Certificates
    List is allocated by this function and must be freed by the caller!

\param[out] crlCreationTime
    Pointer where CRL creation time is stored

\param[out] crlNextUpdTime
    Pointer where CRL next update time is stored

\return #int
\retval #int amount of added CR entries
\retval #-1 on failure
*/
int
addCRLfromURI(
    char* uri,
    CR_LIST_t** list,
    CERT_LIST_s* certList,
    ull_time_t* crlCreationTime,
    ull_time_t* crlNextUpdTime,
    CRL_HASHES_t** crlHashes,
    STATUS_s *stat);

/**
This function is used to verify a CRLs Signature

\param[in] crList
    pointer to CRL

\param[in] certList
    pointer to certificate list

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL verifyCRLSignature(
    CertificateList_t* crList,
    CERT_LIST_s* certList,
    STATUS_s *stat );

/**
This function is used to free a List of CRL Hashes

\param[in] crlHashes
    pointer CRL hashses list

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
void freeCRLHashes(CRL_HASHES_t* crlHashes);

/**
This function is used to free a CR List

\param[in] crList
    pointer CR list

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
void freeCRL(CR_LIST_t* crList);

/**
This function is used to verify a Certificate against a CRL

\param[in] p_certificate
    pointer to certificate

\param[in] p_crList
    pointer to certificate revocation list

\return #BOOL
\retval #FALSE on success (Certificate could be verified)
\retval #TRUE on failure (Certificate is part of the CRL or could not be verified)

*/
BOOL verifyCertificateAgainstCRL(Certificate_t* p_certificate, CR_LIST_t* p_crList);

/**
This function is used to verify a Certificate List against a CRL

\param[in] p_certList
    pointer to certificateList

\param[in] p_crList
    pointer to certificate revocation list

\return #BOOL
\retval #FALSE on success (Certificate could be verified)
\retval #TRUE on failure (Certificate is part of the CRL or could not be verified)

*/
BOOL verifyCertListAgainstCRL(
    CERT_LIST_s* p_certList,
    CR_LIST_t* p_crList,
    STATUS_s *stat );

/**
This function is used to download a CRL file
If the URL contains a File, it will be stored as downloadedCRL.crl in the
working directory
The file is NOT verified nor checked if it is truly a CRL. Verification is done
in loadCRL() resp. verifyCRLSignature()

\param[in] url
    URL from where to download the CRL file

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL downloadCRL(char* uri);

/**
This function is used to get the URI of a corresponding CRL file
The URI is stored in the certificate within X509v3 extensions -> X509v3 CRL Distribution Points


\param[in] certificate
    certificate from which to extract the CRL-File URI

\return #char*
\retval #URI on success
\retval #NULL on failure

*/
char* getCRLURIfromCertificate(Certificate_t*);


/**
This function is used to hash a CertificateList (e.g a CRL)
The hash does not take the timestamp into account

\param[in] crl
    Certificate List which shall be hashed

\param[out] digest
    created hash digest

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL hashCRL(CertificateList_t* crl, UINT8** digest);

BOOL
removeCrListEntryByIssuerCn (
    char *issuerCn,
    CR_LIST_t **lst );


BOOL
removeCrlHashEntryByIssuerCn (
    char *issuerCn,
    CRL_HASHES_t **crl,
    BOOL outdatedOnly);

/***************************************************************************
 * 6. MACRO FUNCTIONS                                                      *
 ***************************************************************************/

/***************************************************************************
 * 7. END                                                                  *
 ***************************************************************************/

#ifdef __cplusplus
}
#endif

#endif /* CRL_INC_CRL_UTILS_H_ */
