/***************************************************************************
 *                                                                         *
 * Copyright                                                               *
 *     escrypt GmbH, Bochum, Germany                                       *
 *     Lise-Meitner-Allee 4                                                *
 *     D-44801 Bochum, Germany                                             *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        esc_asn1_utils.c

   \brief       Common ASN1 utilities module 

   $Rev: 812 $
 */
/***************************************************************************
$Author: mlange $
$Date: 2017-08-25 15:34:36 +0200 (Fr, 25. Aug 2017) $
****************************************************************************/

/***************************************************************************
* 1. INCLUDES                                                              *
****************************************************************************/

/* Linux headers */
#include <stdio.h>

/* common headers */
#include "../../ecm/inc/ecm.h"
#include "../inc/esc_common.h"
#ifdef CONFIG_SIMPLE_DEBUG
# include "../inc/esc_debug_simple.h" 
#else
# include "../inc/esc_debug.h" 
#endif /* CONFIG_SIMPLE_DEBUG */

#include "../inc/esc_utils.h"

/* ASN1 header */
#include "../../cmp/asn1-2005-multi-file-os-asn1c/PKIMessage.h"
#include "../inc/esc_asn1_utils.h"

//#include "../../crl/inc/crl_utils.h"

#ifdef UNIT_TESTING
#include "ecmUnitTests.h"
#endif

/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/***************************************************************************
 * 3. DEFINITIONS                                                          *
 ***************************************************************************/

/** Callback parameters structure */
typedef struct {
    /** Data size */
    UINT32 size;
    /** Pointer where data is stored */
    UINT8 *data;
    /** Size of the currently allocated buffer */
    UINT32 bufSize;
    /** Pointer within data */
    UINT8 *ptr;
} CB_PARAMS_t;

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/

/***************************************************************************
 * 5. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/

#ifdef UNIT_TESTING
#undef calloc
#endif
BOOL alloc ( 
    void **ptr,
    int size )
{
    BOOL failed = FALSE;

    if ( ptr == NULL || size <= 0 ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter\n", DBG_FL );
        return ( TRUE );
    }

    *ptr = calloc(1, (size_t)size);
    if (!(*ptr)) {
        dbgPrint ( DBG_ERR, DBG_FLS "calloc() failed\n", DBG_FL );
        failed = TRUE;
    }

    return ( failed );
}
#ifdef UNIT_TESTING
#define calloc(num, size) _test_calloc(num, size, __FILE__, __LINE__)
#endif


BOOL asn1U322Integer (
    INTEGER_t *asnInt,
    UINT32 u32 )
{

    int ret = asn_long2INTEGER ( asnInt, u32 );

    if ( ret < 0 ) {

        dbgPrint ( DBG_ERR, "ASN1 integer conversion failed\n" );
        return ( TRUE );
    } 

    return ( FALSE );
}

static int
asn1XerPrint2Buf (
    const void *buffer, 
    size_t size,
    void *param )
{
    BOOL failed = FALSE;
    CB_PARAMS_t *params = (CB_PARAMS_t *)param;
    char *ptr;

    /* sanity checks */
    if ( !buffer || !param ) {
        failed = TRUE;
    }

    ifNotFailed {

        /* initial allocation */
        if ( params->data == NULL ) {
            params->data = malloc ( DEFAULT_MALLOC_SIZE );
            params->bufSize = DEFAULT_MALLOC_SIZE;
        }

        /* reallocation */
        while ( ( params->size + size) > params->bufSize ) {
            params->data = realloc ( params->data, params->bufSize + DEFAULT_MALLOC_SIZE );
            params->bufSize += DEFAULT_MALLOC_SIZE;
        }

        ptr = (char *)params->data + params->size;

        memcpy ( ptr, buffer, size );
        params->size += size;
    }

    return 0;
}

BOOL asn1DebugPrint ( 
    asn_TYPE_descriptor_t *asnType,
    void *asnPtr )
{
    asn_enc_rval_t ret;
    CB_PARAMS_t params;

    if ( (asnType == NULL) || (asnPtr == NULL) ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter\n", DBG_FL );
        return ( TRUE );
    }

    memset ( &params, 0x0, sizeof ( params ));

    ret = xer_encode(asnType, asnPtr, XER_F_BASIC, asn1XerPrint2Buf, &params);

    if ( ret.encoded == -1 ) {

        dbgPrint ( DBG_ERR, "ASN1 encoding for logging failed\n" );

        /* free data buffer */
        if ( params.data ) {
            free ( params.data );
        }

        return ( TRUE );

    } else {

        /* just one more byte for string termination */
        if ( params.data ) {

            params.data = realloc (params.data, params.size + 1 );
            params.data[params.size] = '\0';
            dbgPrint ( DBG_ASN1, "ASN1 XML representation:\n%s\n", params.data );

            /* free data buffer */
            free ( params.data );
        }
    }

    return ( FALSE );
}
BOOL asn1Encode (
    void *asnPtr,
    asn_TYPE_descriptor_t *asnType,
    UINT8 **data,
    UINT32 *dataSize,
    BOOL printAllowed )
{
    CB_PARAMS_t params;
    BOOL failed = FALSE;
    asn_enc_rval_t retVal;

    dbgIn ( __func__ );

    if ( (asnType == NULL) || (asnPtr == NULL) || (data == NULL) || (dataSize == NULL) ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter\n", DBG_FL );
        return ( TRUE );
    }

    memset ( &params, 0x0, sizeof ( params ));

    retVal = der_encode ( asnType, asnPtr, asn1XerPrint2Buf, &params );

    if ( retVal.encoded < 0 ) {
        dbgPrint(DBG_ERR, "Encoding failed (at %s)\n",
            retVal.failed_type->name);

        failed = TRUE;
    }

    if ( failed == FALSE ) {

        *dataSize = retVal.encoded;
        *data = malloc ( *dataSize );

        if ( NULL == *data ) {
            failed = TRUE;
        }
        ifNotFailed {
            memcpy( *data, params.data, params.size );

            if ( printAllowed == TRUE ) {
                dbgPrint(DBG_DATA, "%d byte encoded\n", retVal.encoded);
                failed = asn1DebugPrint ( asnType, asnPtr );
            }
        }

    } else {

        *dataSize = 0;
    }

    /* free data buffer */
    if ( params.data ) {
        free ( params.data );
    }

    dbgOut ( __func__ );

    return ( failed );
}

BOOL asn1EncodeToBuffer (
    void *asnPtr,
    asn_TYPE_descriptor_t *asnType,
    UINT8 data[],
    UINT32 *dataSize,
    BOOL printAllowed )
{
    BOOL failed = FALSE;
    asn_enc_rval_t retVal;

    dbgIn ( __func__ );

    if ( (asnType == NULL) || (asnPtr == NULL) || (data == NULL) || (dataSize == NULL) ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter\n", DBG_FL );
        return ( TRUE );
    }

    retVal = der_encode_to_buffer( asnType, asnPtr, data, (size_t)*dataSize );

    if ( retVal.encoded < 0 ) {
        dbgPrint(DBG_ERR, "Encoding failed (at %s)\n",
            retVal.failed_type->name);

        failed = TRUE;
    }

    if ( failed == FALSE ) {

        *dataSize = retVal.encoded;

        if ( printAllowed == TRUE ) {
            dbgPrint(DBG_DATA, "%d byte encoded\n", retVal.encoded);
            failed = asn1DebugPrint ( asnType, asnPtr );
        }

    } else {

        *dataSize = 0;
    }

    dbgOut ( __func__ );

    return ( failed );
}

BOOL asn1Decode (
    const UINT8 data[],
    UINT32 dataSize,
    asn_TYPE_descriptor_t *asnType,
    void *asnPtr,
    BOOL printAllowed )
{
    BOOL failed = FALSE;
    asn_dec_rval_t retVal;

    dbgIn( __func__ );

    if ( (asnType == NULL) || (asnPtr == NULL) || (data == NULL)) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter\n", DBG_FL );
        return ( TRUE );
    }

    retVal = ber_decode(
        /*@-nullpass@*/
        0,
        /*@=nullpass@*/
        asnType,
        (void **)&asnPtr,
        data,
        (size_t) dataSize );

    if ( retVal.code != RC_OK ) {

        dbgPrint(DBG_ERR, "Broken encoding at byte %ld\n",
            (long) retVal.consumed );

        failed = TRUE;
    }

    if ( FALSE == failed ) {

        if ( printAllowed == TRUE ) {
            dbgPrint ( DBG_DATA, "%d bytes decoded\n", (long) retVal.consumed );
            failed = asn1DebugPrint ( asnType, asnPtr );
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}

void asnFreeStructContent (
    void *ptr,
    asn_TYPE_descriptor_t type )
{
    if ( ptr ) {
        ASN_STRUCT_FREE_CONTENTS_ONLY( type, ptr );
    }
}

void asnFreeStruct (
    void **ptr,
    asn_TYPE_descriptor_t type )
{
    if ( ptr ) {
        if ( *ptr ) {
            ASN_STRUCT_FREE ( type, *ptr );
            *ptr = NULL;
        }
    }
}

/***************************************************************************
 * 6. END                                                                  *
 ***************************************************************************/

