/***************************************************************************
 *                                                                         *
 * Copyright                                                               *
 *     escrypt GmbH, Bochum, Germany                                       *
 *     Lise-Meitner-Allee 4                                                *
 *     D-44801 Bochum, Germany                                             *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        esc_utils.h

   \brief       Common utilities module

   ***

   $Rev: 812 $
 */
/***************************************************************************
$Author: mlange $
$Date: 2017-08-25 15:34:36 +0200 (Fr, 25. Aug 2017) $
****************************************************************************/

#ifndef __ESC_UTILS_H__
#define __ESC_UTILS_H__

#include "esc_common.h"
#include "../../ecm/inc/ecm.h"
#include <stdlib.h>    
#include <unistd.h>

#ifdef __cplusplus
extern "C" {
#endif

/***************************************************************************
 * 1. INCLUDES                                                             *
 ***************************************************************************/

/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/** Flag to signal remove trailing blanks */
#define ESC_REM_TRAIL_BLANKS 0x01
/** Flag to signal remove trailing CRLF */
#define ESC_REM_TRAIL_CRLF   0x02
/** Flag to signal remove leading blanks */
#define ESC_REM_LEAD_BLANKS  0x04
/** Count of possible configuration file locations */
#define CONF_FILE_LIST_SIZE 3

/***************************************************************************
 * 3. DECLARATIONS                                                         *
 ***************************************************************************/

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/

/***************************************************************************
 * 5. FUNCTION PROTOTYPES                                                  *
 ***************************************************************************/

BOOL escInt2SizetAndCheckForOv (
    int i,
    size_t *s );

BOOL escCheckSizetForOv (
    size_t s );

/**
This function is used to free the dirent structure allocated by scandir().
We use 'void **dir' because 'struct dirent **dir' is not visible at every
source file.
The pointer will be casted for use.

\param[in] dir
    Pointer to dirent structure to free
\param[in] len
    Amount of struct dirent entries

\return none

*/
void escFreeDirent (
    void **dir,
    UINT32 len );

/**
This function is used to free a given pointer and set to NULL.

\param[in] ptr
    Pointer to pointer tofree

\param[out] ptr
    Pointer to freed pointer

\return none

*/
void
escFreePtr (
    UINT8 **ptr );

/**
This function is used to allocate memory

\param[in] ptr
    Pointer to pointer where memory should be allocated
\param[in] size
    Amount of memory that should be allocated

\param[out] ptr
    Pointer to pointer to allocated memory

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL escMalloc (
    void **ptr,
    int size );

/**
This function is used to copy a configuration file option to the internal
structure.

\param[in] str
    Pointer to a string that contains the parameter

\param[out] buf
    Pointer to pointer where parameter is stored

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
copyParameterValue (
    const char *str,
    char **buf );

/**
This function is used to copy strings from a comma separated list (CSV) to the
internal structure STRING_LIST_s.

\param[in] src
    Pointer to CSV list

\param[out] dst
    Pointer to STRING_LIST_s destination

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
escCsvList2StringList (
    char *src,
    STRING_LIST_s **dst );

/**
This function is used to free a STRING_LIST_s list.

\param[in] lst
    Pointer to STRING_LIST_s list

*/
void
escFreeStringList (
    STRING_LIST_s **lst );

/**
This function is used to convert a string to a #BOOL value.

\param[in] str
    Pointer to a string that contains the value

\param[out] boolVal
    Pointer to #BOOL where parameter is stored

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
str2Bool (
    const char *str,
    BOOL *boolVal );

/**
This function is used to make the application a daemon.

\return none

*/
void escDaemonize (
    void );

/**
This function is used to remove trailing "\r" or "\n" from a string

\return none

*/
void
remTrailingCrLf (
    char **str );

/**
This function is used to read a configuration file.

\param[in] confFile
    Pointer to configuration file name structure.

\param[out] par
    Pointer to global application structure (ECM_PARAMS_t or LRA_PARAMS_t)

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL readConfFile (
    CONF_FILES_t *confFile,
    void *par );

/**
This function is used to retrieve the size of a file which is identified by
the given file pointer.

\param[in] fp
    File pointer to open file

\param[out] size
    Pointer where file size is stored

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
escGetFileSizeByFp (
    FILE *fp,
    UINT32 *size );

/**
This function is used to lookup a corresponding string matching the
given index number at a STR_LOOKUP_t structure.

\param[in] tab
    Pointer to lookup table
\param[in] tabSize
    Amount of table entries
\param[in] id
    Index number

\return char*
\retval Pointer to string

*/
char
*escId2StringLookup (
    const STR_LOOKUP_t *tab,
    const UINT32 tabSize,
    UINT32 id );

/**
This function is used to lookup a corresponding index number matching the
given string at a STR_LOOKUP_t structure.

\param[in] tab
    Pointer to lookup table
\param[in] tabSize
    Amount of table entries
\param[in] str
    String to lookup

\param[out] id
    Pointer where index number will be stored

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
escString2IdLookup (
    const STR_LOOKUP_t *tab,
    const UINT32 tabSize,
    const char* str,
    UINT32 *id );

/***************************************************************************
 * 6. MACRO FUNCTIONS                                                      *
 ***************************************************************************/

/***************************************************************************
 * 7. END                                                                  *
 ***************************************************************************/

#ifdef __cplusplus
}
#endif

#endif /* __ESC_UTILS_H__ */
