/***************************************************************************
 *                                                                         *
 * Copyright                                                               *
 *     escrypt GmbH, Bochum, Germany                                       *
 *     Lise-Meitner-Allee 4                                                *
 *     D-44801 Bochum, Germany                                             *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        esc_asn1_utils.h

   \brief       Common ASN1 utilities module

   ***

   $Rev: 634 $
 */
/***************************************************************************
$Author: mlange $
$Date: 2017-02-15 14:59:54 +0100 (Mi, 15. Feb 2017) $
****************************************************************************/

#ifndef __ESC_ASN1_UTILS_H__
#define __ESC_ASN1_UTILS_H__

#ifdef __cplusplus
extern "C" {
#endif

/***************************************************************************
 * 1. INCLUDES                                                             *
 ***************************************************************************/

/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/** Default size for buffer allocation */
#define DEFAULT_MALLOC_SIZE 2048

/***************************************************************************
 * 3. DECLARATIONS                                                         *
 ***************************************************************************/

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/

/***************************************************************************
 * 5. FUNCTION PROTOTYPES                                                  *
 ***************************************************************************/

/**
This function is used to encode a an ASN1 structure to a byte array.
The buffer where the result is stored, is allocated by this function and
has to be freed by the calling application.

\param[in] asnPtr 
    Pointer where ASN1 structure is stored
\param[in] asnType 
    Pointer to ASN1 type definition we want to convert from
\param[in] printAllowed 
    Flag to signal if printing the XML representation of the ASN1 structure
    is allowed

\param[out] data
    Pointer to pointer where data is stored
\param[out] dataSize 
    Pointer where size of the data buffer is stored


\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
asn1Encode (
    void *asnPtr,
    asn_TYPE_descriptor_t *asnType,
    UINT8 **data,
    UINT32 *dataSize,
    BOOL printAllowed );

/**
This function is used to encode a an ASN1 structure to an
allready allocated byte array.

\param[in] asnPtr
    Pointer where ASN1 structure is stored
\param[in] asnType
    Pointer to ASN1 type definition we want to convert from
\param[in] printAllowed
    Flag to signal if printing the XML representation of the ASN1 structure
    is allowed

\param[out] data
    Pointer where data is stored
\param[out] dataSize
    Pointer where size of the data buffer is stored


\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
asn1EncodeToBuffer (
    void *asnPtr,
    asn_TYPE_descriptor_t *asnType,
    UINT8 data[],
    UINT32 *dataSize,
    BOOL printAllowed );

/**
This function is used to decode a byte array to an ASN1 structure

\param[in] data
    Pointer where data is stored
\param[in] dataSize 
    Size of the data buffer to decode
\param[in] asnType 
    Pointer to ASN1 type definition we want to convert to
\param[in] printAllowed 
    Flag to signal if printing the XML representation of the ASN1 structure
    is allowed

\param[out] asnPtr 
    Pointer where ASN1 structure is stored

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
asn1Decode (
    const UINT8 data[],
    UINT32 dataSize,
    asn_TYPE_descriptor_t *asnType,
    void *asnPtr, 
    BOOL printAllowed );

/**
This function is used to convert an UINT32 to an ASN1 INTEGER type

\param[in] u32 
    UINT32 value we want to convert

\param[out] asnInt
    Pointer where ASN1 INTEGER type is stored

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
asn1U322Integer (
    INTEGER_t *asnInt,
    UINT32 u32 );

/**
This function is used to allocate memory for a given ASN1 type

\param[in] ptr
    Pointer to pointer where memory should be allocated
\param[in] size
    Amount of memory that should be allocated

\param[out] ptr 
    Pointer to pointer to allocated memory

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
alloc ( 
    void **ptr,
    int size );

/**
This function is used to print the XML representation of an ASN1 value.

\param[in] asnType
    ASN1 type identifier
\param[in] asnPtr
    Pointer where ASN value is stored

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL asn1DebugPrint (
    asn_TYPE_descriptor_t *asnType,
    void *asnPtr );

/**
This function is used to free the content of an ASN.1 structure and the pointer.
The pointer is set to NULL by this function.

\param[in] ptr
    ASN1 structure pointer to free
\param[in] type
    ASN1 type

\return none

*/
void
asnFreeStruct (
    void **ptr,
    asn_TYPE_descriptor_t type );

/**
This function is used to free the content of an ASN.1 structure.

\param[in] ptr
    ASN1 structure pointer to free content
\param[in] type
    ASN1 type

\return none

*/
void
asnFreeStructContent (
    void *ptr,
    asn_TYPE_descriptor_t type );

/***************************************************************************
 * 6. MACRO FUNCTIONS                                                      *
 ***************************************************************************/

/***************************************************************************
 * 7. END                                                                  *
 ***************************************************************************/

#ifdef __cplusplus
}
#endif

#endif /* __ESC_ASN1_UTILS_H__ */
