/***************************************************************************
 *                                                                         *
 * Copyright                                                               *
 *     escrypt GmbH, Bochum, Germany                                       *
 *     Lise-Meitner-Allee 4                                                *
 *     D-44801 Bochum, Germany                                             *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        cmp_utils.c

   \brief       Utilities module of CMP library

   $Rev: 812 $
 */
/***************************************************************************
$Author: mlange $
$Date: 2017-08-25 15:34:36 +0200 (Fr, 25. Aug 2017) $
****************************************************************************/

/***************************************************************************
* 1. INCLUDES                                                              *
****************************************************************************/

/* linux headers */
#include <limits.h>
#include <time.h>
#include <arpa/inet.h>
#include <stdio.h>

#include "../inc/cmp_utils.h"

/* Common ASN.1 header */
#include "../asn1-2005-multi-file-os-asn1c/PKIMessage.h"

#include "../../common/inc/esc_common.h"

/* local ASN1 header */
#include <asn1_utils.h>

/* common headers */
#include "../../ecm/inc/ecm.h"
#include "../../ecm/inc/ecm_cm_main.h"
#include "../../ecm/inc/ecm_cm_EventHandler.h"
#include "../../common/inc/esc_debug.h" 
#include "../../common/inc/esc_utils.h"
#include "../../common/inc/esc_asn1_utils.h"

#include "../../ecm/inc/ecm_utils.h"

/* cryto header */
#include "../../crypto/inc/crypto.h"
#include "../inc/cmp_utils.h"




/* CMP header */
#include "../inc/cmp_utils.h"

#include "../inc/asn1_utils.h"

#ifdef UNIT_TESTING
#include "ecmUnitTests.h"
#endif


/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/***************************************************************************
 * 3. DEFINITIONS                                                          *
 ***************************************************************************/

/*#ifndef CONFIG_CM
static UINT32 certRepStatusLutSize =
    sizeof ( certRepStatusLut ) / sizeof ( STR_LOOKUP_t );
#endif  !CONFIG_CM */

static const STR_LOOKUP_t oidLut[] = {
    { oid_country,           "C"                       },
    { oid_commonName,        "CN"                      },
    { oid_serialNumber,      "serialNumber"            },
    { oid_stateOrProvince,   "ST"                      },
    { oid_localityName,      "L"                       },
    { oid_name,              "name"                    },
    { oid_distinguishedName, "distinguishedName"       },
    { oid_organization,      "O"                       },
    { oid_organizationUnit,  "OU"                      },
    { oid_description,       "description"             },
    { oid_uniqueIdentifier,  "uniqueIdentifier"        },
    { oid_uuidpair,          "uuIdPair"                },
    { oid_userPw,            "userPw"                  },
    { oid_regToken,          "regToken"                },
    { oid_rsaEnc,            "rsaEncryption"           },
    { oid_shaWithRsaEnc,     "shaWithRsaEncryption"    },
    { oid_sha256WithRsaEnc,  "sha256WithRsaEncryption" },
    { oid_sha512WithRsaEnc,  "sha256WithRsaEncryption" },
    { oid_title,             "T"                       },
    { oid_printableString,   "printableSting"          },
    { oid_unsupported,       "unsupported"             },
};

static UINT32 oidLutSize =
    sizeof ( oidLut ) / sizeof ( STR_LOOKUP_t );

static const STR_LOOKUP_t oidSrcLut[] = {
    { subject, "subject" },
    { issuer,  "issuer"  },
};

static UINT32 oidSrcLutSize =
    sizeof ( oidSrcLut ) / sizeof ( STR_LOOKUP_t );

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/

/***************************************************************************
 * 5. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/

BOOL copyCertListItem (
    CERT_LIST_s *src,
    CERT_LIST_s **dst )
{
    BOOL failed = FALSE;

    /* sanity checks */
    if ( !src || !dst ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {

        *dst = calloc (1, sizeof ( CERT_LIST_s ));
        if ( NULL == *dst ) {
            failed = TRUE;
        }
    }

    ifNotFailed {

        /* Valid flag */
        (*dst)->validSignature = src->validSignature;

        (*dst)->issuerAvailable = src->issuerAvailable;

        /* Device chain flag */
        (*dst)->devChain = src->devChain;

        /* Store flag */
        (*dst)->store = src->store;

        /* TLS trust flag */
        (*dst)->tlsTrust = src->tlsTrust;

        /* copy file name */
        if ( src->fileName ) {
            (*dst)->fileName = strdup ( src->fileName );
        }

        /* copy cert ID */
        (*dst)->certId = src->certId;

        /* copy title */
        if ( src->title ) {
            (*dst)->title = strdup ( src->title );
        }

        /* Copy subject CN */
        if ( src->subjectCn ) {
            (*dst)->subjectCn = strdup ( src->subjectCn );
        }

        /* Copy issuer CN */
        if ( src->issuerCn ) {
            (*dst)->issuerCn = strdup ( src->issuerCn );
        }

        /* Copy cert size */
        (*dst)->certSize = src->certSize;

        /* Copy certificate content */
        (*dst)->certPtr = malloc ((*dst)->certSize);
        if ( NULL == (*dst)->certPtr ) {
            failed = TRUE;
        }
    }

    ifNotFailed {
        memcpy ( (*dst)->certPtr, src->certPtr, (*dst)->certSize );

        /* Skip decoded certificate
           (will be decoded during next operation if needed) */

        /* Copy CRL URI */
        if ( src->crlUri ) {
            (*dst)->crlUri = strdup ( src->crlUri );
        }

        /* copy validities */
        (*dst)->validNotBefore = src->validNotBefore;
        (*dst)->validNotAfter = src->validNotAfter;
    }

    return ( failed );
}

/* Returns the pointer to the certificate if certificate
   with the given issuer is listed in our certificate list */
CERT_LIST_s *checkCertListForCertByIssuerCN (
    CERT_LIST_s *lst,
    char *issuerName )
{
    BOOL failed = FALSE;
    CERT_LIST_s *ptr = lst, *ret = NULL;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !lst || !issuerName ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        while ( ptr ) {
            if ( strncmp (ptr->issuerCn, issuerName, strlen (ptr->issuerCn)) == 0 ) {
                ret = ptr;
                break;
            } else {
                ptr = ptr->next;
            }
        }
    }

    dbgOut ( __func__ );

    return ( ret );
}


/* Returns the pointer to the certificate if certificate
   with the given subject is listed in our certificate list */
CERT_LIST_s *checkCertListForCertBySubjectCN (
    CERT_LIST_s *lst,
    char *subjName )
{
    BOOL failed = FALSE;
    CERT_LIST_s *ptr = lst, *ret = NULL;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !lst || !subjName ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        while ( ptr ) {
            if ( strncmp (ptr->subjectCn, subjName, strlen (ptr->subjectCn)) == 0 ) {
                ret = ptr;
                break;
            } else {
                ptr = ptr->next;
            }
        }
    }

    dbgOut ( __func__ );

    return ( ret );
}

/* returns TRUE if cert with certId is not listed in lst */
BOOL checkCertListForCertByLocalId (
    CERT_LIST_s *lst,
    CERT_ID_e certId,
    CERT_LIST_s **crtPtr )
{
    BOOL failed = FALSE;
    CERT_LIST_s *ptr = lst;

    dbgIn ( __func__ );

    /* sanity checks, lst may be empty */
    if ( !crtPtr ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        failed = TRUE;
        while ( ptr ) {
            if ( ptr->certId == certId ) {
                failed = FALSE;
                *crtPtr = ptr;
                break;
            } else {
                ptr = ptr->next;
            }
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}

UINT32 cmpGetCertListSize (
    CERT_LIST_s *lst )
{
    CERT_LIST_s *ptr;
    UINT32 cnt = 1;

    dbgIn ( __func__ );

    ptr = lst;

    if ( ptr->next  ) {
        while ( ptr->next ) {
            cnt++;
            ptr = ptr->next;
        }
    }

    dbgOut ( __func__ );

    return ( cnt );
}

/* check if the cert list contains a complete certificate chain */
BOOL checkCertListForCompleteDeviceCertChain (
    CERT_LIST_s *lst )
{
    BOOL failed = FALSE;
    CERT_LIST_s *devCrt = NULL;
    CERT_LIST_s *ptr = NULL;
    char *issuer;
    UINT32 size = 0, i = 0;


    dbgIn ( __func__ );

    size = cmpGetCertListSize( lst );

    failed = checkCertListForCertByLocalId (lst, cert_id_device, &devCrt );

    ifNotFailed {

        devCrt->devChain = TRUE;
        issuer = devCrt->issuerCn;

        while ( i < size ) {
            ptr = checkCertListForCertBySubjectCN( lst, issuer);
            if ( ptr ) {

                /* certificate of issuer found, its part of the device cert chain */
                ptr->devChain = TRUE;

                issuer = ptr->issuerCn;

                if ( ptr->certId == cert_id_root ) {
                    /* found the device root, chain is complete */
                    break;
                }
            } else {
                failed = TRUE;
                break;
            }
            i++;
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}

BOOL checkCertListForCompleteChain (
    CERT_LIST_s *lst,
    STATUS_s *stat,
    BOOL phony)
{
    BOOL failed = FALSE;
    CERT_LIST_s *ptr = lst;
    char *issuerName;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !lst ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        while ( ptr ) {
            issuerName = ptr->issuerCn;
            if ( checkCertListForCertBySubjectCN( lst, issuerName )) {
                /* certificate of issuer found */
                ptr->issuerAvailable = TRUE;
                ptr = ptr->next;
            } else {
                if ( phony ) {
                    dbgPrint ( DBG_INFO, "Could not find certificate 'subject CN=%s'.\n", issuerName );
                }
                if ( stat ) {
                    ecmCmSetStatus( stat, EXT_IF_ERR_CERT_NOT_PRESENT, issuerName, NULL );
                }
                failed = TRUE;
                break;
            }
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}

BOOL freeCertList (
    CERT_LIST_s **lst )
{
    BOOL failed = FALSE;
    CERT_LIST_s *next, *ptr;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !lst ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {

        ptr = *lst;

        while ( ptr != NULL ) {
            next = ptr->next;

            ptr->certSize = 0;
            escFreePtr ( (UINT8 **)&ptr->fileName );
            escFreePtr ( (UINT8 **)&ptr->title );
            escFreePtr ( (UINT8 **)&ptr->issuerCn );
            escFreePtr ( (UINT8 **)&ptr->subjectCn );
            escFreePtr ( (UINT8 **)&ptr->crlUri );
            escFreePtr ( (UINT8 **)&ptr->certPtr );
            asnFreeStruct( (void **) &ptr->crt, asn_DEF_Certificate );
            memset ( ptr, 0x0, sizeof ( CERT_LIST_s ));
            free ( ptr );
            ptr = next;
        }

        *lst = NULL;
        lst = NULL;
    }

    dbgOut ( __func__ );

    return ( failed );
}

BOOL cmpGetObjFromRDNSequence (
    RDNSequence_t *rdnSequence,
    OID_t targetOid,
    BOOL handleError,
    char **str )
{
    BOOL failed = FALSE;
    BOOL found = FALSE;
    int size, i;
    OID_t oid;
    AttributeType_t *type;
    AttributeValue_t *value;
    /* PrintableString and UTF8String are implemented as
       OCTET_STRING so we are safe using a OCTET_STRING_t
       as destination. */
    OCTET_STRING_t *os = NULL;
    asn_TYPE_descriptor_t *asnType = NULL;
    char *oidName = NULL;
    size_t s = 0;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !rdnSequence || !str ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        size = rdnSequence->list.count;

        /* search for common name */
        for ( i = 0; i < size; i++ ) {

            type = &rdnSequence->list.array[i]->list.array[0]->type;
            value = &rdnSequence->list.array[i]->list.array[0]->value;

            failed = getObjIdentifier( type, &oid );

            ifNotFailed {

                if ( oid == targetOid ) {

                    /* check how the object is implemented
                       (should be Printable string but CA gives UTF8String?) */

                    if ( *(value->buf) == 12 ) {

                        /* Its a UTF8String */
                        asnType = &asn_DEF_UTF8String;
                    } else if ( *(value->buf) == 19 ) {

                        /* Its a PrintableString */
                        asnType = &asn_DEF_PrintableString;
                    } else {

                        dbgPrint ( DBG_ERR,
                            DBG_FLS "ASN1 type not supported\n", DBG_FL );
                        failed = TRUE;
                    }

                    ifNotFailed {
                        failed = asn1Any2Type( value, asnType, (void **) &os );
                    }

                    ifNotFailed {
                        failed = escInt2SizetAndCheckForOv( os->size, &s );
                    }
                    ifNotFailed {

                        *str=calloc(1, s+1);
                        if ( NULL == *str ) {
                            failed = TRUE;
                        }
                    }
                    ifNotFailed {
                        memcpy(*str, os->buf, s );
                        found = TRUE;
                    }

                    if ( NULL != asnType ) {
                        asnFreeStruct( (void **)&os, *asnType );
                        asnType = NULL;
                    }

                    ifFailed {
                        break;
                    }
                }
            }

            if ( found == TRUE ) {
                break;
            }
        }
    }

    ifNotFailed {
        if ( found == FALSE ) {
            if ( handleError ) {
                asn1GetOidNameStr( targetOid, &oidName );
                dbgPrint ( DBG_ERR, DBG_FLS
                    "Could not find OID '%s'\n", DBG_FL, oidName );
                failed = TRUE;
            }
        }
    }

    escFreePtr( (UINT8 **) &oidName );

    dbgOut ( __func__ );

    return ( failed );
}

static BOOL cmpDetermineCrtType (
    CMP_DATA_s *cmp,
    CERT_LIST_s *crtPtr )
{
    BOOL failed = FALSE;
    Certificate_t *crt = NULL;
    UINT8 *pubKeyPtr = NULL;
    BOOL found = FALSE;

    dbgIn ( __func__ );

    // Sanity Checks
    if(!cmp || !crtPtr || !crtPtr->certPtr || !crtPtr->certSize
            || !cmp->pkcs1PubKey)
        return failed = TRUE;

    /* decode certificate first */
    failed = alloc( (void **) &crt, sizeof ( Certificate_t ));

    ifNotFailed {
        failed = asn1Decode(
            crtPtr->certPtr, crtPtr->certSize,
            &asn_DEF_Certificate, crt, FALSE );
    }

    ifNotFailed {

        /* search for our public key within SubjectPublickey structure */
        pubKeyPtr = crt->tbsCertificate.subjectPublicKeyInfo.subjectPublicKey.buf;

        if ( memcmp ( pubKeyPtr, cmp->pkcs1PubKey,
            crt->tbsCertificate.subjectPublicKeyInfo.subjectPublicKey.size ) == 0 )
        {
            /* this certificate contains our public key so its definitively the
                device certificate */
            crtPtr->certId = cert_id_device;
            found = TRUE;
        }

        /* Check for root certificate */
        if ( !found ) {

            if ( strncmp ( crtPtr->issuerCn, crtPtr->subjectCn,
                strlen ( crtPtr->subjectCn )) == 0 )
            {
                /* issuer matches subject -> this is the root */
                crtPtr->certId = cert_id_root;
                found = TRUE;
            }
        }

        /* Its not a root certificate and not the device,
           so we assume thats another end point certificate.
           It may however be also an issuer certificate,
           which we determine below */
        if ( !found ) {
            crtPtr->certId = cert_id_endpoint;
        }
    }

    asnFreeStruct( (void **) &crt, asn_DEF_Certificate );
    dbgOut ( __func__ );

    return ( failed );
}

BOOL cmpDetermineCrtTypesAndHierarchy2 (
    CERT_LIST_s *lst,
    UINT8* devPubKey,
    UINT32 devPubKeySize )
{
    BOOL failed = FALSE;
    CMP_DATA_s cmp;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !lst ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        memset ( &cmp, 0x0, sizeof ( CMP_DATA_s ));
        cmp.pkcs1PubKey = malloc ( devPubKeySize );
        if ( NULL == cmp.pkcs1PubKey ) {
            failed = TRUE;
        } else {
            memcpy ( cmp.pkcs1PubKey, devPubKey, devPubKeySize );
            cmp.pkcs1PubKeySize = devPubKeySize;
            cmp.certList = lst;

            failed = cmpDetermineCrtTypesAndHierarchy ( &cmp );

            escFreePtr( (UINT8 **) &cmp.pkcs1PubKey );
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}

BOOL cmpDetermineCrtTypesAndHierarchy (
    CMP_DATA_s *cmp )
{
    BOOL failed = FALSE;
    CERT_LIST_s *lst, *crtLstPtr, *res;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !cmp ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    /* determine certificate type */
    ifNotFailed {

        lst = cmp->certList;
        crtLstPtr = lst;

        while ( crtLstPtr && !failed ) {

            /* Determine type if not already done */
            if ( crtLstPtr->certId == cert_id_none ) {
                failed = cmpDetermineCrtType ( cmp, crtLstPtr );
            }

            crtLstPtr = crtLstPtr->next;
        }
    }

    /* Now determine the hierarchy of the whole list */
    ifNotFailed {

        crtLstPtr = cmp->certList;
        while ( crtLstPtr && !failed ) {

            switch ( crtLstPtr->certId ) {
                case ( cert_id_device):
                case ( cert_id_root):
                case ( cert_id_issuer):
                    /* nothing to do for those */
                    break;
                case ( cert_id_endpoint ):
                    /* check if its really an end point certificate,
                       means that there is no certificate in the chain
                       where this one is the issuer */
                    failed = cmpGetCrtFromListBySbjOrIssuerOid(
                        issuer, oid_commonName, crtLstPtr->subjectCn,
                        lst, &res, TRUE );
                    ifNotFailed {
                        /* found one, so this one is an issuer */
                        crtLstPtr->certId = cert_id_issuer;
                    } else {
                        /* reset failed flag to continue looping */
                        failed = FALSE;
                    }
                    break;
                default:
                    dbgPrint( DBG_ERR, DBG_FLS
                        "Invalid internal certificate ID (%d)\n",
                        DBG_FL, crtLstPtr->certId );
                    failed = TRUE;
                    break;
            }

            crtLstPtr = crtLstPtr->next;
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}

ull_time_t cmpExtractAsn1Time (
    Time_t asnTime )
{
    ull_time_t t = 0;

    switch(asnTime.present){
        case Time_PR_NOTHING:
            t = 0;
            break;

        case Time_PR_utcTime:
            t = asn_UT2time(&asnTime.choice.utcTime, NULL, 0);
            break;

        case Time_PR_generalTime:
            t = asn_GT2time(&asnTime.choice.generalTime, NULL, 0);
            break;
    }

    return ( t );
}

BOOL cmpExtractCrtParamsFromTBSCertificate (
    TBSCertificate_t *tbsCrt,
    CERT_LIST_s *crtEntry )
{
    BOOL failed = FALSE;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !tbsCrt || !crtEntry ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    /* extract validity */
    ifNotFailed {
        crtEntry->validNotBefore = cmpExtractAsn1Time( tbsCrt->validity.notBefore );
        crtEntry->validNotAfter = cmpExtractAsn1Time( tbsCrt->validity.notAfter );

        /* extract issuer and subject CN */
        failed = cmpGetObjFromRDNSequence(
            &tbsCrt->issuer.choice.rdnSequence,
            oid_commonName, TRUE, &crtEntry->issuerCn );
    }

    ifNotFailed {
        failed = cmpGetObjFromRDNSequence(
            &tbsCrt->subject.choice.rdnSequence,
            oid_commonName, TRUE, &crtEntry->subjectCn );
    }

    /* extract title */
    ifNotFailed {
        failed = cmpGetObjFromRDNSequence (
            &tbsCrt->subject.choice.rdnSequence,
            oid_title, FALSE, &crtEntry->title );

        ifFailed {
            /* handle missing title field - no error, keep going */
            dbgPrint ( DBG_INFO, "Subject title not set...\n");
            failed = FALSE;
        }
    }

    ifNotFailed {
        dbgPrint ( DBG_INFO, "Certificate 'subject CN=%s; T=%s' processed\n",
            (crtEntry->subjectCn) ? crtEntry->subjectCn : "n/a",
            (crtEntry->title) ? crtEntry->title : "n/a" );

        /* set flag to mark it for storing */
        crtEntry->store = TRUE;
    }

    dbgOut ( __func__ );

    return ( failed );
}

/* ToDo
    This function may be removed if we decide to handle certificates within
    the ASN1 structure internally.
    loadCerts() can call cmpExtractParamsFromTBSCertificate() directly
    in this case.
 */
BOOL cmpExtractCrtParameters (
    CERT_LIST_s *crtEntry )
{
    BOOL failed = FALSE;
    Certificate_t *crt = NULL;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !crtEntry ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        failed = alloc((void**) &crt, sizeof(Certificate_t));
    }

    /* decode certificate first */
    ifNotFailed {
        failed = asn1Decode( crtEntry->certPtr, crtEntry->certSize,
            &asn_DEF_Certificate, crt, FALSE );
    }

    ifNotFailed {
        failed =  cmpExtractCrtParamsFromTBSCertificate (
            &crt->tbsCertificate, crtEntry );
    }

    ifNotFailed{
        crtEntry->crlUri = getCRLURIfromCertificate(crt);
    }

    asnFreeStruct( (void **) &crt, asn_DEF_Certificate );

    dbgOut ( __func__ );

    return ( failed );
}

BOOL cmpVerifyCertSignatures (
    CERT_LIST_s *certList,
    STATUS_s *stat,
    BOOL stopOnError,
    BOOL quiet )
{
    BOOL failed = FALSE;
    CERT_LIST_s *subjCertPtr = NULL, *issuerCertPtr = NULL, *certListPtr = NULL;
    Certificate_t *subjCert = NULL, *issuerCert = NULL;

    // union for supported key sizes
    KeyPairT keyPairUnion;
    void* keyPair = &keyPairUnion;
    OID_t oid;
    HASH_t hashType;


    UINT8 *encCert = NULL;
    UINT32 encCertSize = MAX_CERT_SIZE;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !certList ) {
        dbgPrint ( DBG_ERR, DBG_FLS "No Certificates to verify\n", DBG_FL );
        return TRUE;
    }

    certListPtr = certList;

    /* check the signature of all certificates, one after another */
    ifNotFailed {

        while ( certListPtr ) {

            memset ( &keyPairUnion, 0x0, sizeof ( KeyPairT ));
            subjCertPtr = certListPtr;

            if ( FALSE == quiet ) {
                dbgPrint ( DBG_INFO,
                    "Verifying signature of certificate 'CN=%s' using "\
                    "certificate 'CN=%s'\n", subjCertPtr->subjectCn,
                    subjCertPtr->issuerCn );
            }

            /* get issuer certificate pointers */
            failed = cmpGetCrtFromListBySbjOrIssuerOid (
                subject, oid_commonName, subjCertPtr->issuerCn,
                certList, &issuerCertPtr, quiet );

            /* verify pointers */
            ifNotFailed {

                if ( !subjCertPtr || !issuerCertPtr ) {
                    dbgPrint ( DBG_ERR, DBG_FLS
                        "Device or CA certificate not available\n", DBG_FL );

                    failed = TRUE;
                }
            }

            ifNotFailed {

                failed = alloc ( (void *)&subjCert, sizeof(Certificate_t));
                failed |= alloc ( (void *)&issuerCert, sizeof(Certificate_t));
            }

            ifNotFailed {

                /* decode subject certificate */
                if ( FALSE == quiet ) {
                    dbgPrint ( DBG_INFO, "Decode subject certificate 'CN=%s'\n", subjCertPtr->subjectCn );
                }
                failed = asn1Decode ( subjCertPtr->certPtr,
                    subjCertPtr->certSize, &asn_DEF_Certificate,
                    subjCert, FALSE );
            }

            ifNotFailed {

                /* determine the signature algorithm */
                failed = getObjIdentifier(
                    &subjCert->signatureAlgorithm.algorithm, &oid );
            }

            ifNotFailed {
                switch ( oid ) {
                    case ( oid_shaWithRsaEnc ):
                        hashType = hash_sha1;
                        break;
                    case ( oid_sha256WithRsaEnc ):
                        hashType = hash_sha256;
                        break;
                    case ( oid_sha512WithRsaEnc ):
                        dbgPrint ( DBG_ERR, DBG_FLS
                            "RSA signatures using SHA 512 are not supported!\n",
                            DBG_FL );
                        break;
                    default:
                        dbgPrint ( DBG_ERR,
                            DBG_FLS "Unknown signature algorithm\n", DBG_FL );
                        failed = TRUE;
                        break;
                }
            }

            ifNotFailed {

                /* decode issuer certificate */
                if ( FALSE == quiet ) {
                    dbgPrint ( DBG_INFO, "Decode issuing certificate 'CN=%s'\n", issuerCertPtr->subjectCn );
                }
                failed = asn1Decode ( issuerCertPtr->certPtr,
                    issuerCertPtr->certSize, &asn_DEF_Certificate,
                    issuerCert, FALSE );
            }

            UINT32 signerKeySize = 0;

            ifNotFailed {

                /* Check the key size of the (Sub-)CA key and decode key.
                   Currently supported:
                   RSA 2048: 256Byte
                   RSA 4096: 512Byte
                   NOT SUPPORTED:
                   RSA 3072: 384Byte */
                UINT32 keySize = 0;
                /* convert BIT_STRING_t to OCTET_STRING to get the exact key size */
                failed = EscCrypto_GetPubKeyModSize (
                    issuerCert->tbsCertificate.\
                    subjectPublicKeyInfo.subjectPublicKey.buf,
                    issuerCert->tbsCertificate.\
                    subjectPublicKeyInfo.subjectPublicKey.size,
                    &keySize );

                signerKeySize = keySize;

                if ( FALSE == quiet ) {
                    dbgPrint ( DBG_INFO, "Decode %d bit issuer public key\n", signerKeySize*8 );
                }
                failed = EscCrypto_DecodePublicKey(
                        keyPair,
                    issuerCert->tbsCertificate.\
                    subjectPublicKeyInfo.subjectPublicKey.buf,
                    issuerCert->tbsCertificate.\
                    subjectPublicKeyInfo.subjectPublicKey.size,
                    signerKeySize*8);
            }

            ifNotFailed {

                /* re-encode device TBSCertificate part for verification */
                if ( FALSE == quiet ) {
                    dbgPrint ( DBG_INFO, "Re-encode subject certificate (parts) for verification\n");
                }

                failed = asn1Encode (
                    &subjCert->tbsCertificate, &asn_DEF_TBSCertificate,
                    &encCert, &encCertSize, FALSE );
            }

            //dbgPrintBlock( DBG_INFO, encCertSize, encCert );

            ifNotFailed {
                /* verify signature */
                failed = EscCrypto_VerifyMessage( keyPair, encCert,
                    encCertSize, subjCert->signature.buf, hashType, signerKeySize*8 );
            }

            ifFailed {
                dbgPrint ( DBG_ERR,
                    DBG_FLS "Signature verification of certificate 'CN=%s' failed\n", DBG_FL, subjCertPtr->subjectCn );
                ecmCmSetStatus( stat, EXT_IF_ERR_CERT_INVALID, subjCertPtr->subjectCn, NULL );
                subjCertPtr->err = EXT_IF_ERR_CERT_INVALID;
            } else {
                dbgPrint ( DBG_INFO, "Signature verification of certificate 'CN=%s' succeeded\n", subjCertPtr->subjectCn );
                subjCertPtr->validSignature = TRUE;
            }

            asnFreeStruct( (void **)&subjCert, asn_DEF_Certificate );
            asnFreeStruct( (void **)&issuerCert, asn_DEF_Certificate );
            escFreePtr( &encCert );

            certListPtr = certListPtr->next;

            if ( failed == TRUE && stopOnError == TRUE ) {
                break;
            }
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}

BOOL cmpGetCrtFromListBySbjOrIssuerOid (
    CRT_OID_SRC_e src,
    OID_t oid,
    char *ref,
    CERT_LIST_s *crtLst,
    CERT_LIST_s **crt,
    BOOL quiet )
{
    BOOL failed = FALSE;
    CERT_LIST_s *ptr = NULL;
    char *str;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !ref || !crtLst || !crt ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameters!\n", DBG_FL );
        failed = TRUE;
    }

    if ( FALSE == quiet ) {
        dbgPrint ( DBG_INFO, "Search for certificate with '%s' '%s=%s'\n",
            escId2StringLookup( oidSrcLut, oidSrcLutSize, src ),
            escId2StringLookup( oidLut, oidLutSize, oid ), ref );
    }

    ifNotFailed {
        *crt = NULL;
        ptr = crtLst;

        while ( ptr ) {

            switch ( src ) {
                case ( issuer ):

                    switch ( oid ) {
                        case ( oid_commonName ):
                            str = ptr->issuerCn;
                            break;
                        default:
                            dbgPrint ( DBG_ERR, DBG_FLS
                                "Invalid issuer OID requested\n", DBG_FL );
                            failed = TRUE;
                            break;
                    }
                    break;

                case (subject ):

                    switch ( oid ) {
                        case ( oid_commonName ):
                            str = ptr->subjectCn;
                            break;
                        case ( oid_title ):
                            str = ptr->title;
                            break;
                        default:
                            dbgPrint ( DBG_ERR, DBG_FLS
                                "Invalid subject OID requested\n", DBG_FL );
                            failed = TRUE;
                            break;
                    }
                    break;

                default:
                    dbgPrint ( DBG_ERR, DBG_FLS
                        "Invalid OID source requested\n", DBG_FL );
                    failed = TRUE;
                    break;
            }

            if ( failed ) {
                break;
            }

            if ( str ) {
                if ( strncasecmp( str, ref, strlen ( ref )) == 0 ) {
                    *crt = ptr;
                    break;
                }
            }

            ptr = ptr->next;
        }

        if ( !*crt ) {

            if ( !quiet ) {
                dbgPrint ( DBG_ERR,
                    DBG_FLS "Requested certificate not found\n", DBG_FL );
            }
            failed = TRUE;
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}

#if defined ( UNIT_TESTING )
BOOL cmpGetDeviceCertFromCertList (
    CERT_LIST_s *crtLst,
    CERT_LIST_s **devCrt )
{
    BOOL failed = FALSE;
    CERT_LIST_s *ptr = NULL;

    dbgIn ( __func__ );

    /* Sanity checks */
    if ( !crtLst || !devCrt ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {

        *devCrt = NULL;
        ptr = crtLst;
        while ( ptr ) {
            if ( ptr->certId == cert_id_device ) {
                *devCrt = ptr;
                break;
            }
            ptr = ptr->next;
        }

        if ( !*devCrt ) {
            dbgPrint ( DBG_ERR,
                DBG_FLS "Device certificate not found\n", DBG_FL );
            failed = TRUE;
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}


// The main function that checks if two given strings match. The first
// string may contain wildcard characters
BOOL match(char *wildcardIssuer, char *issuerCN)
{
    // If we reach the end of both strings, we are done
    if (*wildcardIssuer == '\0' && *issuerCN == '\0')
        return TRUE;

    // Make sure that the characters after '*' are present in second string.
    // This function assumes that the first string will not contain two
    // consecutive '*'
    if (*wildcardIssuer == '*' && *(wildcardIssuer+1) != '\0' && *issuerCN == '\0')
        return FALSE;

    // If the first string contains '?', or current characters of both
    // strings match
    if (*wildcardIssuer == '?' || *wildcardIssuer == *issuerCN)
        return match(wildcardIssuer+1, issuerCN+1);

    // If there is *, then there are two possibilities
    // a) We consider current character of second string
    // b) We ignore current character of second string.
    if (*wildcardIssuer == '*')
        return match(wildcardIssuer+1, issuerCN) || match(wildcardIssuer, issuerCN+1);
    return FALSE;
}

BOOL cmpCheckCert (
    CMP_DATA_s *cmpData,
    BOOL checkForRenew,
    BOOL checkForCertValidity)
{
    BOOL failed = FALSE;
    CERT_LIST_s *certList = NULL;
    CERT_LIST_s *devCrtPtr = NULL;
    BIT_STRING_t *bsPtr = NULL;
    ull_time_t validUntil = 0;
    ull_time_t currTime = 0;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !cmpData ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        certList = cmpData->certList;
        failed = cmpGetDeviceCertFromCertList( certList, &devCrtPtr );
    }

    /* de-serialize the certificate */
    ifNotFailed {

        if ( !devCrtPtr->crt ) {
            dbgPrint ( DBG_INFO, "Decode device certificate\n" );
            failed = alloc( (void **) &devCrtPtr->crt, sizeof ( Certificate_t ));
            ifNotFailed{
                failed = asn1Decode (
                    devCrtPtr->certPtr, devCrtPtr->certSize,
                    &asn_DEF_Certificate, devCrtPtr->crt, FALSE );
            }
        }
    }

    /* compare our device public key with the public key inside the
       device certificate */

    ifNotFailed {

        bsPtr = &devCrtPtr->crt->tbsCertificate.subjectPublicKeyInfo.subjectPublicKey;

        if ( memcmp ( cmpData->pkcs1PubKey, bsPtr->buf, bsPtr->size ) != 0 ) {

            failed = TRUE;
        }
    }

    /* check valid until */
    ifNotFailed {

        Time_t *t = NULL;
        t = &devCrtPtr->crt->tbsCertificate.validity.notAfter;

        switch ( t->present ) {

            case ( Time_PR_utcTime ):
                validUntil = asn_UT2time ( &t->choice.utcTime, NULL, 0 );
                break;

            case ( Time_PR_generalTime ):
                validUntil = asn_GT2time ( &t->choice.generalTime, NULL, 0 );
                break;

            default:
                dbgPrint ( DBG_ERR, DBG_FLS
                    "Certificate verification failed,"\
                    "unsupported time format\n", DBG_FL);
                failed = TRUE;
                break;
        }

        if ( (validUntil == INT_MAX) && (sizeof ( time_t ) == 4 )) {

            dbgPrint ( DBG_INFO,
                "ASN1 library time convert function returns "\
                "a value of INT_MAX and time_t is implemented "\
                "using a 4 byte type. "\
                "This means that validity is beyond 2038 and "\
                "therefore cannot be checked properly. "\
                "Skip validity check.\n" );
            checkForCertValidity = FALSE;
            checkForRenew = FALSE;
        }
    }

    ifNotFailed {

        if ( validUntil < 0 ) {
            dbgPrint ( DBG_ERR,
                "Certificate verification failed,"\
                "time conversion failed\n");
            failed = TRUE;
        }
    }

    ifNotFailed {

        currTime = time ( NULL );
        // check the certificate validity if checkForCertValidity is set to TRUE
        if ( checkForCertValidity == TRUE) {
            if ( currTime >= validUntil ) {
                dbgPrint ( DBG_ERR,
                    "Certificate verification failed, validity expired\n");
                failed = TRUE;
            }
        }

    }

    ifNotFailed {
        if ( checkForRenew == TRUE ) {
            if ( (currTime + cmpData->certRefreshThreshold) > validUntil ) {
                dbgPrint ( DBG_ERR, DBG_FLS
                    "Certificate verification failed, "\
                    "renew threshold reached\n", DBG_FL );
                failed = TRUE;
            }
        }
    }

    dbgPrint ( DBG_INFO, "Device certificate verification %s\n",
        (failed == FALSE) ? "success" : "failed" );

    dbgOut ( __func__ );

    return ( failed );
}

BOOL cmpVerifyIssuer (
    CMP_DATA_s *cmpData )
{
    BOOL failed = FALSE;
    CERT_LIST_s *certListPtr = NULL;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !cmpData ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    certListPtr = cmpData->certList;

    /* check the Issuer of all certificates, one after another */
    ifNotFailed {

        while ( certListPtr && !failed ) {
            failed = !match(cmpData->issuerCN, certListPtr->issuerCn);
            certListPtr = certListPtr->next;
        }
    }
    return failed;
}


#endif /* defined ( UNIT_TESTING ) */



BOOL freeCmpData (
    CMP_DATA_s *cmp )
{
    BOOL failed = FALSE;
    UINT8 *ptr;

    /* sanity checks */
    if ( !cmp ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameters!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {

        ptr = (UINT8 *)cmp->deviceUniqueId;
        escFreePtr ( &ptr );

        ptr = cmp->pkcs1PubKey;
        escFreePtr ( &ptr );

        ptr = (UINT8*) cmp->issuerCN;
        escFreePtr( &ptr );

        freeCertList ( &cmp->certList );

        escFreeStringList( &cmp->mandCrtTitles );
    }

    return (failed);
}

/***************************************************************************
 * 6. END                                                                  *
 ***************************************************************************/

