/***************************************************************************
 *                                                                         *
 * Copyright                                                               *
 *     escrypt GmbH, Bochum, Germany                                       *
 *     Lise-Meitner-Allee 4                                                *
 *     D-44801 Bochum, Germany                                             *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        asn1_utils.c

   \brief       ASN1 utilities module of CMP library

   $Rev: 812 $
 */
/***************************************************************************
$Author: mlange $
$Date: 2017-08-25 15:34:36 +0200 (Fr, 25. Aug 2017) $
****************************************************************************/

/***************************************************************************
* 1. INCLUDES                                                              *
****************************************************************************/

/* ASN.1 header */
#include "../asn1-2005-multi-file-os-asn1c/PKIMessage.h"

#include "../asn1-2005-multi-file-os-asn1c/PKIMessage.h"

/* common headers */
#include "../../ecm/inc/ecm.h"
#include "../../common/inc/esc_common.h"
#include "../../common/inc/esc_debug.h"
#include "../../common/inc/esc_utils.h"
#include "../../common/inc/esc_asn1_utils.h"

/* CMP header */
#include "../inc/asn1_utils.h"

#ifdef UNIT_TESTING
#include "ecmUnitTests.h"
#endif


/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/***************************************************************************
 * 3. DEFINITIONS                                                          *
 ***************************************************************************/

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/
/** OID RSA encryption */
const int oidRsaEnc[] =              { 1, 2, 840, 113549, 1, 1, 1 };
/** OID RSA encryption with SHA1 */
const int oidShaWithRsaEnc[] =       { 1, 2, 840, 113549, 1, 1, 5 };
/** OID RSA encryption with SHA256 */
const int oidSha256WithRsaEnc[] =    { 1, 2, 840, 113549, 1, 1, 11 };
/** OID token */
const int oidToken[] =               { 1, 3, 6, 1, 5, 5, 7, 5, 1, 1 };
/** OID common name */
const int oidCommonName[] =          { 2, 5, 4, 3 };
/** OID serial number */
const int oidSerialNumber[] =        { 2, 5, 4, 5 };
/** OID country name */
const int oidCountryName[] =         { 2, 5, 4, 6 };
/** OID locality name */
const int oidLocalityName[] =        { 2, 5, 4, 7 };
/** OID state or province name */
const int oidStateOrProvinceName[] = { 2, 5, 4, 8 };
/** OID organization */
const int oidOrganization[] =        { 2, 5, 4, 10 };
/** OID organization unit */
const int oidOrganizationUnit[] =    { 2, 5, 4, 11 };
/** OID title */
const int oidTitle[] =               { 2, 5, 4, 12 };
/** OID description */
const int oidDescription[] =         { 2, 5, 4, 13 };
/** OID user password */
const int oidUserPw[] =              { 2, 5, 4, 35 };
/** OID name */
const int oidName[] =                { 2, 5, 4, 41 };
/** OID unique ID */
const int oidUniqueId[] =            { 2, 5, 4, 45 };
/** OID distinguished name */
const int oidDistinguishedName[] =   { 2, 5, 4, 49 };
/** OID uuid */
const int oidUuid[] =                { 2, 5, 4, 77 };
/** OID printable string */
const int oidPrintableString[] =     { 1, 3, 6, 1, 4, 1, 1466, 115, 121, 1, 44 };
/** OID subject alternative name */
const int oidExtSubjectAltName[] =  { 2, 5, 29, 17 };

/** OID lookup table */
OIDS_t aOids[] = {
    { oid_rsaEnc,                     "1.2.840.113549.1.1.1",          "RsaEnc" },
    { oid_shaWithRsaEnc,              "1.2.840.113549.1.1.5",          "ShaWithRsaEnc" },
    { oid_sha256WithRsaEnc,           "1.2.840.113549.1.1.11",         "Sha256WithRsaEnc" },
    { oid_sha512WithRsaEnc,           "1.2.840.113549.1.1.13",         "Sha515WithRsaEnc" },
    { oid_printableString,            "1.3.6.1.4.1.1466.115.121.1.44", "PrintableSring" },
    { oid_commonName,                 "2.5.4.3",                       "CommonName" },
    { oid_serialNumber,               "2.5.4.5",                       "SerialNumber" },
    { oid_country,                    "2.5.4.6",                       "Country" },
    { oid_localityName,               "2.5.4.7",                       "Locality" },
    { oid_stateOrProvince,            "2.5.4.8",                       "StateOrProvince" },
    { oid_organization,               "2.5.4.10",                      "Organization" },
    { oid_organizationUnit,           "2.5.4.11",                      "OrganizationUnit" },
    { oid_title,                      "2.5.4.12",                      "Title" },
    { oid_description,                "2.5.4.13",                      "Description" },
	{ oid_ext_subjectAlternativeName, "2.5.29.17" },
};

/***************************************************************************
 * 5. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/
BOOL asn1GetOidNameStr (
    OID_t oid,
    char **str )
{
    BOOL failed = FALSE;
    int i;

    dbgIn ( __func__ );

    if ( !str ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {

        /* pre set variable */
        failed = TRUE;

        for ( i = 0; i < sizeof(aOids) / sizeof(OIDS_t); i++ ) {
            if ( aOids[i].type == oid ) {
                *str = calloc(1, strlen(aOids[i].nameStr)+1);
                if ( NULL != *str ) {
                    strncpy(*str, aOids[i].nameStr, strlen ( aOids[i].nameStr ));
                    failed = FALSE;
                } else {
                    failed = TRUE;
                    break;
                }
            }
        }
    }

    ifFailed {
        dbgPrint ( DBG_ERR, DBG_FLS
            "Could not find requested OID (no %d)\n", DBG_FL, oid );
    }

    dbgOut ( __func__ );

    return ( failed );
}

BOOL getObjIdentifier (
    OBJECT_IDENTIFIER_t *oid,
    OID_t *result )
{
    BOOL failed = FALSE;
    BOOL match = FALSE;

    int count, i, j;
    unsigned long fixed_arcs[10];
    unsigned long *arcs = fixed_arcs;
    size_t arc_type_size = sizeof ( fixed_arcs[0] );
    size_t slots = sizeof ( fixed_arcs ) / sizeof ( fixed_arcs[0] );
    unsigned long refFixedArcs[10];
    unsigned long *refArcs = refFixedArcs;
    size_t refSlots = sizeof ( refFixedArcs ) / sizeof ( refFixedArcs[0]);
    size_t refCount;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !oid || !result ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        /* pre set variable */
        failed = TRUE;

        count = OBJECT_IDENTIFIER_get_arcs( oid, arcs, arc_type_size, slots );

        if ( count > slots) {
            slots = count;
            arcs = malloc ( arc_type_size * slots );
            count = OBJECT_IDENTIFIER_get_arcs( oid, arcs, arc_type_size, slots );
        }

        /* compare with supported oids */
        for ( i = 0; i < sizeof(aOids) / sizeof(OIDS_t); i++ ) {

            /* convert reference OID string */
            refCount = OBJECT_IDENTIFIER_parse_arcs(
                aOids[i].noStr, strlen(aOids[i].noStr), (long *)refArcs, refSlots, NULL);
            if ( refCount > refSlots ) {
                refSlots = count;
                refArcs = malloc ( arc_type_size * refSlots );
                refCount = OBJECT_IDENTIFIER_parse_arcs(
                    aOids[i].noStr, strlen(aOids[i].noStr), (long *) refArcs, refSlots, NULL);
            }

            for ( j = 0; j < count; j++ ) {

                if ( *(arcs+j) == *(refArcs+j) ) {
                    match = TRUE;
                } else {
                    match = FALSE;
                    break;
                }
            }

            if ( match == TRUE ) {
                *result = aOids[i].type;
                failed = FALSE;
                break;
            }
        }
    }

    if ( arcs != fixed_arcs ) {
        free ( arcs );
    }

    if ( refArcs != refFixedArcs ) {
        free ( refArcs );
    }

    dbgOut ( __func__ );

    return ( failed );
}

/* ToDo: use OID-Array as getObjIdentifier! */
BOOL setObjIdentifier (
    OBJECT_IDENTIFIER_t *dst,
    OID_t id )
{
    BOOL failed = FALSE;
    int ret = 0;

    /* sanity checks */
    if ( !dst ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {

        switch ( id ) {

            case ( oid_printableString ):
                ret  = OBJECT_IDENTIFIER_set_arcs (
                    dst, oidPrintableString,
                    sizeof ( oidPrintableString[0]),
                    sizeof(oidPrintableString) /
                        sizeof ( oidPrintableString[0]));
                break;

            case ( oid_localityName ):
                ret  = OBJECT_IDENTIFIER_set_arcs (
                    dst, oidLocalityName,
                    sizeof ( oidLocalityName[0]),
                    sizeof(oidLocalityName) /
                        sizeof ( oidLocalityName[0]));
                break;

            case ( oid_stateOrProvince ):
                ret  = OBJECT_IDENTIFIER_set_arcs (
                    dst, oidStateOrProvinceName,
                    sizeof ( oidStateOrProvinceName[0]),
                    sizeof(oidStateOrProvinceName) /
                        sizeof ( oidStateOrProvinceName[0]));
                break;

            case ( oid_country ):
                ret  = OBJECT_IDENTIFIER_set_arcs (
                    dst, oidCountryName,
                    sizeof ( oidCountryName[0]),
                    sizeof(oidCountryName) /
                        sizeof ( oidCountryName[0]));
                break;

            case ( oid_serialNumber ):
                ret  = OBJECT_IDENTIFIER_set_arcs (
                    dst, oidSerialNumber,
                    sizeof ( oidSerialNumber[0]),
                    sizeof(oidSerialNumber) /
                        sizeof ( oidSerialNumber[0]));
                break;

            case ( oid_commonName ):
                ret  = OBJECT_IDENTIFIER_set_arcs (
                    dst, oidCommonName,
                    sizeof ( oidCommonName[0]),
                    sizeof(oidCommonName) /
                        sizeof ( oidCommonName[0]));
                break;

            case ( oid_distinguishedName ):
                ret  = OBJECT_IDENTIFIER_set_arcs (
                    dst, oidDistinguishedName,
                    sizeof ( oidDistinguishedName[0]),
                    sizeof(oidDistinguishedName) /
                        sizeof ( oidDistinguishedName[0]));
                break;

            case ( oid_name ):
                ret  = OBJECT_IDENTIFIER_set_arcs (
                    dst, oidName,
                    sizeof ( oidName[0]),
                    sizeof(oidName) / sizeof ( oidName[0]));
                break;

            case ( oid_organization ):
                ret  = OBJECT_IDENTIFIER_set_arcs (
                    dst, oidOrganization,
                    sizeof ( oidOrganization[0]),
                    sizeof(oidOrganization) /
                        sizeof ( oidOrganization[0]));
                break;

            case ( oid_organizationUnit ):
                ret  = OBJECT_IDENTIFIER_set_arcs (
                    dst, oidOrganizationUnit,
                    sizeof ( oidOrganizationUnit[0]),
                    sizeof(oidOrganizationUnit) /
                        sizeof ( oidOrganizationUnit[0]));
                break;

            case ( oid_title ):
                ret  = OBJECT_IDENTIFIER_set_arcs (
                    dst, oidTitle,
                    sizeof ( oidTitle[0]),
                    sizeof(oidTitle) /
                        sizeof ( oidTitle[0]));
                break;

            case ( oid_description ):
                ret  = OBJECT_IDENTIFIER_set_arcs (
                    dst, oidDescription,
                    sizeof ( oidDescription[0]),
                    sizeof(oidDescription) /
                        sizeof ( oidDescription[0]));
                break;

            case ( oid_regToken ):
                ret  = OBJECT_IDENTIFIER_set_arcs (
                    dst, oidToken, sizeof ( oidToken[0]),
                    sizeof(oidToken) / sizeof ( oidToken[0]));
                break;

            case ( oid_uniqueIdentifier ):
                ret  = OBJECT_IDENTIFIER_set_arcs (
                    dst, oidUniqueId, sizeof ( oidUniqueId[0]),
                    sizeof(oidUniqueId) / sizeof ( oidUniqueId[0]));
                break;

            case ( oid_userPw ):
                ret  = OBJECT_IDENTIFIER_set_arcs (
                    dst, oidUserPw, sizeof ( oidUserPw[0]),
                    sizeof(oidUserPw) / sizeof ( oidUserPw[0]));
                break;

            case ( oid_uuidpair ):
                ret  = OBJECT_IDENTIFIER_set_arcs (
                    dst, oidUuid, sizeof ( oidUuid[0]),
                    sizeof(oidUuid) / sizeof ( oidUuid[0]));
                break;

            case ( oid_rsaEnc ):
                ret  = OBJECT_IDENTIFIER_set_arcs (
                    dst, oidRsaEnc, sizeof ( oidRsaEnc[0]),
                    sizeof(oidRsaEnc) / sizeof ( oidRsaEnc[0]));

                break;

            case ( oid_shaWithRsaEnc ):
                ret  = OBJECT_IDENTIFIER_set_arcs (
                    dst, oidShaWithRsaEnc, sizeof ( oidShaWithRsaEnc[0]),
                    sizeof(oidShaWithRsaEnc) / sizeof ( oidShaWithRsaEnc[0]));

                break;

            case ( oid_sha256WithRsaEnc ):
                ret  = OBJECT_IDENTIFIER_set_arcs (
                    dst, oidSha256WithRsaEnc, sizeof ( oidSha256WithRsaEnc[0]),
                    sizeof(oidSha256WithRsaEnc) / sizeof ( oidSha256WithRsaEnc[0]));

                break;

            case ( oid_sha512WithRsaEnc ):

                /* not supported to set */
                ret = -1;
                break;

            case ( oid_ext_subjectAlternativeName ):
                ret  = OBJECT_IDENTIFIER_set_arcs (
                    dst, oidExtSubjectAltName, sizeof ( oidExtSubjectAltName[0]),
                    sizeof(oidExtSubjectAltName) / sizeof ( oidExtSubjectAltName[0]));

                      break;

            default:
                failed = TRUE;
                break;
        }
    }

    if ( ret != 0 ) {
        failed = TRUE;
    }

    return ( failed );
}
 
BOOL addToAttrList (
    void *list,
    OID_t id,
    char *str ) 
{
    BOOL failed = FALSE;
    int ret = 0;
    AttributeTypeAndValue_t *attr = NULL;
    OCTET_STRING_t octStr;
    BIT_STRING_t *subjUid;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !list || !str ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        failed = alloc ( (void *)&attr, sizeof(AttributeTypeAndValue_t ));
    }

    ifNotFailed {
        failed = setObjIdentifier ( &attr->type, id );
    }

    ifNotFailed {

        /* unique identifier is of type bit string */
        if ( id == oid_uniqueIdentifier ) {

            failed = alloc ( (void **) &subjUid, sizeof (BIT_STRING_t ));

            ifNotFailed {

                failed = alloc ((void**) &subjUid->buf, strlen (str));
            }

            ifNotFailed {
                memcpy ( subjUid->buf, str, strlen(str));
                subjUid->size = strlen ( str);
                subjUid->bits_unused = 0;

                failed = asn1Type2Any( subjUid, &asn_DEF_BIT_STRING, &attr->value );
            }

            if ( subjUid ) {
                ASN_STRUCT_FREE ( asn_DEF_BIT_STRING, subjUid );
            }

        /* other supported types are printable strings,
           mapped to octet string */
        } else {

            octStr.buf = (UINT8 *) str;
            octStr.size = strlen ( str );

            failed = asn1Type2Any( (void *)&octStr,
                    &asn_DEF_PrintableString, &attr->value );
        }
    }

    ifNotFailed {

        ret = ASN_SET_ADD ( list, attr );  

        if ( ret != 0 ) {
            failed = TRUE;
        }
    }

    ifFailed {
        if ( attr ) {
            ASN_STRUCT_FREE( asn_DEF_AttributeTypeAndValue, attr );
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}

BOOL addToRdnList (
    void *rdnSeq,
    OID_t id,
    char *str )
{
    BOOL failed = FALSE;
    int ret;
    RelativeDistinguishedName_t *rdn = NULL;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !rdnSeq || !str ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        failed = alloc ( (void **)&rdn, sizeof ( RelativeDistinguishedName_t ));
    }

    ifNotFailed {
        failed = addToAttrList ( rdn, id, str );
    }

    ifNotFailed {

        ret = ASN_SEQUENCE_ADD( 
            rdnSeq,
            rdn);

        if ( ret != 0 ) {
            failed = TRUE;
        }
    }

    ifFailed {
        if ( rdn ) {
            ASN_STRUCT_FREE( asn_DEF_RelativeDistinguishedName, rdn );
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}

BOOL asn1Any2Type (
    ANY_t *any,
    asn_TYPE_descriptor_t *td,
    void **type)
{
    BOOL failed = FALSE;
    int r;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !any || !td || !type ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        r = ANY_to_type ( any, td, type );
        if ( r < 0 ) {
            failed = TRUE;
            dbgPrint ( DBG_ERR,
                DBG_FLS "ASN1 type conversion failed\n", DBG_FL );
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}

BOOL asn1Type2Any (
    void *type,
    asn_TYPE_descriptor_t *td,
    ANY_t *any )
{
    BOOL failed = FALSE;
    int r;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !type || !td || !any ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        r = ANY_fromType(any, td, type );
        if ( r < 0 ) {
            failed = TRUE;
            dbgPrint ( DBG_ERR, DBG_FLS "ASN1 type conversion failed\n", DBG_FL );
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}

/***************************************************************************
 * 6. END                                                                  *
 ***************************************************************************/

