/******************************************************************
 *FILE: UserEncryptDecrypt_ConversionProcess.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UserEncryptDecrypt
 *COPYRIGHT: © 2018 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Guilherme Ferreira  (guilhermedaniel.ferreira@altran.com)
 * @date Jan, 2018
 */
#ifndef SRC_UPGRADE_USERENCRYPTDECRYPT_CONVERSIONPROCESS_H_
#define SRC_UPGRADE_USERENCRYPTDECRYPT_CONVERSIONPROCESS_H_

#include <map>
#include <memory>
#include <mutex>
#include <string>
#include <vector>

#include "upgrade/UserEncryptDecrypt_IConvert.h"

class UEDPacket;
class Notifier;
class Logger1;

class ConversionProcess {
 public:
  class Observer {
   public:
    virtual void onUpgrade() = 0;
  };

  /**
   * @brief Destructor for the class.
   */
  virtual ~ConversionProcess();

  /**
*@brief This method returns the unique instance of ConversionProcess class.
*
*@return The unique instance of this class.
*/
  static std::shared_ptr<ConversionProcess>& getInstance();

  /**
   * @brief Calling this function starts the data receiving part of the update
   * process.
   */
  void startData();

  /**
   * @brief Calling this function stops the data receiving part of the update
   * process.
       * When this function returns, it means that ALL of the configurations and
   * keys are configured to the new version.
   */
  void stopData();

  /**
   * @brief This function checks if the provided data requires conversion.
   *
   * @param data is the data that was generated by our lib that may require
   * upgrade.
   *
   * @return returns true if conversion is needed and false otherwise.
   */
  bool isConversionNeeded(std::vector<unsigned char> data);

  /**
   * @brief This function converts the provided data to the previously provided
   * destination version.
   *
   * @param data is the data that was generated by our lib that may require
   * upgrade.
   *
   * @return returns the upgraded data.
   */
  std::vector<unsigned char> convertData(std::vector<unsigned char> data);

  /**
   * @brief This function has the purpose of notifying the UserEncryptDecrypt
   * lib that an upgrade is starting.
   *
   * @param msB are the first 8 bits of the version to upgrade to.
   * @param middleByte are the middle 8 bits of the version to upgrade to.
   * @param lsB are the last 8 bits of the version to upgrade to.
   */
  void notifyUpgradeProcess(uint8_t msB, uint8_t middleByte, uint8_t lsB);

  std::vector<unsigned char> startRuntimeUpgradeProcess(
      std::vector<unsigned char> data);

  /**
*@brief This function has the purpose of adding an observer that will be
* notified when an upgrade is started.
*
*@param observer is the new user to notify.
*/
  void addObserver(Observer& observer);

  /**
*@brief This function has the purpose of removing an observer that will stop
* being notified when an upgrade is started.
*
*@param observer is the user to stop notifying.
*/
  void removeObserver(Observer& observer);

 private:
  ConversionProcess();
  void isPacketUsable(UEDPacket packet);
  static std::shared_ptr<ConversionProcess> _instance;
  std::shared_ptr<Notifier> _notifier;
  int _destinationVersion;
  bool _updating;
  static std::mutex _mutex;
  std::shared_ptr<Logger1> _logger;
  std::shared_ptr<LogContext> _ctx;

  std::map<std::string, IConvert*> _mapTypeDataConverter;
};

#endif /* SRC_UPGRADE_USERENCRYPTDECRYPT_CONVERSIONPROCESS_H_ */
