/******************************************************************
 *FILE: UserEncryptDecrypt_AuthenticationConvert.cpp
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UserEncryptDecrypt
 *COPYRIGHT: © 2018 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author José Oliveira  (josearmando.oliveira@altran.com)
 * @date Jan, 2018
 */

#include "UserEncryptDecrypt_AuthenticationConvert.h"
#include <core/UserEncryptDecrypt_ConsistentFile.h>
#include <core/authentication/UserEncryptDecrypt_Authentication.h>
#include <core/user/UserEncryptDecrypt_UserManager.h>

std::shared_ptr<AuthenticationConvert> AuthenticationConvert::_instance;

std::mutex AuthenticationConvert::_mutex;

std::vector<unsigned char> AuthenticationConvert::convertData(
    std::vector<unsigned char> data, int sourceVersion,
    int destinationVersion) {
  if (isConversionNeeded(sourceVersion, destinationVersion)) {
    return _mapAuthenticationDatabaseFunction[sourceVersion]
                                             [destinationVersion](data);
  } else {
    THROW_UED_EXCEPTION(ErrType::UPGRADE_NotInScope);
  }
}

AuthenticationConvert::AuthenticationConvert() : IConvert() {}

std::vector<unsigned char> AuthenticationConvert::v0v1(
    std::vector<unsigned char>) {
  auto convert = AuthenticationConvert::getConverter();
  Authentication::getInstance();
  std::shared_ptr<UserManager> userManager = UserManager::getInstance();
  std::shared_ptr<Configurations> conf     = Configurations::getInstance();
  char delimiter   = conf->get<char>(Configurations::DELIMITER);
  std::string wtry = std::to_string(UED_CURRENT_VERSION) + "\nWTRY";
  wtry += delimiter;
  wtry += std::to_string(0);
  wtry += "  \n";
  std::string toSave              = wtry;
  std::shared_ptr<SDCBackend> sdc = SDCBackend::getInstance();
  unsigned int keySize =
      conf->get<unsigned int>(Configurations::DERIVED_KEY_SIZE);
  unsigned int systemKeyID =
      conf->get<unsigned int>(Configurations::USER_ENCRYPT_DECRYPT_SIG_KEYID);
  int maxSize = 1024;
  for (unsigned int uid : userManager->getCreatedUsers()) {
    std::vector<unsigned char> encKeydata = userManager->getEncDataKey(uid);
    std::vector<unsigned char> keydata =
        sdc->Decrypt(userManager->getWrapperKeyID(uid), encKeydata,
                     SDCBackend::AES, SDCBackend::CBC);
    encKeydata = sdc->Encrypt(userManager->getWrapperKeyID(uid), keydata,
                              SDCBackend::RSA, SDCBackend::CBC);
    userManager->insertEncDataKey(uid, encKeydata);

    keydata = sdc->Decrypt(systemKeyID, userManager->getEncWrapperKey(uid),
                           SDCBackend::AES, SDCBackend::CBC);
    encKeydata =
        sdc->Encrypt(systemKeyID, keydata, SDCBackend::RSA, SDCBackend::CBC);

    std::string muk = "MUK" + createMUKv0v1(uid);
    if (muk.length() < maxSize) {
      std::string size(maxSize - muk.length(), ' ');
      muk.insert(muk.length(), size);
    }
    muk += "\n";
    toSave += muk;
  }
  convert->toSave = toSave;
  return std::vector<unsigned char>();
}

std::string AuthenticationConvert::createMUKv0v1(unsigned int uid) {
  std::shared_ptr<Configurations> conf = Configurations::getInstance();
  std::string delimiter                = "";
  delimiter += conf->get<char>(Configurations::DELIMITER);
  return UserManager::getInstance()->createMUK(uid) + delimiter +
         std::to_string(false) + delimiter;
}

void AuthenticationConvert::init() {
  convertionIsNeeded(UED_VERSION(0, 1, 0), UED_VERSION(1, 0, 0), true);
  convertionIsNeeded(UED_VERSION(1, 0, 0), UED_VERSION(0, 1, 0), true);

  _mapAuthenticationDatabaseFunction[UED_VERSION(0, 1, 0)]
                                    [UED_VERSION(1, 0, 0)] =
                                        std::bind(AuthenticationConvert::v0v1,
                                                  std::placeholders::_1);
  _mapAuthenticationDatabaseFunction[UED_VERSION(1, 0, 0)]
                                    [UED_VERSION(0, 1, 0)] =
                                        std::bind(AuthenticationConvert::v0v1,
                                                  std::placeholders::_1);
}

std::shared_ptr<AuthenticationConvert> AuthenticationConvert::getConverter() {
  if (!_instance) {
    std::lock_guard<std::mutex> lock(_mutex);
    if (!_instance) {
      _instance.reset(new AuthenticationConvert());
    }
  }
  return _instance;
}

void AuthenticationConvert::end() {
  auto convert = AuthenticationConvert::getConverter();
  if (convert->toSave != "") {
    std::shared_ptr<Configurations> conf = Configurations::getInstance();
    std::shared_ptr<ConsistentFile> _consistentFile = ConsistentFile::getFile(
        conf->get<std::string>(Configurations::UED_AUTH_FILE));
    _consistentFile->openFile();
    _consistentFile->write(convert->toSave);
  }
}
