/******************************************************************
 *FILE: UserEncryptDecrypt_UEDPacketDomain.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UEDPacketDomain
 *COPYRIGHT: © 2018 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Artur Bento  (artur.bento@altran.com)
 * @date Fev, 2018
 */

#ifndef USERENCRYPTDECRYPT_UEDPACKETDOMAIN_H
#define USERENCRYPTDECRYPT_UEDPACKETDOMAIN_H

#ifndef ROBS
#include <boost/property_tree/json_parser.hpp>
#include <boost/property_tree/ptree.hpp>
#else
#include <mock/RobsMock.h>
#endif
#include <string>
#include "UserEncryptDecrypt_IDomain.h"
#include "UserEncryptDecrypt_PacketStringDomain.h"
#include "UserEncryptDecrypt_Utils.h"
#include "UserEncryptDecrypt_Version.h"
#include "error/UserEncryptDecrypt_ErrorMessage.h"

namespace bptree = boost::property_tree;
/**
 * Class to ensure the robustness of UEDPacket in string format
 */
class UEDPacketDomain : public std::string {
 public:
  UEDPacketDomain(const std::string& string) : std::string(string){};
};

/**
 * This function validate if the Packet in string format is correct.
 * First the packet parts is verified, they can'b be less and more than tree.
 * The second validation is to each part, checking if each one is in base64
 * format.
 *
 * In case of string format of packet changes this function need to be modified
 * (eg. fields or structure).
 */
template <>
void isValidInDomain<UEDPacketDomain>(
    const UEDPacketDomain& uedpacketStr) throw(ErrorMessage) {
  isValidInDomain<PacketString>(uedpacketStr);

  int endPos               = uedpacketStr.find(".");
  std::string packetHeader = uedpacketStr.substr(0, endPos);

  std::vector<unsigned char> decHeader = base64_decode(packetHeader);
  std::stringstream ssHeader(std::string(decHeader.begin(), decHeader.end()));
  bptree::ptree ptHeader;
  try {
    bptree::read_json(ssHeader, ptHeader);
  } catch (boost::property_tree::json_parser_error&) {
    THROW_UED_EXCEPTION(ErrType::PCKT_InvalidFormat);
  }

  std::string libVersionValue;
  std::string headerVersionValue;
  try {
    headerVersionValue = ptHeader.get<std::string>("headerVersion");
  } catch (boost::property_tree::ptree_error& e) {
    THROW_UED_EXCEPTION(ErrType::PCKT_HeaderVersionNotFound);
  }

  try {
    libVersionValue = ptHeader.get<std::string>("libVersion");
  } catch (boost::property_tree::ptree_error& e) {
    THROW_UED_EXCEPTION(ErrType::PCKT_LibVersionNotFound);
  }

  if (headerVersionValue != std::to_string(UED_HEADER_CURRENT_VERSION) ||
      libVersionValue != std::to_string(UED_CURRENT_VERSION)) {
    THROW_UED_EXCEPTION(ErrType::VERSION_notCompatible);
  }

  std::string packetPayload = uedpacketStr.substr(endPos + 1);

  packetPayload = packetPayload.substr(0, packetPayload.find("."));

  std::vector<unsigned char> decPayload = base64_decode(packetPayload);
  std::stringstream ssPayload(
      std::string(decPayload.begin(), decPayload.end()));
  bptree::ptree ptPayload;
  try {
    bptree::read_json(ssPayload, ptPayload);
  } catch (boost::property_tree::json_parser_error&) {
    THROW_UED_EXCEPTION(ErrType::PCKT_InvalidFormat);
  }

  try {
    ptPayload.get<std::string>("payload");
  } catch (boost::property_tree::ptree_error& e) {
    THROW_UED_EXCEPTION(ErrType::PCKT_InvalidPayload);
  }
}
#endif  // USERENCRYPTDECRYPT_UEDPACKETDOMAIN_H
