/******************************************************************
 *FILE: UserEncryptDecrypt_User.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UserEncryptDecrypt
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/ /*
/**
* @author Guilherme Ferreira  (guilhermedaniel.ferreira@altran.com)
* @date Dec, 2017
*/

#ifndef SRC_CORE_USERENCRYPTDECRYPT_USER_H_
#define SRC_CORE_USERENCRYPTDECRYPT_USER_H_

#include <boost/optional/optional.hpp>
#include <string>
#include <vector>
#include "core/UserEncryptDecrypt_Configurations.h"

/**
 * This class describes an user of the UserEncryptDecrypt application.
 * Contains methods to handle all information about the user and its
 * permissions.
 */
class User {
 public:
  /**
   * @brief Constructor for the class.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   * @param hashPassSalt is the hash of the concatenated passphrase and salt.
   * @param wrapperKeyID is the ID of the key used to wrap the user data key.
   * @param encDataKey is the wrapped data key.
   * @param hashDataKey is the hash of the user data key.
   * @param hashWrapperKey is the hash of the user wrapper key.
   * @param salt is the random number used for the generation of the user hash.
   */
  User(unsigned int userID, std::vector<unsigned char> hashPassSalt,
       unsigned int wrapperKeyID, std::vector<unsigned char> encDataKey,
       std::vector<unsigned char> hashDataKey,
       std::vector<unsigned char> hashWrapperKey,
       std::vector<unsigned char> salt);

  /**
   * @brief Constructor for the class.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   * @param wrapperKeyID is the ID of the key used to wrap the user data key.
   * @param encDataKey is the wrapped data key.
   * @param hashDataKey is the hash of the user data key.
   * @param hashWrapperKey is the hash of the user wrapper key.
   */
  User(unsigned int userID, unsigned int wrapperKeyID,
       std::vector<unsigned char> encDataKey,
       std::vector<unsigned char> hashDataKey,
       std::vector<unsigned char> hashWrapperKey);

  /**
   * @brief Destructor for the class.
   */
  virtual ~User();

  /**
   * tells is the User user is able to encrypt/decrypt data.
   * @return
   */
  bool canEncrypt() { return true; }

  /**
   * @brief Provides the unique identifier of the user.
   */
  unsigned int getUserID();

  /**
   * @brief Provides the hash of the concatenated passphrase and salt.
   */
  boost::optional<std::vector<unsigned char>> getHashPassSalt();

  /**
   * @brief Provides the ID of the key used to wrap the user data key.
   */
  unsigned int &getWrapperKeyID();

  /**
   * @brief Provides the encrypted wrapped data key.
   */
  std::vector<unsigned char> &getEncDataKey();

  /**
   * @brief Provides the hash of the user data key.
   */
  std::vector<unsigned char> getHashDataKey();

  /**
   * @brief Provides the hash of the user wrapper key.
   */
  std::vector<unsigned char> &getHashWrapperKey();

  /**
   * @brief Provides the user's data key ID.
   */
  boost::optional<unsigned int> getDataKeyID();

  /**
   * @brief Provides the salt of the user.
   * @return salt of the user;
   */
  boost::optional<std::vector<unsigned char>> getSalt();

  /**
   * @brief Associates the given hash of the concatenated passphrase and salt to
   * the user.
   *
   * @param hashPassSalt is the hash of the concatenated passphrase and salt.
   */
  void setHashPassSalt(
      boost::optional<std::vector<unsigned char>> hashPassSalt);

  /**
   * @brief Associates the given wrapper key ID to the user.
   *
   * @param wrapperKeyID is the ID of the key used to wrap the user data key.
   */
  void setWrapperKeyID(unsigned int &wrapperKeyID);

  /**
   * @brief Associates the given encrypted data key to the user.

   * @param encDataKey is the wrapped data key.
   */
  void setEncDataKey(std::vector<unsigned char> &encDataKey);

  /**
   * @brief Associates the given wrapper key hash to the user.
   *
   * @param hashWrapperKey is the hash of the user wrapper key.
   */
  void setHashWrapperKey(std::vector<unsigned char> &hashWrapperKey);

  /**
   * @brief  Associates the given data key ID to the user.
   *
   * @param dataKeyID is the id of the user data key.
   */
  void setDataKeyID(boost::optional<unsigned int> dataKeyID);

  /**
   * @brief  Associates the given salt to the user.
   *
   * @param salt is the random number used for the generation of the user hash.
   */
  void setSalt(boost::optional<std::vector<unsigned char>> salt);

  /**
   * @brief Generates and returns the formatted data string with the users
   * associated data.
   */
  std::string createMUK();

  /**
   * @brief Provides the encryption wrapper key for the current user.
   *
   * @return returns the encrypted wrapped data key.
   */
  const std::vector<unsigned char> &getEncWrapperKey() const;

  /**
   * @brief Set encryption wrapper key for the current user.
   *
   * @param _encWrapperKey is the encrypted wrapped data key.
   */
  void set_encWrapperKey(const std::vector<unsigned char> &_encWrapperKey);

  /**
   * @brief Provides the old encryption wrapper key for the current user.
   *
   * @return returns the old encrypted wrapped data key.
   */
  const std::vector<unsigned char> &getOldEncWrapperKey() const;

  /**
   * @brief Set old encryption wrapper key for the current user.
   *
   * @param _oldWrapperKeyID is the old encrypted wrapped data key.
   */
  void setOldEncWrapperKey(std::vector<unsigned char> &_oldWrapperKeyID);

  /**
   * @brief Provides the old encryption data key for the current user.
   *
   * @return returns the old encrypted data key.
   */
  const std::vector<unsigned char> &getOldEncDataKey() const;

  /**
   * @brief Set old encryption data key for the current user.
   *
   * @param _oldEncDataKey is the old encrypted data key.
   */
  void setOldEncDataKey(const std::vector<unsigned char> &_oldEncDataKey);

 private:
  const unsigned int _userID;
  boost::optional<std::vector<unsigned char>> _hashPassSalt;
  unsigned int _wrapperKeyID;
  std::vector<unsigned char> _oldEncWrapperKey;
  std::vector<unsigned char> _encWrapperKey;

  std::vector<unsigned char> _encDataKey;
  std::vector<unsigned char> _oldEncDataKey;
  const std::vector<unsigned char> _hashDataKey;
  std::vector<unsigned char> _hashWrapperKey;
  boost::optional<unsigned int> _dataKeyID;
  boost::optional<std::vector<unsigned char>> _salt;
};

/**
 * This class inherits User functionalities.
 * MasterUser is an User with more permissions and can perform administration
 * operations.
 */
class MasterUser : public User {
 public:
  /**
   * @brief Constructor for the class.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   * @param hashPassSalt is the hash of the concatenated passphrase and salt.
   * @param wrapperKeyID is the ID of the key used to wrap the user data key.
   * @param encDataKey is the wrapped data key.
   * @param hashDataKey is the hash of the user data key.
   * @param hashWrapperKey is the hash of the user wrapper key.
   * @param salt is the random number used for the generation of the user hash.
   */
  MasterUser(unsigned int userID, std::vector<unsigned char> hashPassSalt,
             unsigned int wrapperKeyID, std::vector<unsigned char> encDataKey,
             std::vector<unsigned char> hashDataKey,
             std::vector<unsigned char> hashWrapperKey,
             std::vector<unsigned char> salt)
      : User(userID, hashPassSalt, wrapperKeyID, encDataKey, hashDataKey,
             hashWrapperKey, salt),
        _canEncrypt(Configurations::getInstance()->get<bool>(
            Configurations::Conf::ENC_DEC_MASTER)) {}

  /**
   * @brief Constructor for the class.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   * @param wrapperKeyID is the ID of the key used to wrap the user data key.
   * @param encDataKey is the wrapped data key.
   * @param hashDataKey is the hash of the user data key.
   * @param hashWrapperKey is the hash of the user wrapper key.
   */
  MasterUser(unsigned int userID, unsigned int wrapperKeyID,
             std::vector<unsigned char> encDataKey,
             std::vector<unsigned char> hashDataKey,
             std::vector<unsigned char> hashWrapperKey)
      : User(userID, wrapperKeyID, encDataKey, hashDataKey, hashWrapperKey),
        _canEncrypt(Configurations::getInstance()->get<bool>(
            Configurations::Conf::ENC_DEC_MASTER)) {}
  /**
   * @brief Destructor for the class.
   */
  virtual ~MasterUser() {}

 private:
  bool _canEncrypt;

 public:
  /**
   * tells is the Master user is able to encrypt/decrypt data.
   * @return
   */
  bool canEncrypt() const { return _canEncrypt; }
};

#endif /* SRC_CORE_USERENCRYPTDECRYPT_USER_H_ */
