/******************************************************************
 *FILE: UserEncryptDecrypt_UEDPacketManager.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UEDPacketManager
 *COPYRIGHT: © 2018 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Artur Bento  (artur.bento@altran.com)
 * @date Jan, 2018
 */

#ifndef USERENCRYPTDECRYPT_UEDPACKETMANAGER_H_
#define USERENCRYPTDECRYPT_UEDPACKETMANAGER_H_

#include <boost/optional.hpp>
#include <string>
#include "UserEncryptDecrypt_UEDPacket.h"
#include "core/token/UserEncryptDecrypt_Token.h"

/**
 * This is an static class that provides the management of all UEDPacket's in
 * the system.
 *
 * The Authentication and CoreFunctions classes uses this class to get the
 * manage Tokens and Packets.
 */
class UEDPacketManager {
 public:
  /**
   * @brief Creates and associates a Token for the user identified with userID.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   *
   *   @return returns a Packet with Token in payload and the encrypted
   * information of the issued user.
   */
  static std::string createUEDPacketToken(unsigned int userID);

  /**
   * @brief Receives a Token to decrypt and decode information to get the user
   * ID from that Token.
   *
   * @param packetString contains the token in payload
   *
   *   @return returns the user ID of the issued Token.
   */
  static boost::optional<unsigned int> getUserIDFromUEDPacketToken(
      std::string& packetString);

  /**
   * @brief Verify and validate the integrity of the issued Token. Verifies if
   * the Token was created by the system
   * and if it was adulterated.
   *
   * @param packetString contains the token in payload
   *
   *   @return returns True if the token is valid.
   *   return False if the Token was adulterated or the user is not valid.
   */
  static bool validateUEDPacketToken(std::string& packetString);

  /**
   * @brief Creates and associates a Packet for a given data.
   *
   * @param data is any information inside the A-IVI.
   *
   *   @return returns a Packet with data in payload.
   */
  static std::vector<unsigned char> createUEDPacket(
      std::vector<unsigned char>& data);

  /**
   * @brief Receives a Packet to decode information to get the data inside of
   * it.
   *
   * @param packetStr contains the data in payload
   *
   *   @return returns the data from the issued Packet.
   */
  static boost::optional<std::vector<unsigned char>> getDataFromUEDPacket(
      std::vector<unsigned char>& packet);

  /**
   * @brief Verify and validate the integrity of the issued Packet. Verifies if
   * the Packet was created by the system
   * and if it was adulterated.
   *
   * @param packetStr
   *
   *   @return returns True if the packet is valid.
   *   return False if the Packet was adulterated.
   */
  static bool validateUEDPacket(std::vector<unsigned char>& packet);

 private:
  /**
   * @brief Verify and validate the integrity of the issued Packet and returns
   * the data that it contains.
   *
   * @param packet is the issued Packet
   *
   *   @return returns the data inside the issued Packet.
   */
  static boost::optional<std::vector<unsigned char>>
  getDataFromUEDPacketPrivate(std::vector<unsigned char>& packet);

  /**
   * @brief Verify and validate the integrity of the issued Packet and returns
   * the Token object that it contains.
   *
   * @param packetString is the issued Packet with the Token object in payload
   *
   *   @return returns the Token inside the issued Packet.
   */
  static boost::optional<Token> getTokenObject(std::string& packetString);
};
#endif  // USERENCRYPTDECRYPT_UEDPACKETMANAGER_H_
