/******************************************************************
 *FILE: UserEncryptDecrypt_UEDPacket.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UEDPacket
 *COPYRIGHT: © 2018 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Artur Bento  (artur.bento@altran.com)
 * @date Jan, 2018
 */

#ifndef USERENCRYPTDECRYPT_UEDPACKET_H_
#define USERENCRYPTDECRYPT_UEDPACKET_H_

#include <boost/optional.hpp>
#ifndef ROBS
#include <boost/property_tree/ptree.hpp>
#else
#include <mock/RobsMock.h>
#endif
#include <string>
#include "core/packet/UserEncryptDecrypt_Packet.h"
#include "core/packet/UserEncryptDecrypt_PacketSignature.h"
#include "error/UserEncryptDecrypt_ErrorMessage.h"

/**
 * This enum class provides the data type of UEDPacket
 */
enum class UEDPacketDataType { TOKEN, CYPHERTEXT };

/**
 * This class is an inheritance of Packet that is JSON Web Token (JWT) based.
 * The information about entire Packet is inside the Header, content/message is
 * inside the Payload and to ensure integrity and security of the data signature
 * ensures that information isn't modified containing an hash.
 * The UEDPacket provides an easy way to communicate with UserEncryptDecrypt
 * application. To
 * build an UEDPacket it is necessary to provide the UEDPacket data and type to
 * send using the constructor to this effect. The function UEDPacket::encode
 * returns the string representation of UEDPacket and the UEDPacket::decode is
 * used to get the object of UEDPacket string.
 * UEDPacket provides two functions to get information of it,
 * UEDPacket::getHeaderField and UEDPacket::getPayload.
 * The first one requires a key that identifies an header attribute (attributes
 * can be found in constructor or read bellow).
 * The second retrieves the data inside the payload.
 *
 * The format of UEDPacket will be in the form of:
 *  - Header: contains some fields that describe what packet contains and
 * information about compatible versions.
 *            This header version contains the following characteristics:
 *              - headerVersion
 *              - libVersion
 *              - dataType
 *              - input (optional)
 *  - Payload: contains the data/message to send/receive (it can be encrypted)
 *  - Signature: contains an hash to verify if the Header and Payload are the
 * same as the ones presents on the Signature
 *
 */
class UEDPacket : private Packet {
 private:
  static std::map<UEDPacketDataType, std::string> _uedPacketDataTypesMap;
  static std::string _dataTypeStr;
  static std::string _headerVersionStr;
  static std::string _libVersionStr;

  std::string _payloadStr;
  boost::optional<UEDPacketDataType> _dataType;
  boost::optional<boost::property_tree::ptree> _optionalParameter;

  void buildUEDPacket();

 public:
  /**
   * @brief This constructor only can be used when a decode action is requested.
   * If an encode operation is requested and the header and payload are empty,
   * it will generate an invalid UEDPacket
   */
  UEDPacket();

  /**
   * @brief Builds an UEDPacket with header and payload
   *
   * @param dt Content type of payload
   * @param payload The content of the packet
   * @param pt Optional paramenter, it will be inserted in the header (input
   * field)
   */
  UEDPacket(UEDPacketDataType dt, std::string payload,
            boost::optional<boost::property_tree::ptree> pt = boost::none);
  /**
   * @brief Destructor for the class
   */
  ~UEDPacket();

  /**
   * @brief Verify if the UEDPacket is valid
   *   @return return True if the UEDPacket is valid, returns false otherwise.
   */
  bool isValid();

  std::string encode();

  /**
   * @brief Decodes and verify the authenticity of a UEDPacket from a given
   * base64
   * string.
   *
   * @param str base64 string from designated UEDPacket
   *
   *   @return returns True if the PacketSignature is valid, returns false
   * otherwise.
   */
  bool decode(std::string str);

  /**
   * @brief This method retrieves a field content inside the packet header.
   *
   * @param key Name of header field
   *
   * @return Returns the content of an header field
   */
  template <typename T>
  boost::optional<T> getHeaderField(std::string key) {
    return _header.getField<T>(key);
  }

  /**
   * @brief This method retrieves all data inside the packet payload, it can be
   * the any type, described when the functions is called
   *
   * @return Returns the content of payload
   */
  template <typename T>
  boost::optional<T> getPayload() {
    return _payload.getField<T>();
  }

  /**
   * @brief This method retrieves a string value of UEDPacketDataType enum.
   *
   * @param dataType Enum that describes the content of payload
   *
   * @return Returns a string with the value of dataType
   */
  static std::string getStringFromDataTypeEnum(UEDPacketDataType dataType);

  /**
   * @brief This method retrieves the UEDPacketDataType enum of a given string
   * value.
   *
   * @param dataTypeStr String that identifies the content of payload
   *
   * @return Returns an UEDPacketDataType enum if the dataTypeStr exists in the
   * enum, otherwise and boost::none is returned
   */
  static boost::optional<UEDPacketDataType> getEmumDataTypeFromString(
      std::string dataTypeStr);
#ifdef TESTING
  void print() {
    _header.print();
    _payload.print();
    PacketSignature p = PacketSignature(_header, _payload);
    p.encode();
    p.print();
  }
#endif
};
#endif  // USERENCRYPTDECRYPT_UEDPACKET_H_
