/******************************************************************
 *FILE: UserEncryptDecrypt_UEDPacket.cpp
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UEDPacket
 *COPYRIGHT: © 2018 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Artur Bento  (artur.bento@altran.com)
 * @date Jan, 2018
 */

#include "UserEncryptDecrypt_UEDPacket.h"
#include "UserEncryptDecrypt_Version.h"
#include "domains/UserEncryptDecrypt_UEDPacketDomain.h"

std::string UEDPacket::_dataTypeStr      = "dataType";
std::string UEDPacket::_headerVersionStr = "headerVersion";
std::string UEDPacket::_libVersionStr    = "libVersion";

std::map<UEDPacketDataType, std::string> UEDPacket::_uedPacketDataTypesMap = {
    {UEDPacketDataType::TOKEN, "TOKEN"},
    {UEDPacketDataType::CYPHERTEXT, "CYPHERTEXT"}};

UEDPacket::UEDPacket()
    : Packet(false), _dataType(boost::none), _optionalParameter(boost::none) {}

UEDPacket::UEDPacket(UEDPacketDataType updt, std::string payload,
                     boost::optional<boost::property_tree::ptree> pt)
    : Packet(false),
      _dataType(updt),
      _payloadStr(payload),
      _optionalParameter(pt) {}

UEDPacket::~UEDPacket() {}

void UEDPacket::buildUEDPacket() {
  _header.addField(_headerVersionStr, UED_HEADER_CURRENT_VERSION);
  _header.addField(_libVersionStr, UED_CURRENT_VERSION);

  if (_dataType != boost::none) {
    _header.addField(_dataTypeStr, getStringFromDataTypeEnum(_dataType.get()));
  }
  if (_optionalParameter != boost::none) {
    _header.addField("input", _optionalParameter.get());
  }
  _payload.addField("payload", _payloadStr);
}

std::string UEDPacket::encode() {
  buildUEDPacket();
  return Packet::encode();
}

bool UEDPacket::decode(std::string str) {
  isValidInDomain<UEDPacketDomain>(str);
  return Packet::decode(str);
}

std::string UEDPacket::getStringFromDataTypeEnum(UEDPacketDataType updt) {
  if (_uedPacketDataTypesMap.find(updt) != _uedPacketDataTypesMap.end()) {
    return _uedPacketDataTypesMap[updt];
  }

  return "UNKNOWN";
}

boost::optional<UEDPacketDataType> UEDPacket::getEmumDataTypeFromString(
    std::string updtStr) {
  for (auto str : _uedPacketDataTypesMap) {
    if (str.second == updtStr) {
      return str.first;
    }
  }

  return boost::none;
}

bool UEDPacket::isValid() {
  bool validPacket = Packet::isValid();
  boost::optional<std::string> dataTypeStr =
      _header.getField<std::string>(_dataTypeStr);
  boost::optional<std::string> headerVersionStr =
      _header.getField<std::string>(_headerVersionStr);
  boost::optional<std::string> libVersionStr =
      _header.getField<std::string>(_libVersionStr);

  if (dataTypeStr != boost::none) {
    boost::optional<UEDPacketDataType> updt =
        getEmumDataTypeFromString(dataTypeStr.get());

    return (validPacket && dataTypeStr != boost::none &&
            headerVersionStr != boost::none && libVersionStr != boost::none &&
            !dataTypeStr.get().empty() && !headerVersionStr.get().empty() &&
            !libVersionStr.get().empty() && (updt != boost::none));
  }

  return false;
}
