/******************************************************************
 *FILE: UserEncryptDecrypt_PacketSignature.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: PacketSignature
 *COPYRIGHT: © 2018 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Artur Bento  (artur.bento@altran.com)
 * @date Jan, 2018
 */

#ifndef USERENCRYPTDECRYPT_PACKETSIGNATURE_H
#define USERENCRYPTDECRYPT_PACKETSIGNATURE_H

#ifdef ROBS
#include <mock/RobsMock.h>
#else
#include <boost/property_tree/json_parser.hpp>
#include <boost/property_tree/ptree.hpp>
#endif

#include "UserEncryptDecrypt_PacketHeader.h"
#include "UserEncryptDecrypt_PacketPayload.h"
#include "backends/sdc/UserEncryptDecrypt_SDCBackend.h"
#include "error/UserEncryptDecrypt_ErrorMessage.h"

namespace bptree = boost::property_tree;

/**
 * This class provides security and ensures integrity of PacketHeader and
 * PacketPayload.
 * Signature is generated using encode function, that retrieves an hash of
 * concatenated PacketHeader and PacketPayload.
 * For that generated signature it can be verified if a given packet string is
 * provided using PacketSignature::verifySignature.
 * In case of adulterated packet, signature will ensure that packet is invalid,
 * because the given header and paylaod didn't generated the same hash of
 * current signature.
 */
class PacketSignature : public bptree::ptree {
 private:
  std::shared_ptr<SDCBackend> _sdcBackend;
  PacketPayload _payload;
  PacketHeader _header;
  void initSignatureKey();
  void generateSignature() throw(ErrorMessage);
  std::vector<unsigned char> digestHeaderData(PacketHeader ph);
  std::vector<unsigned char> digestPayloadData(PacketPayload pp);

 public:
  /**
   * @brief Default constructor for the class.
   * This constructor only should be used to decode an PacketStignature in
   * string format, using ::decode function;
   */
  PacketSignature();
  /**
   * @brief Constructor for the class.
   *
   * @param packetHeader An PacketHeader of packet to be hashed.
   * @param packetPayload An PacketPayload of packet to be hashed.
   */
  PacketSignature(PacketHeader packetHeader, PacketPayload packetPayload);
  /**
   * @brief Destructor for the class
   */
  ~PacketSignature();

  /**
   * @brief Encodes a PacketSignature in a base64 string.
   *
   *   @return returns a base64 string with the encoded information.
   */
  std::string encode() throw(ErrorMessage);

  /**
   * @brief Decodes a PacketSignature from a given base64 string.
   *
   * @param str base64 string from designated PacketSignature
   *
   */
  void decode(std::string) throw(ErrorMessage);

  /**
   * @brief Verifies if the HMAC of the PacketHeader and PacketPayload are the
   * same as the ones presents on the PacketSignature
   *
   * @param ph PacketHeader of the designated Packet.
   * @param pp PacketPayload of the designated Packet.
   *
   *  @return returns True if the the HMAC's are the same, returns false
   * otherwise.
   */
  bool verifySignature(PacketHeader packetHeader,
                       PacketPayload packetPayload) throw(ErrorMessage);

#ifdef TESTING
  void print() {
    std::ostringstream ss;
    bptree::ptree* pt = this;
    bptree::write_json(ss, *pt);
    std::cout << "Signature: " << ss.str() << std::endl;
  }
#endif
};
#endif  // USERENCRYPTDECRYPT_PACKETSIGNATURE_H
