/******************************************************************
 *FILE: UserEncryptDecrypt_PacketPayload.cpp
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: PacketPayload
 *COPYRIGHT: © 2018 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Artur Bento  (artur.bento@altran.com)
 * @date Jan, 2018
 */

#ifdef ROBS
#include <mock/RobsMock.h>
#else
#include <boost/property_tree/exceptions.hpp>
#endif

#include <core/UserEncryptDecrypt_Configurations.h>
#include <boost/current_function.hpp>
#include "UserEncryptDecrypt_PacketPayload.h"
#include "UserEncryptDecrypt_Utils.h"
#include "backends/UserEncryptDecrypt_BackendManager.h"
#include "backends/sdc/UserEncryptDecrypt_SDCBackend.h"
#include "domains/UserEncryptDecrypt_Base64Domain.h"

PacketPayload::PacketPayload() {}

PacketPayload::~PacketPayload() {}

std::string PacketPayload::encode() {
  std::string encoded = encodeB64();
  isValidInDomain<Base64>(encoded);

  std::shared_ptr<SDCBackend> sdcBackend =
      BackendManager::getInstance()->get<SDCBackend>();
  std::vector<unsigned char> encoded_data;
  try {
    encoded_data = std::vector<unsigned char>(encoded.begin(), encoded.end());
  } catch (std::bad_alloc&) {
    THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
  }

  unsigned int kid = Configurations::getInstance()->get<unsigned int>(
      Configurations::USER_ENCRYPT_DECRYPT_SIG_KEYID);

  std::vector<unsigned char> hash =
      BackendManager::getInstance()->get<SDCBackend>()->Encrypt(
          kid, encoded_data, SDCBackend::Algorithm::AES,
          SDCBackend::AlgorithmMode::CBC);

  std::string encodedPayloadString = base64_encode(hash.data(), hash.size());
  isValidInDomain<Base64>(encodedPayloadString);
  return encodedPayloadString;
}

void PacketPayload::decode(std::string str) throw(ErrorMessage) {
  isValidInDomain<Base64>(str);

  std::vector<unsigned char> vec = base64_decode(str);
  unsigned int kid = Configurations::getInstance()->get<unsigned int>(
      Configurations::USER_ENCRYPT_DECRYPT_SIG_KEYID);

  std::vector<unsigned char> vec1 =
      BackendManager::getInstance()->get<SDCBackend>()->Decrypt(
          kid, vec, SDCBackend::Algorithm::AES, SDCBackend::AlgorithmMode::CBC);

  if (vec1.size() == 0) {
    THROW_UED_EXCEPTION(ErrType::PCKT_InvalidPayload);
  }

  decodeB64(std::string(vec1.begin(), vec1.end()));
}

std::string PacketPayload::encodeB64() throw(ErrorMessage) {
  std::ostringstream ss;
  bptree::ptree* pt = this;

  try {
    bptree::write_json(ss, *pt);
  } catch (boost::property_tree::json_parser_error&) {
    THROW_UED_EXCEPTION(ErrType::PCKT_InvalidFormat);
  }
  std::string encodedPayloadString;
  std::string st = ss.str();

  try {
    std::vector<unsigned char> vec(st.begin(), st.end());
    encodedPayloadString = base64_encode(vec.data(), vec.size());
  } catch (std::bad_alloc&) {
    THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
  }

  isValidInDomain<Base64>(encodedPayloadString);
  return encodedPayloadString;
}

void PacketPayload::decodeB64(std::string str) throw(ErrorMessage) {
  isValidInDomain<Base64>(str);
  std::vector<unsigned char> vec = base64_decode(str);
  std::stringstream ss;
  try {
    ss = std::stringstream(std::string(vec.begin(), vec.end()));
  } catch (std::bad_alloc&) {
    THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
  }

  bptree::ptree pt;
  try {
    bptree::read_json(ss, pt);
    swap(pt);
  } catch (boost::property_tree::json_parser_error&) {
    THROW_UED_EXCEPTION(ErrType::PCKT_InvalidFormat);
  }
}

void PacketPayload::addField(std::string key, std::string value) {
  addFieldPrivate<std::string>(key, value);
}

void PacketPayload::addField(std::string key, int value) {
  addFieldPrivate<int>(key, value);
}

void PacketPayload::addField(std::string key, double value) {
  addFieldPrivate<double>(key, value);
}

void PacketPayload::addField(std::string key, unsigned int value) {
  addFieldPrivate<unsigned int>(key, value);
}

void PacketPayload::addField(std::map<std::string, std::string> fields) {
  for (auto pair : fields) {
    addFieldPrivate(pair.first, pair.second);
  }
}

void PacketPayload::addField(std::string key,
                             bptree::ptree value) throw(ErrorMessage) {
  try {
    put_child(key, value);
  } catch (boost::property_tree::ptree_error& e) {
    THROW_UED_EXCEPTION(ErrType::DOMAIN_notValid);
  }
}

bool PacketPayload::removeField(std::string key) {
  if (find(key) != not_found()) {
    erase(key);
    return true;
  }
  return false;
}

template <typename T>
void PacketPayload::addFieldPrivate(std::string key,
                                    T value) throw(ErrorMessage) {
  try {
    put(key, value);
  } catch (boost::property_tree::ptree_error& e) {
    THROW_UED_EXCEPTION(ErrType::DOMAIN_notValid);
  }
}
