/******************************************************************
 *FILE: UserEncryptDecrypt_PacketHeader.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: PacketHeader
 *COPYRIGHT: © 2018 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Artur Bento  (artur.bento@altran.com)
 * @date Jan, 2018
 */

#ifndef USERENCRYPTDECRYPT_PACKETHEADER_H_
#define USERENCRYPTDECRYPT_PACKETHEADER_H_

#ifdef ROBS
#include <mock/RobsMock.h>
#else
#include <boost/property_tree/json_parser.hpp>
#include <boost/property_tree/ptree.hpp>
#endif

#include <boost/optional.hpp>
#include <map>
#include <string>
#include "error/UserEncryptDecrypt_ErrorMessage.h"

namespace bptree = boost::property_tree;

/**
 * This class extends boost::property_tree::ptree
 * [http://www.boost.org/doc/libs/1_65_1/doc/html/property_tree.html]
 * and generates a PacketHeader used by Packet.\n
 * PacketHeader has the information about the data inside the PacketPayload.
 */
class PacketHeader : private bptree::ptree {
 public:
  /**
   * @brief Default constructor for the class.
   */
  PacketHeader();

  /**
   * @brief Destructor for the class
   */
  ~PacketHeader();

  /**
   * @brief Encodes a PacketHeader in a base64 string.
   *
   *   @return returns a base64 string with the encoded information.
   */
  std::string encode() throw(ErrorMessage);

  /**
   * @brief Decodes a PacketHeader from a given base64 string.
   *
   * @param str base64 string from designated PacketHeader
   *
   */
  void decode(std::string str) throw(ErrorMessage);

  /**
   * @brief Get a field from PacketHeader
   *
   * @param key is the name of field inside HeaderHeader
   *
   * @returns returns the value, the type is dynamic and assigned in runtime
   */
  template <typename T>
  boost::optional<T> getField(std::string key) {
    if (find(key) != not_found()) {
      return get<T>(key);
    }

    return boost::none;
  }

  /**
   * @brief add the value related with key inside the ptree inherited from boost
   * lib
   *
   * @param key is the identifier of value
   * @param value is the value that we want to store in the header, this can be
   * any type of TypesAllowed
   */
  void addField(std::string key, std::string value);
  void addField(std::string key, int value);
  void addField(std::string key, double value);
  void addField(std::string key, unsigned int value);
  /**
   * @brief add the value related with key inside the ptree inherited from boost
   * lib
   *
   * @param fields is a collection of values that we want to store in the header
   */
  void addField(std::map<std::string, std::string> fields);
  /**
   * @brief add the value related with key inside the ptree inherited from boost
   * lib
   *
   * @param key is the identifier of value
   * @param fields is a ptree and will be a child of ptree
   */
  void addField(std::string key, bptree::ptree fields) throw(ErrorMessage);

  /**
   * @brief remove the value and key inside the ptree inherited from boost lib
   *
   * @param key is the identifier to remove from ptree. If it exists True is
   * returned, otherwise False is returned
   */
  bool removeField(std::string key);

#ifdef TESTING
  void print() {
    std::ostringstream ss;
    bptree::ptree* pt = this;
    bptree::write_json(ss, *pt);
    std::cout << "Header: " << ss.str() << std::endl;
  }
#endif
 private:
  template <typename T>
  void addFieldPrivate(std::string key, T value) throw(ErrorMessage);
};
#endif  // USERENCRYPTDECRYPT_PACKETHEADER_H_
