/******************************************************************
 *FILE: UserEncryptDecrypt_PacketHeader.cpp
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: PacketHeader
 *COPYRIGHT: © 2018 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Artur Bento  (artur.bento@altran.com)
 * @date Jan, 2018
 */

#include "UserEncryptDecrypt_PacketHeader.h"
#include <core/UserEncryptDecrypt_Configurations.h>
#include "UserEncryptDecrypt_Utils.h"
#include "backends/UserEncryptDecrypt_BackendManager.h"
#include "backends/sdc/UserEncryptDecrypt_SDCBackend.h"
#include "domains/UserEncryptDecrypt_Base64Domain.h"

#ifdef ROBS
#include <mock/RobsMock.h>
#else
#include <boost/property_tree/exceptions.hpp>
#endif

PacketHeader::PacketHeader() {}

PacketHeader::~PacketHeader() {}

std::string PacketHeader::encode() throw(ErrorMessage) {
  std::ostringstream ss;
  bptree::ptree* pt = this;
  try {
    bptree::write_json(ss, *pt);
  } catch (boost::property_tree::json_parser_error&) {
    THROW_UED_EXCEPTION(ErrType::PCKT_InvalidFormat);
  }
  std::string encodedHeaderStr =
      base64_encode((unsigned char*)ss.str().data(), ss.str().length());

  isValidInDomain<Base64>(encodedHeaderStr);
  return encodedHeaderStr;
}

void PacketHeader::decode(std::string str) throw(ErrorMessage) {
  isValidInDomain<Base64>(str);

  std::vector<unsigned char> dec = base64_decode(str);
  std::stringstream ss(std::string(dec.begin(), dec.end()));
  bptree::ptree pt;
  try {
    bptree::read_json(ss, pt);
    swap(pt);
  } catch (boost::property_tree::json_parser_error&) {
    THROW_UED_EXCEPTION(ErrType::PCKT_InvalidFormat);
  }
}

void PacketHeader::addField(std::string key, std::string value) {
  addFieldPrivate<std::string>(key, value);
}

void PacketHeader::addField(std::string key, int value) {
  addFieldPrivate<int>(key, value);
}

void PacketHeader::addField(std::string key, double value) {
  addFieldPrivate<double>(key, value);
}

void PacketHeader::addField(std::string key, unsigned int value) {
  addFieldPrivate<unsigned int>(key, value);
}

void PacketHeader::addField(std::map<std::string, std::string> fields) {
  for (auto pair : fields) {
    addFieldPrivate(pair.first, pair.second);
  }
}

void PacketHeader::addField(std::string key,
                            bptree::ptree value) throw(ErrorMessage) {
  try {
    put_child(key, value);
  } catch (boost::property_tree::ptree_error& e) {
    THROW_UED_EXCEPTION(ErrType::DOMAIN_notValid);
  }
}

bool PacketHeader::removeField(std::string key) {
  if (find(key) != not_found()) {
    erase(key);
    return true;
  }
  return false;
}

template <typename T>
void PacketHeader::addFieldPrivate(std::string key,
                                   T value) throw(ErrorMessage) {
  try {
    put(key, value);
  } catch (boost::property_tree::ptree_error& e) {
    THROW_UED_EXCEPTION(ErrType::DOMAIN_notValid);
  }
}
