/******************************************************************
 *FILE: UserEncryptDecrypt_Logger.cpp
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: Logger interface for UserEncryptDecrypt using DLT
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Ulisses Costa (marioulisses.costa@altran.com)
 * @date Dec, 2017
 *
 * Date      | Author             | Modification
 * 14/08/2018| AKM7COB			  | changed class name logger to logger1. class name logger is already used in ASF framework
 */

#include "core/logger/UserEncryptDecrypt_Logger.h"
#include <dlt.h>
#include <error/UserEncryptDecrypt_ErrorMessage.h>

std::mutex Logger1::_mutex;
std::mutex Logger1::_logActionLock;
std::shared_ptr<Logger1> Logger1::_instance = nullptr;

Logger1::Logger1() : _loggerOut(LoggerOutput::STDOUT) {}

Logger1::~Logger1() {
  for (int i = _contexts.size(); i > 0; i--) {
    unregisterContext(_contexts.back());
  }
  unregisterApplication();
}

std::shared_ptr<Logger1>& Logger1::getLogger() {
  if (!_instance) {
    std::lock_guard<std::mutex> lock(_mutex);
    if (!_instance) {
      try {
        _instance.reset(new Logger1());
      } catch (std::bad_alloc&) {
        THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
      }
    }
  }
  return _instance;
}

void Logger1::registerApplication(std::string appID,
                                 std::string appDescription) {
  _appID = appID;

  if (_loggerOut == LoggerOutput::DLT) {
    DLT_REGISTER_APP(appID.c_str(), appDescription.c_str());
  } else {
    std::cout << _appID << " context registered." << std::endl;
  }
}

void Logger1::unregisterApplication() {
  if (_loggerOut == LoggerOutput::DLT) {
    DLT_UNREGISTER_APP();
  } else {
    std::cout << _appID << " context unregistered." << std::endl;
  }
}

std::shared_ptr<LogContext> Logger1::registerContext(
    const std::string& contextID, const std::string& contextDescription) {
  std::lock_guard<std::mutex> lock(_logActionLock);
  std::shared_ptr<LogContext> context;

  for (auto context : _contexts) {
    if (context->getContextId() == contextID) {
      return context;
    }
  }

  if (_loggerOut == LoggerOutput::DLT) {
    DLT_DECLARE_CONTEXT(dltContext);
    DLT_REGISTER_CONTEXT(dltContext, contextID.c_str(),
                         contextDescription.c_str());

    context =
        std::make_shared<LogContext>(contextID, contextDescription, dltContext);
  } else {
    context = std::make_shared<LogContext>(contextID, contextDescription);
  }
  _contexts.push_back(context);
  return context;
}

void Logger1::unregisterContext(const std::string& contextID) {
  std::lock_guard<std::mutex> lock(_logActionLock);

  for (auto it = _contexts.begin(); it != _contexts.end(); ++it) {
    auto context = *it;

    if (context->getContextId() == contextID) {
      if (_loggerOut == LoggerOutput::DLT) {
        DltContext ctx = context->getDltContext();
        DLT_UNREGISTER_CONTEXT(ctx);
      }
      _contexts.erase(it);
      return;
    }
  }
}

void Logger1::unregisterContext(std::shared_ptr<LogContext> ctx) {
  unregisterContext(ctx->getContextId());
}
