/******************************************************************
 *FILE: UserEncryptDecrypt_LogContext.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: Logger interface for UserEncryptDecrypt using DLT
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Ulisses Costa (marioulisses.costa@altran.com)
 * @date Dec, 2017
 *
 * Date      | Author             | Modification
 * 14/08/2018| AKM7COB			  | changed class name logger to logger1. class name logger is already used in ASF framework
 */

#ifndef SRC_CORE_LOGGER_USERENCRYPTDECRYPT_LOGCONTEXT_H_
#define SRC_CORE_LOGGER_USERENCRYPTDECRYPT_LOGCONTEXT_H_

#include <dlt.h>

/**
 * This class has important information for the logging system.
 * The context allow tracking actions of an application user through the logs.
 * It provides an identifier and a description of the context where the messages
 * are logged.
 * This class permits an easy way to search for logged messages in the DLT
 * Viewer application that
 * provides filters to find logged messages by contexts or other relevant
 * parameters.
 *
 * To register a LogContext in the DLT Viewer application is necessary to use
 * the Logger object that allow us to register the intended context.
 * The next example show how you can register a LogContext to identify log
 * messages:
 * std::shared_ptr<Logger> logger = Logger1::getLogger();
 * std::shared_ptr<LogContext> logContext = logger->registerContext("LOG",
 * "UserEncryptDecrypt_LogContext");
 * Each time a message is logged using this logContext it will be easily
 * identified by the contextID 'LOG'
 */
class LogContext {
 public:
  /**
   * @brief Default constructor for the class.
   */
  LogContext() {}

  /**
   * @brief Constructor of LogContext creates a new DLT context
   *
   * */
  LogContext(const std::string& contextID,
             const std::string& contextDescription, DltContext ctx)
      : _contextID(contextID),
        _contextDescription(contextDescription),
        _dltContext(ctx) {}

  /**
   * @brief Constructor of LogContext creates a new STDOUT context
   *
   * */
  LogContext(const std::string& contextID,
             const std::string& contextDescription)
      : _contextID(contextID), _contextDescription(contextDescription) {}

  /**
   * @brief Destructor for the class
   */
  ~LogContext() {}

  /**
   * @brief Access the context unique identifier
   *
   * @return The unique string identifier of the context
   *
   * */
  std::string getContextId() const { return _contextID; }

  /**
   * @brief Access the context description
   *
   * @return The string description of the context
   *
   * */
  std::string getContextDescription() const { return _contextDescription; }

  /**
   * @brief Access the DLT context
   *
   * @return The DLT context
   *
   * */
  DltContext getDltContext() const { return _dltContext; }

 private:
  std::string _contextID;
  std::string _contextDescription;
  DltContext _dltContext;
};

#endif /* SRC_CORE_LOGGER_USERENCRYPTDECRYPT_LOGCONTEXT_H_ */
