/******************************************************************
 *FILE: UserEncryptDecrypt_SaltDatabase.cpp
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UserEncryptDecrypt
 *COPYRIGHT: © 2018 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author José Oliveira  (josearmando.oliveira@altran.com)
 * @date Jan, 2018
 */

#include "UserEncryptDecrypt_SaltDatabase.h"
#include "UserEncryptDecrypt_Utils.h"
#include "core/UserEncryptDecrypt_Configurations.h"
#include "domains/UserEncryptDecrypt_SaltDomain.h"

std::shared_ptr<SaltDatabase> SaltDatabase::_instance;
std::mutex SaltDatabase::_mutex;

SaltDatabase::SaltDatabase()
    : _file(ConsistentFile::getFile(
          Configurations::getInstance()->get<std::string>(
              Configurations::SALT_FILE))) {}

SaltDatabase::~SaltDatabase() {}
#ifdef TESTING
std::shared_ptr<SaltDatabase> SaltDatabase::getInstance(bool reload) {
  if (!_instance || reload) {
    std::lock_guard<std::mutex> lock(_mutex);
    if (!_instance || reload) {
      try {
        _instance.reset(new SaltDatabase());
      } catch (std::bad_alloc &) {
        THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
      }
      _instance->_file->openFile(true);
    }
  }
  return _instance;
}
#else
std::shared_ptr<SaltDatabase> SaltDatabase::getInstance() {
  if (!_instance) {
    std::lock_guard<std::mutex> lock(_mutex);
    if (!_instance) {
      try {
        _instance.reset(new SaltDatabase());
      } catch (std::bad_alloc &) {
        THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
      }
      _instance->_file->openFile();
    }
  }
  return _instance;
}
#endif

bool SaltDatabase::insert(std::vector<unsigned char> salt) {
  _file->append(base64_encode(salt.data(), salt.size()) + "\n");
  return true;
}

bool SaltDatabase::exists(std::vector<unsigned char> salt) {
  isValidInDomain<Salt>(salt);
  _file->openFile();
  std::string Initialsalt = base64_encode(salt.data(), salt.size());
  std::string Filesalt    = "";
  do {
    Filesalt                      = _file->readLine();
    Filesalt[Filesalt.size() - 1] = '\0';
    if (strcmp(Filesalt.data(), Initialsalt.data()) == 0) {
      return true;
    }
  } while (!Filesalt.empty());
  return false;
}

bool SaltDatabase::insertIfNotExists(std::vector<unsigned char> salt) {
  if (!exists(salt)) {
    insert(salt);
    return true;
  }
  return false;
}
