/******************************************************************
 *FILE: UserEncryptDecrypt_AuthenticationDatabase.cpp
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UserEncryptDecrypt
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/ /*
/**
* @author Guilherme Ferreira  (guilhermedaniel.ferreira@altran.com)
* @date Dec, 2017
*/

#include <fstream>
#include <iostream>
#include <list>
#include <list>
#include <map>
#include <memory>
#include <mutex>
#include <string>

#include "UserEncryptDecrypt_Utils.h"
#include "UserEncryptDecrypt_Version.h"
#include "backends/sdc/UserEncryptDecrypt_SDCBackend.h"
#include "backends/sdc/UserEncryptDecrypt_SDCBackend.h"
#include "core/UserEncryptDecrypt_Configurations.h"
#include "core/authentication/UserEncryptDecrypt_AuthenticationDatabase.h"
#include "core/user/UserEncryptDecrypt_UserManager.h"
#include "domains/UserEncryptDecrypt_UserIDDomain.h"

std::mutex AuthenticationDatabase::_mutex;
std::shared_ptr<AuthenticationDatabase> AuthenticationDatabase::_instance =
    nullptr;
int AuthenticationDatabase::_wrongTries;

AuthenticationDatabase::AuthenticationDatabase()
    : _consistentFile(ConsistentFile::getFile(
          Configurations::getInstance()->get<std::string>(
              Configurations::Conf::UED_AUTH_FILE))) {
  initKey();
  _userManager = UserManager::getInstance();
  _delimiter =
      Configurations::getInstance()->get<char>(Configurations::Conf::DELIMITER);
  _wrongTries = 0;
  _logger     = Logger1::getLogger();
  _ctx        = _logger->registerContext("AUTHDB",
                                  "UserEncryptDecrypt_AuthenticationDatabase");
}

AuthenticationDatabase::~AuthenticationDatabase() {}

std::shared_ptr<AuthenticationDatabase> &AuthenticationDatabase::getInstance(
    bool createNew) {
  if (createNew || !_instance) {
    std::lock_guard<std::mutex> lock(_mutex);
    if (createNew || !_instance) {
      try {
        _instance.reset(new AuthenticationDatabase());
      } catch (std::bad_alloc &) {
        THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
      }
    }
  }
  return _instance;
}

std::shared_ptr<AuthenticationDatabase> &AuthenticationDatabase::getInstance() {
  if (!_instance) {
    std::lock_guard<std::mutex> lock(_mutex);
    if (!_instance) {
      try {
        _instance.reset(new AuthenticationDatabase());
      } catch (std::bad_alloc &) {
        THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
      }
    }
  }
  return _instance;
}

void AuthenticationDatabase::createFile() {
  _consistentFile->openFile();
  std::string textToWrite = std::to_string(UED_HEADER_CURRENT_VERSION) + "\n";
  textToWrite += "WTRY";
  textToWrite += _delimiter;
  textToWrite += std::to_string(0);
  textToWrite += "  \n";
  _consistentFile->write(textToWrite, 0);
  _consistentFile->closeFile();
}

int AuthenticationDatabase::checkKey(std::string line) {
  if (line.empty()) {
    return 0;
  }
  std::string data;
  data = line.substr(0, line.find(_delimiter));
  if (!data.compare("MUK")) {
    return 1;
  }
  if (!data.compare("WTRY")) {
    return 2;
  }
  return 0;
}

std::string AuthenticationDatabase::createMUK(unsigned int userID) throw(
    ErrorMessage) {
  isValidInDomain<UserID>(userID);

  int maxSize = 1024;

  std::string textToWrite = "MUK";
  textToWrite += _userManager->createMUK(userID);
  if (textToWrite.length() < maxSize) {
    std::string size(maxSize - textToWrite.length(), ' ');
    textToWrite.insert(textToWrite.length(), size);
  }
  textToWrite += "\n";

  return textToWrite;
}

void AuthenticationDatabase::loadFromFile() throw(ErrorMessage) {
  unsigned int userID;
  unsigned int wrapperKeyID;
  std::string data;
  std::string line;
  std::vector<unsigned char> hashPassSalt;
  std::vector<unsigned char> encDataKey;
  std::vector<unsigned char> hashDataKey;
  std::vector<unsigned char> hashWrapperKey;
  bool master;
  std::vector<unsigned char> salt;
  unsigned int version;

  checkFile();
  line = _consistentFile->readLine();
  if (line.empty()) {
    _consistentFile->closeFile();
    createFile();
    _consistentFile->openFile();
    line = _consistentFile->readLine();
  }
  try {
    version = std::stoul(line);
  } catch (std::exception &) {
    THROW_UED_EXCEPTION(ErrType::AUTHDB_fileDataCorrupted);
  }
  if (version < UED_HEADER_CURRENT_VERSION) {
    THROW_UED_EXCEPTION(ErrType::VERSION_notCompatible);
  }
  do {
    line = _consistentFile->readLine();
    if (checkKey(line) == 1) {
      line.erase(0, getNextDelimiter(line) + 1);

      data = line.substr(0, getNextDelimiter(line));
      try {
        userID = std::stoul(data);
      } catch (std::exception &) {
        THROW_UED_EXCEPTION(ErrType::AUTHDB_fileDataCorrupted);
      }
      line.erase(0, getNextDelimiter(line) + 1);

      data         = line.substr(0, getNextDelimiter(line));
      hashPassSalt = base64_decode(data);
      line.erase(0, getNextDelimiter(line) + 1);

      data = line.substr(0, getNextDelimiter(line));
      try {
        wrapperKeyID = std::stoul(data);
      } catch (std::exception &) {
        THROW_UED_EXCEPTION(ErrType::AUTHDB_fileDataCorrupted);
      }
      line.erase(0, getNextDelimiter(line) + 1);

      data       = line.substr(0, getNextDelimiter(line));
      encDataKey = base64_decode(data);
      line.erase(0, getNextDelimiter(line) + 1);

      data        = line.substr(0, getNextDelimiter(line));
      hashDataKey = base64_decode(data);
      line.erase(0, getNextDelimiter(line) + 1);

      data           = line.substr(0, getNextDelimiter(line));
      hashWrapperKey = base64_decode(data);
      line.erase(0, getNextDelimiter(line) + 1);

      data   = line.substr(0, getNextDelimiter(line));
      master = data.compare("0");
      line.erase(0, getNextDelimiter(line) + 1);

      data = line.substr(0, getNextDelimiter(line));
      salt = base64_decode(data);
      line.erase(0, getNextDelimiter(line) + 1);

      data               = line.substr(0, getNextDelimiter(line));
      auto encWrapperKey = base64_decode(data);

      if (!hashPassSalt.empty()) {
        if (master) {
          _userManager->createMaster(userID, hashPassSalt, wrapperKeyID,
                                     encDataKey, hashDataKey, hashWrapperKey,
                                     salt);
        } else {
          _userManager->createUser(userID, hashPassSalt, wrapperKeyID,
                                   encDataKey, hashDataKey, hashWrapperKey,
                                   salt);
        }
      } else {
        if (master) {
          _userManager->createMaster(userID, wrapperKeyID, encDataKey,
                                     hashDataKey, hashWrapperKey);
        } else {
          _userManager->createUser(userID, wrapperKeyID, encDataKey,
                                   hashDataKey, hashWrapperKey);
        }
      }
      _userManager->insertEncWrapperKey(userID, encWrapperKey);
    }
    if (checkKey(line) == 2) {
      line.erase(0, getNextDelimiter(line) + 1);
      data = line;

      try {
        _wrongTries = std::stoul(data);
      } catch (const std::exception &ia) {
        _wrongTries = 0;
        saveWrongTries(_wrongTries);
      }
    }
  } while (!line.empty());
  _consistentFile->closeFile();
}

void AuthenticationDatabase::saveWrongTries(int wrongTries) throw(
    ErrorMessage) {
  int i = 0;
  std::string line;
  std::string size = std::to_string(wrongTries);
  unsigned int version;

  checkFile();
  line = _consistentFile->readLine();
  i += line.size();
  try {
    version = std::stoul(line);
  } catch (std::exception &) {
    THROW_UED_EXCEPTION(ErrType::AUTHDB_fileDataCorrupted);
  }
  if (version < UED_HEADER_CURRENT_VERSION) {
    THROW_UED_EXCEPTION(ErrType::VERSION_notCompatible);
  }
  do {
    line = _consistentFile->readLine();
    if (checkKey(line) == 2) {
      _consistentFile->write(size, i + getNextDelimiter(line) + 1);
      _consistentFile->closeFile();
      return;
    }
    i += line.size() + 1;
  } while (!line.empty());
  _consistentFile->closeFile();
}

void AuthenticationDatabase::saveMUK(unsigned int userID) throw(ErrorMessage) {
  isValidInDomain<UserID>(userID);

  std::string line;

  if (_userManager->hasUser(userID)) {
    unsigned int version;
    int i                   = 0;
    std::string textToWrite = createMUK(userID);

    checkFile();
    line = _consistentFile->readLine();
    i += line.size();
    try {
      version = std::stoul(line);
    } catch (std::exception &) {
      THROW_UED_EXCEPTION(ErrType::AUTHDB_fileDataCorrupted);
    }
    if (version < UED_HEADER_CURRENT_VERSION) {
      THROW_UED_EXCEPTION(ErrType::VERSION_notCompatible);
    }
    do {
      line = _consistentFile->readLine();
      if (checkKey(line) == 0) {
        _consistentFile->write(textToWrite, i);
        _consistentFile->closeFile();
        return;
      }
      i += line.size();
    } while (!line.empty());
    _consistentFile->write(textToWrite, i);
    _consistentFile->closeFile();
  }
}

void AuthenticationDatabase::deleteFromFile(unsigned int userID) throw(
    ErrorMessage) {
  isValidInDomain<UserID>(userID);

  int i = 0;
  std::string line;
  unsigned int version;

  std::string data = "MUK";
  data += _delimiter;
  data += std::to_string(userID);
  data += _delimiter;

  checkFile();

  line = _consistentFile->readLine();
  i += line.size();
  try {
    version = std::stoul(line);
  } catch (std::exception &) {
    THROW_UED_EXCEPTION(ErrType::AUTHDB_fileDataCorrupted);
  }
  if (version < UED_HEADER_CURRENT_VERSION) {
    THROW_UED_EXCEPTION(ErrType::VERSION_notCompatible);
  }
  do {
    line = _consistentFile->readLine();
    if (line.find(data) != line.npos) {
      std::string size(line.size(), ' ');
      size += '\0';
      _consistentFile->write(size.data(), i);
      size.clear();
    }
    i += line.size();
  } while (!line.empty());
  _consistentFile->closeFile();
}

int AuthenticationDatabase::getWrongTries() { return _wrongTries; }

int AuthenticationDatabase::incrementWrongTries() {
  _wrongTries++;
  saveWrongTries(_wrongTries);
  return _wrongTries;
}

void AuthenticationDatabase::resetWrongTries() {
  _wrongTries = 0;
  saveWrongTries(_wrongTries);
}

size_t AuthenticationDatabase::getNextDelimiter(std::string line) throw(
    ErrorMessage) {
  if (line.empty()) {
    THROW_UED_EXCEPTION(ErrType::AUTHDB_fileDataCorrupted);
  }
  size_t position = line.find(_delimiter);

  if (position == std::string::npos) {
    THROW_UED_EXCEPTION(ErrType::AUTHDB_fileDataCorrupted);
  }
  return position;
}

void AuthenticationDatabase::checkFile() {
  if (!_consistentFile->openFile()) {
    _consistentFile->closeFile();
    createFile();
    _consistentFile->openFile();
  }
}
