/******************************************************************
 *FILE: UserEncryptDecrypt_Authentication.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UserEncryptDecrypt
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Guilherme Ferreira  (guilhermedaniel.ferreira@altran.com)
 * @date Nov, 2017
 *
 * Date      | Author             | Modification
 * 14/08/2018| AKM7COB			  | changed class name logger to logger1. class name logger is already used in ASF framework
 * 25/06/2019| AKM7COB			  | Added new member function isUserExist(user) to check whether user is already existing in system
 * 24/07/2019| AKM7COB			  | Added a new API createUserCredentials() with fixed wkeyid and member function CreateUserCredentialsLogic_FixedKey(user) to create user with fixed wrap keyid.
 */

#ifndef SRC_CORE_AUTHENTICATION_H_
#define SRC_CORE_AUTHENTICATION_H_

//#include <cstring>
#include <list>
#include <map>
#include <memory>
#include <mutex>
#include <vector>

#include <boost/optional.hpp>

#include "core/UserEncryptDecrypt_ConsistentFile.h"
#include "core/authentication/UserEncryptDecrypt_AuthenticationDatabase.h"

class SDCBackend;
class OpensslBackend;
class UserManager;
class Logger1;

/**
 * This class provides a method to authenticate users and can be used as a
 * framework
 * for any purpose (for instance, to authenticate a user to access the
 * PhoneBook, the user account, etc).
 *
 * The CoreFunctions class uses this class to allow only authenticated users to
 * access the encryption and
 * decryption functions with the key associated to the user that requests
 * CoreFunctions API.
 *
 * A typical class permits callers to create as many instances of the class as
 * they want, but this class use the Singleton Pattern.
 * A singleton object provides a global point of access to the resources of its
 * class and returns the same instance no matter how many times an application
 * requests it.
 * The function Authentication::getInstance() provides the access to the
 * resources of this class.
 * */

class Authentication {
public:
	/**
	 * @brief Destructor for the class
	 */
	virtual ~Authentication();

	/**
	 * @brief Creates and associates an hash and a key for the user identified
	 * with userID.
	 * If the provided salt creates a key that already exists, an exception is
	 * thrown to notify the client.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 * @param pass is the PIN or the password the user issues via the A-IVI HMI.
	 * @param salt is a random byte array used for the generation of the user hash
	 * in order to introduce entropy.
	 * @param master is the flag identifying if the user should be created has
	 * master. The default value is false.
	 *
	 *   @return returns true if the credentials are created and false if the
	 * userID already exists or if the salt is the same of other user.
	 */
	bool createUserCredentials(unsigned int userID,
			std::vector<unsigned char> pass,
			std::vector<unsigned char> salt,
			bool master = false) throw(ErrorMessage);

	/**
	 * @brief Creates and associates an hash and a key for the user identified
	 * with userID.
	 * Generates the user salt and stores it
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 * @param pass is the PIN or the password the user issues via the A-IVI HMI.
	 * @param master is the flag identifying if the user should be created has
	 * master. The default value is false.
	 *
	 *   @return returns true if the credentials are created and false if the user
	 * already exists.
	 */
	bool createUserCredentials(unsigned int userID,
			std::vector<unsigned char> pass, bool master);

	/**
	 * @brief Creates and associates a key for the user identified with userID.
	 * This method should be called when
	 * it is intended to create a user without a passphrase associated.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 * @param master is the flag identifying if the user should be created has
	 * master. The default value is false.
	 *
	 *   @return returns true if the credentials are created and false if the user
	 * already exists.
	 */
	bool createUserCredentials(unsigned int userID,
			bool master = false) throw(ErrorMessage);

	/**
	 * @brief Creates and associates a key for the user identified with userID.
	 * This method should be called when
	 * it is intended to create a user without a passphrase associated.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 * @param wKeyID is the unique identifier of the key inside the SDC.
	 * @param master is the flag identifying if the user should be created has
	 * master. The default value is false.
	 *
	 *   @return returns true if the credentials are created and false if the user
	 * already exists.
	 */
	bool createUserCredentials(unsigned int userID, unsigned int wKeyID,
			bool master = false) throw(ErrorMessage);


	/**
	 * @brief This method verifies if the userID received as parameter is
	 * currently exists.
	 *
	 * @param userID is the unique identifier of the user
	 *
	 *
	 * @return true if the userID is currently authenticated.
	 */
	bool isUserExist(unsigned int userID);

	/**
	 * @brief Authenticates a user into the system. An incremental delay is also
	 * triggered in the case of
	 * the current number of users authenticated reached the maximum allowed.
	 * A token is returned when the user and the combination of pass and salt
	 * exists.
	 * When user don't exist or the combination of pass and salt is invalid, an
	 * exception is thrown.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 * @param pass is the PIN or the password the user issues via the A-IVI HMI.
	 * @param salt is a random byte array used for the generation of the user hash
	 * in order to introduce entropy.
	 *
	 * @return The Token in base64 format or nothing if the user was not able to
	 * authenticate.
	 *
	 */
	boost::optional<std::string> authenticateUser(
			unsigned int userID, std::vector<unsigned char> pass,
			std::vector<unsigned char> salt) throw(ErrorMessage);

	/**
	 * @brief Authenticates a user into the system. An incremental delay is also
	 * triggered in the case of
	 * the current number of users authenticated reached the maximum allowed.
	 * A token is returned when the user and the combination of pass and salt
	 * exists.
	 * When user don't exist or the combination of pass and salt is invalid, an
	 * exception is thrown.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 * @param pass is the PIN or the password the user issues via the A-IVI HMI.
	 *
	 * @return The Token in base64 format or nothing if the user was not able to
	 * authenticate.
	 *
	 */
	boost::optional<std::string> authenticateUser(
			unsigned int userID, std::vector<unsigned char> pass);

	/**
	 * @brief Authenticates a user, that does not have a passphrase into the
	 * system. An incremental delay is also triggered in the case of
	 * the current number of users authenticated reached the maximum allowed.
	 * A token is returned when the user exists.
	 * When user don't exist, an exception is thrown.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 *
	 * @return The Token in base64 format or nothing if the user was not able to
	 * authenticate.
	 *
	 */
	boost::optional<std::string> authenticateUser(unsigned int userID) throw(
			ErrorMessage);

	/**
	 * @brief This method verifies if the userID received as parameter is
	 * currently authenticated.

	 * @param userID is the unique identifier of the user that is currently
	 * authenticated.
	 *
	 * @return true if the userID is currently authenticated.
	 */
	bool isUserAuthenticated(unsigned int userID);

	/**
	 * @brief This function unauthenticates an authenticated user.
	 *
	 * @param userID is the unique identifier of a user.
	 *
	 * @return true if userID is unauthenticated successfully, otherwise false is
	 * returned.
	 *
	 * */
	bool unauthenticateUser(unsigned int userID) throw(ErrorMessage);

	/**
	 * @brief This function deletes the user associated with the given userID.
	 * To delete a user, the user has to be authenticated and the hash(pass ||
	 * salt) is equal to the one that the user uses to authenticate via
	 * Authentication::authenticateUser.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 * @param pass is the PIN or the password the user issues via the A-IVI HMI.
	 * @param salt is a random byte array used for the generation of the user hash
	 * in order to introduce entropy.
	 *
	 * @return true if the user key was deleted from the SDC and the
	 * unauthentication process was
	 * correctly performed (via Authentication::unauthenticateUser).
	 * Returns false in all the other possibilities.
	 */
	bool deleteUserCredentials(
			unsigned int userID, std::vector<unsigned char> pass,
			std::vector<unsigned char> salt) throw(ErrorMessage);

	/**
	 * @brief This function deletes the user associated with the given userID.
	 * To delete a user, the user  has to be authenticated and the hash(pass ||
	 * salt)
	 * is equals to the one the user used to authenticate via
	 * Authentication::authenticateUser.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 * @param pass is the PIN or the password the user issues via the A-IVI HMI.
	 *
	 * @return true if the user key was deleted from the SDC and the
	 * unauthentication process was
	 * correctly performed (via Authentication::unauthenticateUser).
	 * Returns false in all the other possibilities.
	 */
	bool deleteUserCredentials(unsigned int userID,
			std::vector<unsigned char> pass);

	/**
	 * @brief This function deletes the user associated with the given userID.
	 * To delete a user, the user has to be authenticated.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 *
	 * @return true if the user key was deleted from the SDC and the
	 * unauthentication process was
	 * correctly performed (via Authentication::unauthenticateUser).
	 * Returns false in all the other possibilities.
	 */
	bool deleteUserCredentials(unsigned int userID) throw(ErrorMessage);

	/**
	 * @brief This function deletes the user associated with the given
	 * userIDToDelete.
	 * The delete action is perpetrated by the user with ID userID.
	 * To delete a user, the user  has to be authenticated and the hash(pass ||
	 * salt) is equals to the one the user used to authenticate via
	 * Authentication::authenticateUser.
	 * This function can only be used by the master or by a user with permission
	 * to use Administrative actions.
	 * Administrative actions group functions can only be used by the master or if
	 * there are no master on the system,
	 * the last authenticated user can use this group of functions.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 * @param pass is the PIN or the password the user issues via the A-IVI HMI.
	 * @param salt is a random byte array used for the generation of the user hash
	 * in order to introduce entropy.
	 * @param userIDToDelete is the unique identifier of the user to be deleted
	 * inside the A-IVI.
	 *
	 *  @return true if the user was properly deleted.
	 * Returns false in all the other possibilities.
	 */
	bool deleteUserCredentials(unsigned int userID,
			std::vector<unsigned char> pass,
			std::vector<unsigned char> salt,
			unsigned int userIDToDelete) throw(ErrorMessage);

	/**
	 * @brief This function deletes the user associated with the given
	 * userIDToDelete.
	 * The delete action is perpetrated by the user with ID userID.
	 * To delete a user, the user  has to be authenticated and the hash(pass ||
	 * salt)
	 * is equals to the one the user used to authenticate via
	 * Authentication::authenticateUser.
	 * This function can only be used by the master or by a user with permission
	 * to use Administrative actions.
	 * Administrative actions group functions can only be used by the master or if
	 * there are no master on the system,
	 * the last authenticated user can use this group of functions.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 * @param pass is the PIN or the password the user issues via the A-IVI HMI.
	 * @param userIDToDelete is the unique identifier of the user to be deleted
	 * inside the A-IVI.
	 *
	 *  @return true if the user was properly deleted.
	 * Returns false in all the other possibilities.
	 */
	bool deleteUserCredentials(unsigned int userID,
			std::vector<unsigned char> pass,
			unsigned int userIDToDelete);

	/**
	 * @brief This function deletes the user associated with the given
	 * userIDToDelete.
	 * The delete action is perpetrated by the user with ID userID.
	 * To delete a user authentication, the user has to be authenticated.
	 * This function can only be used by the master or by a user with permission
	 * to use Administrative actions.
	 * Administrative actions group functions can only be used by the master or if
	 * there are no master on the system,
	 * the last authenticated user can use this group of functions.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 * @param userIDToDelete is the unique identifier of the user to be deleted
	 * inside the A-IVI.
	 *
	 * @return true if the user was properly deleted.
	 * Returns false in all the other possibilities.
	 */
	bool deleteUserCredentials(unsigned int userID,
			unsigned int userIDToDelete) throw(ErrorMessage);

	/**
	 * @brief This function deletes all the users in the system.
	 * To delete the users, the user  has to be authenticated and the hash(pass ||
	 * salt)
	 * is equals to the one the user used to authenticate via
	 * Authentication::authenticateUser.
	 * This function can only be used by the master or by a user with permission
	 * to use Administrative actions.
	 * Administrative actions group functions can only be used by the master or if
	 * there are no master on the system,
	 * the last authenticated user can use this group of functions.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 * @param pass is the PIN or the password the user issues via the A-IVI HMI.
	 * @param salt is a random byte array used for the generation of the user hash
	 * in order to introduce entropy.
	 *
	 * @return true if the users were deleted.
	 * Returns false in all the other possibilities.
	 */
	bool deleteAllUsers(unsigned int userID, std::vector<unsigned char> pass,
			std::vector<unsigned char> salt) throw(ErrorMessage);

	/**
	 * @brief This function deletes all the users in the system.
	 * To delete the users, the user  has to be authenticated and the hash(pass ||
	 * salt)
	 * is equals to the one the user used to authenticate via
	 * Authentication::authenticateUser.
	 * This function can only be used by the master or by a user with permission
	 * to use Administrative actions.
	 * Administrative actions group functions can only be used by the master or if
	 * there are no master on the system,
	 * the last authenticated user can use this group of functions.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 * @param pass is the PIN or the password the user issues via the A-IVI HMI.
	 *
	 * @return true if the users were deleted.
	 * Returns false in all the other possibilities.
	 */
	bool deleteAllUsers(unsigned int userID,
			std::vector<unsigned char> pass) throw(ErrorMessage);

	/**
	 * @brief This function deletes all the users in the system.
	 * To delete the users, the user  has to be authenticated via
	 * Authentication::authenticateUser.
	 * This function can only be used by the master or by a user with permission
	 * to use Administrative actions.
	 * Administrative actions group functions can only be used by the master or if
	 * there are no master on the system,
	 * the last authenticated user can use this group of functions.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 *
	 * @return true if the users were deleted.
	 * Returns false in all the other possibilities.
	 */
	bool deleteAllUsers(unsigned int userID) throw(ErrorMessage);

	/**
	 * @brief This function allow the change of a user’s passphrase to a new one.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 * @param oldPassphrase is the current PIN or the password the user issues via
	 * the A-IVI HMI.
	 * @param oldSalt is the current salt used for the key derivation
	 * cryptographic function.
	 * @param newPassphrase is the new PIN or the password to be used.
	 * @param newSalt is the new salt to be used.
	 *
	 * @return true if the credentials are correctly changed. And false if it
	 * fails, either because
	 * the user does not exist or because the passphrase or salt are incorrect.
	 */
	bool changePassphrase(unsigned int userID,
			std::vector<unsigned char> oldPassphrase,
			std::vector<unsigned char> oldSalt,
			std::vector<unsigned char> newPassphrase,
			std::vector<unsigned char> newSalt) throw(ErrorMessage);

	/**
	 * @brief This function will allow the change of a user’s passphrase to a new
	 * one.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 * @param oldPassphrase is the current PIN or the password the user issues via
	 * the A-IVI HMI.
	 * @param newPassphrase is the new PIN or the password to be used.
	 *
	 * @return true if the credentials are correctly changed. And false if it
	 * fails, either because
	 * the user does not exist or because the passphrase or salt are incorrect.
	 */
	bool changePassphrase(
			unsigned int userID, std::vector<unsigned char> oldPassphrase,
			std::vector<unsigned char> newPassphrase) throw(ErrorMessage);

	/**
	 * @brief This function will allow a user without a passphrase to set a
	 * passphrase.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 * @param newPassphrase is the new PIN or the password the user issues via the
	 * A-IVI HMI.
	 * @param newSalt is the new salt used for the key derivation cryptographic
	 * function.
	 *
	 * @return true if the credentials are correctly changed. And false if it
	 * fails due to the
	 * user not existing or because the given user has a passphrase associated.
	 */
	bool setPassphrase(unsigned int userID,
			std::vector<unsigned char> newPassphrase,
			std::vector<unsigned char> newSalt) throw(ErrorMessage);

	/**
	 * @brief This function will allow a user without a passphrase to set a
	 * passphrase.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 * @param newPassphrase is the new PIN or the password the user issues via the
	 * A-IVI HMI.
	 *
	 * @return true if the credentials are correctly changed. And false if it
	 * fails due to the
	 * user not existing or because the given user has a passphrase associated.
	 */
	bool setPassphrase(
			unsigned int userID,
			std::vector<unsigned char> newPassphrase) throw(ErrorMessage);

	/**
	 * @brief This function will allow the a user to remove his passphrase.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 * @param oldPassphrase is the current PIN or the password the user issues via
	 * the A-IVI HMI.
	 * @param oldSalt is the current salt used for the key derivation
	 * cryptographic function.
	 *
	 * @return true if the credentials are correctly changed. And false if it
	 * fails due to the
	 * user not existing or because the passphrase or salt are incorrect.
	 */
	bool removePassphrase(unsigned int userID,
			std::vector<unsigned char> oldPassphrase,
			std::vector<unsigned char> oldSalt) throw(ErrorMessage);

	/**
	 * @brief This function will allow the user to stop having a passphrase.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 * @param oldPassphrase is the current PIN or the password the user issues via
	 * the A-IVI HMI.
	 *
	 * @return true if the credentials are correctly changed. And false if it
	 * fails due to the
	 * user not existing or because the passphrase or salt are incorrect.
	 */
	bool removePassphrase(
			unsigned int userID,
			std::vector<unsigned char> oldPassphrase) throw(ErrorMessage);

	/**
	 * @brief This function will allow the masterUser to remove another users
	 * passphrase.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 * @param pass is the current PIN or the password the user issues via the
	 * A-IVI HMI.
	 * @param salt is the current salt used for the key derivation cryptographic
	 * function.
	 * @param userIDToDelete is the unique identifier of the user to delete the
	 * passphrase inside the A-IVI.
	 *
	 * @return true if the credentials are correctly changed. And false otherwise.
	 */
	bool removeAnotherUserPassphrase(
			unsigned int userID, std::vector<unsigned char> pass,
			std::vector<unsigned char> salt,
			unsigned int userIDToDelete) throw(ErrorMessage);

	/**
	 * @brief This function will allow the masterUser to remove another users
	 * passphrase.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 * @param pass is the current PIN or the password the user issues via the
	 * A-IVI HMI.
	 * @param userIDToDelete is the unique identifier of the user to delete the
	 * passphrase inside the A-IVI.
	 *
	 * @return true if the credentials are correctly changed. And false otherwise.
	 */
	bool removeAnotherUserPassphrase(
			unsigned int userID, std::vector<unsigned char> pass,
			unsigned int userIDToDelete) throw(ErrorMessage);

	/**
	 * @brief This function will allow the masterUser to remove another users
	 * passphrase.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 * @param userIDToDelete is the unique identifier of the user to delete the
	 * passphrase inside the A-IVI.
	 *
	 * @return true if the credentials are correctly changed. And false otherwise.
	 */
	bool removeAnotherUserPassphrase(
			unsigned int userID, unsigned int userIDToDelete) throw(ErrorMessage);

	/**
	 * @brief This function returns a list with all of the stored users.
	 * @return The list of all users that were authenticated before and never
	 * deleted.
	 */
	std::list<unsigned int> getCreatedUsers();

	/**
	 * @brief This function returns a list with all the currently authenticated
	 * users.
	 * @return The list of all users that are currently authenticated.
	 */
	std::list<unsigned int> getCurrentAuthenticatedUsers();

	/**
	 * @brief Receives a function that needs the keyID and executes it based on
	 * the userID.
	 * Like so the keyID never gets out of Authentication instance.
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 * @param fun  is the function to be called with the keyID as parameter.
	 *
	 * @return returns the return of the provided function.
	 */
	std::vector<unsigned char> callWithUserIDKey(
			unsigned int userID,
			std::function<std::vector<unsigned char>(unsigned int)> fun);

	/**
	 * @brief Delete a user from the Diagnosis method
	 *
	 * @param userID is the unique identifier of the user inside the A-IVI.
	 */
	void delUserDiagnosis(unsigned int userID) throw(ErrorMessage);

	/**
	 *   @brief Provides an instance of the Authentication class object.
	 *
	 *   This function provides the access to class object, but only one instance
	 * is
	 * returned as mentioned above in the definition of singleton pattern.
	 *
	 *   @return returns a shared pointer of the instance of the Authentication
	 */
	static std::shared_ptr<Authentication> &getInstance(bool createNew = false);
#ifdef TESTING
	std::vector<unsigned char> getKey(unsigned int userID);
	unsigned int getKID(unsigned int userID);
	std::vector<unsigned char> getWrappingKey();
#endif

private:
	void insertCurrentAuthenticatedUser(unsigned int userID);

	Authentication();

	boost::optional<std::tuple<unsigned int, std::vector<unsigned char>,
	std::vector<unsigned char>>>
	keyDerivation(std::vector<unsigned char> &pass,
			std::vector<unsigned char> &salt);

	bool CreateUserCredentialsLogic(
			unsigned int userID, boost::optional<std::vector<unsigned char>> pass,
			boost::optional<std::vector<unsigned char>> salt, bool master);
	bool CreateUserCredentialsLogic_FixedKey(
			unsigned int userID, bool master, unsigned int uWKeyID);
	boost::optional<std::string> authenticateUserLogic(unsigned int userID);
	bool deleteUserLogic(unsigned int userID);
	std::vector<unsigned char> generateKey(std::vector<unsigned char> &random);
	std::vector<unsigned char> generateSalt();
	bool keyCompare(std::vector<unsigned char> hash);
	std::vector<unsigned char> concatAndHash(std::vector<unsigned char> &pass,
			std::vector<unsigned char> salt);
	void insertKeysAndPass(
			unsigned int userID, std::vector<unsigned char> newPassphrase,
			std::vector<unsigned char> newSalt) throw(ErrorMessage);
	void deleteKeysAndPass(unsigned int userID);
	void wrongTry();
	void throwDeleteAllUsersException(unsigned int userId) throw(ErrorMessage);
	static std::shared_ptr<Authentication> _instance;

	std::shared_ptr<Logger1> _logger;
	std::shared_ptr<LogContext> _ctx;
	static std::mutex _mutex;

	std::list<unsigned int> _currentAuthUserID;
	std::shared_ptr<SDCBackend> _sdcBackend;
	std::shared_ptr<OpensslBackend> _opensslBackend;
	std::shared_ptr<AuthenticationDatabase> _authenticationDatabase;
	std::shared_ptr<UserManager> _userManager;
	const unsigned int _guestAuthUserID;
	static const unsigned int SALT_SIZE;
	static int _maxAuthUser;

	/* 0, 5, 60, 3600, 3600 * 3, 3600 * 6, 3600 * 12 */
	static const std::array<unsigned int, 7> _incrementalDelay;

#ifdef TESTING
	std::vector<unsigned char> UserKey;
#endif
};

#endif /* SRC_CORE_AUTHENTICATION_H_ */
