/******************************************************************
 *FILE: UserEncryptDecrypt_CoreFunctions.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: CoreFunctions
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Guilherme Ferreira  (guilhermedaniel.ferreira@altran.com)
 * @date Nov, 2017
 */

#ifndef SRC_CORE_COREFUNCTIONS_H_
#define SRC_CORE_COREFUNCTIONS_H_

#include <memory>
#include <vector>

class SDCBackend;
class Authentication;

/**
 * This class provides methods to encrypt, decrypt and digest data.
 * It contains the main functionalities of UserEncryptDecrypt application.
 */
class CoreFunctions {
 public:
  /**
   * @brief Function to encrypt plain text using AES256 algorithm<br/>
   * Wiki: https://en.wikipedia.org/wiki/Advanced_Encryption_Standard<br/>
   * UserId must be authenticated with Authentication::authenticateUser
   *
   * Our implementation makes a slight change to the AES algorithm.
   * The first IV (the one that must be provided) is XORed with the first block.
   * We generate a random number for the first block and append it to the
   * beginning of the data (new data size will become data.size()+16).
   * Therefore, we can use the first IV as a 16 byte array of zeros.
   *
   * @param token is the private Token of the user
   * @param data is plain text to be encrypted
   * @return A UEDPacket with base64 encrypted data
   */
  static std::vector<unsigned char> AESEncrypt(std::string token,
                                               std::vector<unsigned char> data);

  /**
   * @brief Function to decrypt encrypted text using AES256 algorithm<br/>
   * Wiki: https://en.wikipedia.org/wiki/Advanced_Encryption_Standard<br/>
   * UserId must be authenticated with Authentication::authenticateUser
   *
   * As a consequence of the change to the AES encrypt algorithm we remove
   * the first 16 bytes from the decrypted data.
   *
   * @param token is the private Token of the user
   * @param data is encrypted text to be decrypted, this must be an
   * UEDPacket(generated by AESEncrypt)
   * @return The result of decrypt operation, a plain text
   */
  static std::vector<unsigned char> AESDecrypt(std::string token,
                                               std::vector<unsigned char> data);

  /**
   * @brief Function that produces a hash of the provided data using the SHA256
   * algorithm<br/>
   * Wiki: https://en.wikipedia.org/wiki/SHA-2<br/>
   * UserId must be authenticated with Authentication::authenticateUser
   *
   * @param token is the private Token of the user
   * @param data is what we want to ensure integrity
   * @return A UEDPacket with base64 generated hash of data
   */
  static std::vector<unsigned char> Digest(std::string token,
                                           std::vector<uint8_t> &data);
};

#endif /* SRC_CORE_COREFUNCTIONS_H_ */
