/******************************************************************
 *FILE: UserEncryptDecrypt_ConsistentFile.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: ConsistentFile
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author José Oliveira  (josearmando.oliveira@altran.com)
 * @date Dec, 2017
 *
 * Date      | Author             | Modification
 * 14/08/2018| AKM7COB			  | changed class name logger to logger1. class name logger is already used in ASF framework
 * 24/07/2019| AKM7COB			  | File path modified. Changed file path to persistent folder. As per requirement from telematics key needs to be retained over software update.
 */

#ifndef USERENCRYPTDECRYPT_CONSISTENTFILE_H
#define USERENCRYPTDECRYPT_CONSISTENTFILE_H

#include <domains/UserEncryptDecrypt_FilenameDomain.h>
#include <cstdio>
#include <map>
#include <memory>
#include <sstream>
#include <vector>
#include "core/logger/UserEncryptDecrypt_Logger.h"

/**
 * This class provides consistency of files used by UserEncryptDecrypt.
 * Each time a caller want to write in file, consistency is ensured creating a
 * backup file and the signature of both files.
 * After that, in case of disaster(original file is corrupted) original file is
 * replaced by the backup.
 *
 * An example of it's usage:
 * std::shared_ptr<ConsistentFile> consistentFile =
 * ConsistentFile::getFile("filename");
 * consistentFile->openFile();
 * consistentFile->write("textToWrite");
 * consistentFile->closeFile();
 */
class ConsistentFile {
 public:
  /**
   * @brief Obtain a ConsistentFile shared pointer of an existing file.
   * Case filename does't exists in file list the filename is added to the list.
   *
   * @param filename File name to handle
   *
   * @return Return a ConsistentFile shared pointer
   */
  static std::shared_ptr<ConsistentFile> getFile(std::string filename);

  /**
   * @brief Destructor for the class.
   */
  virtual ~ConsistentFile();

  /**
   * @brief This function open the current file to be read or modified.
   *
   * @param reload Optional parameter with default value false. In case of true
   * the current file is reloaded.
   *
   * @return Returns true if the file is successfully opened and false if
   * something wrong happen and the file can't be opened.
   */
  bool openFile(bool reload = false);

  /**
   * @brief Close the file and it can't be modified.
   */
  void closeFile();

  /**
   * @brief writes bytes in data to the file
   * @param data - data to be written
   * @param offset - offset from the beginning of the file
   */
  void write(std::vector<unsigned char> data, size_t offset = 0);
  /**
   * @brief writes the string to the file
   * @param data - string to be written
   * @param offset - offset from the beginning of the file
   */
  void write(std::string data, size_t offset = 0);

  /**
   * @brief appends the string to the end of the file
   * @param data - string to be appended
   */
  void append(std::string data);

  /**
   * @brief appends the string to the end of the file
   * @param data - string to be appended
   */
  void append(std::vector<unsigned char> data);

  /**
   * @brief reads the next n bytes
   * @param size (n) - number the bytes to the read
   * @return read bytes
   */
  std::vector<unsigned char> readBytes(size_t size);

  /**
   * @brief Reads a line in the current file
   *
   * @return line
   */
  std::string readLine();

#ifdef TESTING
  /**
   * @brief gets the file descriptor
   * @details note that the consistency properties can not be ensured if writes
   * are done directly using the file
   * descriptor, and will actually be roled back if the file is open again
   * @return A File pointer to the file
   */
  FILE* getFILE() { return _file; }
#endif

 private:
  static std::map<std::string, std::shared_ptr<ConsistentFile>> _files;

  explicit ConsistentFile(std::string filename);

  void copyConsistent(std::string file, std::string fileBackup);
  void closeFileDiscriptors(FILE* reader, FILE* bckreader);

  bool _openFile();

  const std::string _filename;
  const std::string _bkpFilename;
  const std::string _sigFilename;

  FILE* _file;
  FILE* _fileBackup;
  long _readOffset;
  bool _filesOpen;
  std::shared_ptr<Logger1> _logger;
  std::shared_ptr<LogContext> _ctx;
};

#endif  // USERENCRYPTDECRYPT_CONSISTENTFILE_H
