/******************************************************************
 *FILE: UserEncryptDecrypt_ConsistentFile.cpp
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: ConsistentFile
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author José Oliveira  (josearmando.oliveira@altran.com)
 * @date Dec, 2017
* 24/07/2019| AKM7COB			  | File path modified. Changed file path to persistent folder. As per requirement from telematics key needs to be retained over software update.
 */

#include "core/UserEncryptDecrypt_ConsistentFile.h"
#include <UserEncryptDecrypt_Utils.h>
#include <backends/UserEncryptDecrypt_BackendManager.h>
#include <fstream>

#define CONSISTENT_FILE_PATH std::string("/var/opt/bosch/persistent/cryptoapp/")

std::map<std::string, std::shared_ptr<ConsistentFile>> ConsistentFile::_files;

void ConsistentFile::closeFileDiscriptors(FILE *reader, FILE *bckreader) {
  if (_file == NULL || _fileBackup == NULL) {
    if (reader != NULL) {
      fclose(reader);
    }
    if (bckreader != NULL) {
      fclose(bckreader);
    }
    THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
  }
}

ConsistentFile::ConsistentFile(std::string filename)
    : _filename(CONSISTENT_FILE_PATH + filename),
      _bkpFilename(CONSISTENT_FILE_PATH + "." + filename + ".bkp"),
      _sigFilename(CONSISTENT_FILE_PATH + "." + filename + ".sig"),
      _readOffset(0),
      _filesOpen(false),
      _file(NULL),
      _fileBackup(NULL),
      _logger(Logger1::getLogger()),
      _ctx(_logger->registerContext("ConF", "Consistent File")) {}

ConsistentFile::~ConsistentFile() {
  if (_filesOpen) {
    if (_file == NULL || _fileBackup == NULL) {
      return;
    }
    _logger->log(_ctx, LogLevel::INFO, "--closing file");
    fflush(_file);
    fflush(_fileBackup);
    fclose(_file);
    fclose(_fileBackup);
    _file       = NULL;
    _fileBackup = NULL;
    _filesOpen  = false;
  }
}

std::shared_ptr<ConsistentFile> ConsistentFile::getFile(std::string filename) {
  if (_files.count(filename) == 0) {
    try {
      std::pair<std::string, std::shared_ptr<ConsistentFile>> pair(
          filename, std::make_shared<ConsistentFile>(ConsistentFile(filename)));
      _files.insert(pair);
    } catch (std::bad_alloc &) {
      THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
    }
  }
  return _files[filename];
}

bool ConsistentFile::_openFile() {
  FILE *reader       = fopen(_filename.c_str(), "rb");
  FILE *backupReader = fopen(_bkpFilename.c_str(), "rb");

  std::shared_ptr<SDCBackend> sdc =
      BackendManager::getInstance()->get<SDCBackend>();
  if (reader == NULL) {
    FILE *a = NULL;
    a       = fopen(_filename.c_str(), "wb+");
    if (a != NULL) {
      fclose(a);
      a = NULL;
    } else {
      THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
    }
    a = fopen(_bkpFilename.c_str(), "wb+");
    if (a != NULL) {
      fclose(a);
      a = NULL;
    } else {
      THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
    }

    FILE *sig = fopen(_sigFilename.c_str(), "wb+");
    if (sig == NULL) {
      fclose(backupReader);
      THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
    }
    std::vector<unsigned char> data = sdc->Digest(std::vector<unsigned char>());
    if (fputs(base64_encode(data.data(), data.size()).c_str(), sig) == EOF) {
      fclose(sig);
      fclose(backupReader);
      THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
    }
    if (fputs("\n", sig) == EOF) {
      fclose(sig);
      fclose(backupReader);
      THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
    }
    if (fputs(base64_encode(data.data(), data.size()).c_str(), sig) == EOF) {
      fclose(sig);
      fclose(backupReader);
      THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
    }
    fclose(sig);
    ;
    sig = NULL;
    if (_file != NULL) {
      fclose(_file);
      _file = NULL;
    }
    _file = fopen(_filename.c_str(), "rb+");
    if (_file == NULL) {
      fclose(backupReader);
      THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
    }
    if (_fileBackup != NULL) {
      fclose(_file);
      _file = NULL;
    }
    _fileBackup = fopen(_bkpFilename.c_str(), "rb+");
    if (_fileBackup == NULL) {
      fclose(backupReader);
      THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
    }
    _filesOpen = true;

    if (backupReader != NULL) {
      fclose(backupReader);
      backupReader = NULL;
    }
    closeFileDiscriptors(reader, backupReader);
    if (reader != NULL) {
      fclose(reader);
      reader = NULL;
    }
    if (backupReader != NULL) {
      fclose(backupReader);
      backupReader = NULL;
    }
    return false;
  } else if (backupReader == NULL) {
    FILE *bck = fopen(_bkpFilename.c_str(), "wb+");
    if (bck == NULL) {
      THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
    }
    FILE *sig = fopen(_sigFilename.c_str(), "wb+");
    if (sig == NULL) {
      fclose(bck);
      THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
    }
    size_t lineSize = 256;
    unsigned char line[lineSize];
    std::vector<unsigned char> fileData;
    int readBytes = 0;
    do {
      readBytes = fread(line, sizeof(unsigned char), lineSize, reader);
      fileData.insert(fileData.end(), line, line + readBytes);
      fwrite(line, sizeof(unsigned char), readBytes, bck);
    } while (readBytes != 0);
    std::vector<unsigned char> data(fileData.begin(), fileData.end());
    data = sdc->Digest(data);

    if (fputs(base64_encode(data.data(), data.size()).c_str(), sig) == EOF) {
      fclose(sig);
      fclose(bck);
      THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
    }
    if (fputs("\n", sig) == EOF) {
      fclose(sig);
      fclose(bck);
      THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
    }
    if (fputs(base64_encode(data.data(), data.size()).c_str(), sig) == EOF) {
      fclose(sig);
      fclose(bck);
      THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
    }

    fclose(sig);
    sig = NULL;
    fclose(bck);
    bck = NULL;
    if (reader != NULL) {
      fclose(reader);
      reader = NULL;
    }
    if (backupReader != NULL) {
      fclose(backupReader);
      backupReader = NULL;
    }
  } else {
    size_t lineSize = 256;
    unsigned char line[lineSize];
    int readBytes = 0;

    std::vector<unsigned char> fileData;
    do {
      readBytes = fread(line, sizeof(unsigned char), lineSize, reader);
      fileData.insert(fileData.end(), line, line + readBytes);
    } while (readBytes != 0);
    std::string tmp(fileData.begin(), fileData.end());

    std::vector<unsigned char> fileBkpData;
    do {
      readBytes = fread(line, sizeof(unsigned char), lineSize, backupReader);
      fileBkpData.insert(fileBkpData.end(), line, line + readBytes);
    } while (readBytes != 0);
    tmp.erase(tmp.begin(), tmp.end());
    tmp.insert(tmp.begin(), fileBkpData.begin(), fileBkpData.end());

    std::ifstream sigReader(_sigFilename);

    std::string lineSig;
    getline(sigReader, lineSig);
    std::string lineSig1;
    getline(sigReader, lineSig1);

    std::vector<unsigned char> fileSigV = sdc->Digest(fileData);
    std::string filesig = base64_encode(fileSigV.data(), fileSigV.size());

    std::vector<unsigned char> fileBkpSigV = sdc->Digest(fileBkpData);
    std::string fileBkpsig =
        base64_encode(fileBkpSigV.data(), fileBkpSigV.size());

    if (filesig == lineSig) {
      if (fileBkpsig != lineSig1) {
        copyConsistent(_bkpFilename, _filename);
        if (reader != NULL) {
          fclose(reader);
          reader = NULL;
        }
        if (backupReader != NULL) {
          fclose(backupReader);
          backupReader = NULL;
        }
      }
    } else {
      if (fileBkpsig == lineSig1) {
        copyConsistent(_filename, _bkpFilename);
        if (reader != NULL) {
          fclose(reader);
          reader = NULL;
        }
        if (backupReader != NULL) {
          fclose(backupReader);
          backupReader = NULL;
        }
      } else {
        std::ofstream ofs;
        ofs.open(_filename, std::ofstream::out | std::ofstream::trunc);
        ofs.close();
        ofs.open(_bkpFilename, std::ofstream::out | std::ofstream::trunc);
        ofs.close();
        if (_file != NULL) {
          fclose(_file);
          _file = NULL;
        }
        if (_fileBackup != NULL) {
          fclose(_fileBackup);
          _fileBackup = NULL;
        }
        _file = fopen(_filename.c_str(), "rb+");
        if (_file == NULL) {
          THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
        }
        _fileBackup = fopen(_bkpFilename.c_str(), "rb+");
        if (_fileBackup == NULL) {
          THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
        }
        _filesOpen = true;
        if (reader != NULL) {
          fclose(reader);
          reader = NULL;
        }
        if (backupReader != NULL) {
          fclose(backupReader);
          backupReader = NULL;
        }
        closeFileDiscriptors(reader, backupReader);
        if (reader != NULL) {
          fclose(reader);
          reader = NULL;
        }
        if (backupReader != NULL) {
          fclose(backupReader);
          backupReader = NULL;
        }
        return false;
      }
    }
  }

  if (_file != NULL) {
    fclose(_file);
    _file = NULL;
    std::cout << "file closed";
  }

  if (_fileBackup != NULL) {
    fclose(_fileBackup);
    _fileBackup = NULL;
    std::cout << "bkp file closed";
  }

  _file = fopen(_filename.c_str(), "rb+");
  if (_file == NULL) {
	  std::cout << "file open error";
    THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
  }
  _fileBackup = fopen(_bkpFilename.c_str(), "rb+");
  if (_fileBackup == NULL) {
	  std::cout << "bkp file open error";
    THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
  }
  _filesOpen = true;

  closeFileDiscriptors(reader, backupReader);

  if (reader != NULL) {
    fclose(reader);
    reader = NULL;
  }
  if (backupReader != NULL) {
    fclose(backupReader);
    backupReader = NULL;
  }
  return true;
}

void ConsistentFile::closeFile() {
  if (_filesOpen) {
    _readOffset = 0;
    fflush(_file);
    fflush(_fileBackup);
  }
}

void ConsistentFile::copyConsistent(std::string file, std::string fileBackup) {
  FILE *f = fopen(file.c_str(), "wb+");
  if (f == NULL) {
    THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
  }
  FILE *fb = fopen(fileBackup.c_str(), "rb+");
  if (fb == NULL) {
    fclose(f);
    THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
  }
  size_t size = 256;
  unsigned char dat[size];
  std::vector<unsigned char> data;
  size_t readedBytes = 0;
  do {
    readedBytes = fread(dat, sizeof(unsigned char), size, fb);
    data.insert(data.end(), dat, dat + readedBytes);
  } while (readedBytes != 0);

  fwrite(data.data(), sizeof(unsigned char), data.size(), f);
  fflush(f);
  fclose(f);
  fclose(fb);
}

void ConsistentFile::write(std::vector<unsigned char> data, size_t offset) {
  if (_file == NULL || _fileBackup == NULL) {
    THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
  }
  if (fseek(_file, offset, SEEK_SET) != 0) {
    THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
  }

  size_t wBytes =
      fwrite(data.data(), sizeof(unsigned char), data.size(), _file);

  std::string line;
  std::string filedata;
  std::shared_ptr<SDCBackend> sdc =
      BackendManager::getInstance()->get<SDCBackend>();
  size_t size = 256;
  unsigned char dat[size];
  std::vector<unsigned char> fileData;
  size_t readedBytes = 0;
  if (fseek(_file, 0, SEEK_SET)) {
    THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
  }

  do {
    readedBytes = fread(dat, sizeof(unsigned char), size, _file);
    fileData.insert(fileData.end(), dat, dat + readedBytes);
  } while (readedBytes != 0);

  std::vector<unsigned char> fileSigV = sdc->Digest(fileData);
  std::string filesig = base64_encode(fileSigV.data(), fileSigV.size());

  std::ofstream sigfile(_sigFilename);
  sigfile << filesig << std::endl;

  if (fseek(_fileBackup, offset, SEEK_SET) != 0) {
    THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
  }

  fwrite(data.data(), sizeof(unsigned char), data.size(), _fileBackup);
  fflush(_fileBackup);
  sigfile << filesig << std::endl;
  sigfile.flush();
  sigfile.close();
  if (_file == NULL || _fileBackup == NULL) {
    THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
  }
}

void ConsistentFile::write(std::string data, size_t offset) {
  if (_file == NULL || _fileBackup == NULL) {
    THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
  }
  std::vector<unsigned char> vec(data.begin(), data.end());

  write(vec, offset);

  std::vector<unsigned char> v((unsigned char)0);
  if (fseek(_file, 0, SEEK_SET)) {
    THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
  }
  long f = ftell(_file);
  if (f == 1L) {
    THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
  }
  write(v, f);
  if (_file == NULL || _fileBackup == NULL) {
    THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
  }
}

std::vector<unsigned char> ConsistentFile::readBytes(size_t size) {
  if (_file == NULL || _fileBackup == NULL) {
    THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
  }
  if (fseek(_file, _readOffset, SEEK_SET)) {
    THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
  }
  unsigned char dat[size] = {0};
  size_t readedBytes      = fread(dat, sizeof(unsigned char), size, _file);
  _readOffset += readedBytes;
  if (_file == NULL || _fileBackup == NULL) {
    THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
  }
  return std::vector<unsigned char>(dat, dat + readedBytes);
}

std::string ConsistentFile::readLine() {
  if (_file == NULL || _fileBackup == NULL) {
    THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
  }
  if (fseek(_file, _readOffset, SEEK_SET)) {
    THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
  }
  size_t n = 1;
  size_t readedBytes;
  char *str = (char *)malloc(1);
  if (str == nullptr) {
    THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
  }
  readedBytes = getline(&str, &n, _file);
  if (readedBytes == -1) {
    readedBytes = getline(&str, &n, _file);
  }
  if (readedBytes != -1) {
    _readOffset += readedBytes;
  } else {
    free(str);
    if (_file == NULL || _fileBackup == NULL) {
      THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
    }
    return std::string();
  }
  std::string line(str);
  free(str);
  if (_file == NULL || _fileBackup == NULL) {
    THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
  }
  return line;
}

bool ConsistentFile::openFile(bool reload) {
  //isValidInDomain<Filename>(_filename);
  _readOffset = 0;
  if (reload || !_filesOpen) {
    if (_file != NULL) {
      fflush(_file);
      fflush(_fileBackup);
      fclose(_file);
      _file = NULL;
      fclose(_fileBackup);
      _fileBackup = NULL;
    }
    _file       = NULL;
    _fileBackup = NULL;
    _filesOpen  = false;
    return _openFile();
  }
  return true;
}

void ConsistentFile::append(std::vector<unsigned char> data) {
  if (fseek(_file, 0, SEEK_END)) {
    THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
  }

  size_t f = ftell(_file);
  write(data, f);
}

void ConsistentFile::append(std::string data) {
  if (fseek(_file, 0, SEEK_END)) {
    THROW_UED_EXCEPTION(ErrType::CONSISTENT_fileNotOpen);
  }
  size_t f = ftell(_file);
  write(data, f);
}
