/******************************************************************
 *FILE: UserEncryptDecrypt_BackendManager.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: Manager of all Backends
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Ulisses Costa (marioulisses.costa@altran.com)
 * @date Oct, 2017
 */

#ifndef SRC_BACKENDS_BACKENDMANAGER_H_
#define SRC_BACKENDS_BACKENDMANAGER_H_

#include <boost/variant.hpp>
#include <memory>
#include <mutex>
#include <set>
#include "error/UserEncryptDecrypt_ErrorMessage.h"

class SDCBackend;
class OpensslBackend;

typedef boost::variant<std::shared_ptr<SDCBackend>,
                       std::shared_ptr<OpensslBackend>>
    BackendType;

/**
 * This class provides the management of all the backends and
 * controls the single instance of each backend that exists at any given time
 *
 * The Authentication and CoreFunctions classes uses this class to get the
 * OpensslBackend and SDCBackend objects.
 *
 * A typical class permits callers to create as many instances of the class as
 * they want, but this class use the Singleton Pattern.
 * A singleton object provides a global point of access to the resources of its
 * class and returns the same instance no matter how many times an application
 * requests it.
 * The function BackendManager::getInstance() provides the access to the
 * resources of this class.
 */
class BackendManager {
 public:
  /**
   * @brief Destructor for the class
   */
  virtual ~BackendManager();

  /**
   *   @brief Provides an instance of the BackendManager class object.
   *
   *   This function provides the access to class object, but only one instance
   * is returned as mentioned above in the definition of singleton pattern.
   *
   *   @return returns a shared pointer of the instance of the BackendManager
   */
  static std::shared_ptr<BackendManager> getInstance();

  /**
   *   @brief Provides a centralized way to access an instance of the desired
   * backend
   *
   *   @return returns a shared pointer of the desired Backend
   */
  template <typename T>
  std::shared_ptr<T> get() throw(ErrorMessage) {
    for (auto b : _backends) {
      if (b.type() == typeid(std::shared_ptr<T>)) {
        return boost::get<std::shared_ptr<T>>(b);
      }
    }
    THROW_UED_EXCEPTION(ErrType::BACKEND_invalidTypeOfBackend);
  }

 private:
  BackendManager();
  void initBackends();
  void addBackend(BackendType backend);

  static std::shared_ptr<BackendManager> _instance;
  static std::mutex _mutex;
  std::set<BackendType, std::less<BackendType>> _backends;
};

#endif /* SRC_BACKENDS_BACKENDMANAGER_H_ */
