/******************************************************************
 *FILE: UserEncryptDecrypt_BackendManager.cpp
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: Manager of all Backends
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Ulisses Costa (marioulisses.costa@altran.com)
 * @date Oct, 2017
 */

#include "UserEncryptDecrypt_BackendManager.h"
#include "backends/openssl/UserEncryptDecrypt_OpensslBackend.h"
#include "backends/sdc/UserEncryptDecrypt_SDCBackend.h"
#include "error/UserEncryptDecrypt_ErrorMessage.h"

std::mutex BackendManager::_mutex;
std::shared_ptr<BackendManager> BackendManager::_instance;

BackendManager::BackendManager() { initBackends(); }

BackendManager::~BackendManager() {}

void BackendManager::initBackends() {
  try {
    addBackend(SDCBackend::getInstance());
    addBackend(OpensslBackend::getInstance());
  } catch (ErrorMessage& e) {
    throw;
  } catch (std::bad_alloc&) {
    THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
  }
}

std::shared_ptr<BackendManager> BackendManager::getInstance() {
  if (!_instance) {
    std::lock_guard<std::mutex> lock(_mutex);
    if (!_instance) {
      try {
        _instance.reset(new BackendManager());
      } catch (std::bad_alloc&) {
        THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
      }
    }
  }
  return _instance;
}

void BackendManager::addBackend(BackendType backend) {
  bool emplaceResult = false;
  try {
    auto emplacePairResult = _backends.emplace(backend);
    emplaceResult          = emplacePairResult.second;
  } catch (...) {
    throw;
  }
  if (!emplaceResult) {
    THROW_UED_EXCEPTION(ErrType::BACKEND_BackendsNotLoaded);
  }
}
