/******************************************************************
 *FILE: UserEncryptDecrypt_Utils.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UserEncryptDecrypt
 *COPYRIGHT: © 2017-2018 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author José Oliveira  (josearmando.oliveira@altran.com)
 * @date Dec, 2017
 */

#ifndef USERENCRYPTDECRYPT_USERENCRYPTDECRYPT_UTILS_H
#define USERENCRYPTDECRYPT_USERENCRYPTDECRYPT_UTILS_H

#include <vector>
#include "backends/sdc/UserEncryptDecrypt_SDCBackend.h"
#include "core/UserEncryptDecrypt_Configurations.h"

/**
   * @brief Generates and stores a the UED system key.
   *
   */
static void initKey() {
  unsigned int kid = Configurations::getInstance()->get<unsigned int>(
      Configurations::USER_ENCRYPT_DECRYPT_SIG_KEYID);
  auto _sdcBackend               = SDCBackend::getInstance();
  std::vector<unsigned char> key = _sdcBackend->GetRandom(32);

  unsigned int errorCode = _sdcBackend->Store(key, kid);
}

static const std::string base64_chars =
    "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
    "abcdefghijklmnopqrstuvwxyz"
    "0123456789+/";

/**
   * @brief Checks if is in char set of Base64
   *
   * @param c
   *
   * @return returns true if the received char is in base64, false otherwise.
   */
static inline bool is_base64(unsigned char c) {
  return (isalnum(c) || (c == '+') || (c == '/'));
}

/**
   * @brief Encodes the provided byte array to base64.
   *
   * @param bytes_to_encode
   * @param in_len
   *
   * @return Returns a string with the provided byte array in base64.
   */
static std::string base64_encode(unsigned char const* bytes_to_encode,
                                 unsigned int in_len) {
  std::string ret;
  int i = 0;
  int j = 0;
  unsigned char char_array_3[3];
  unsigned char char_array_4[4];

  while (in_len--) {
    char_array_3[i++] = *(bytes_to_encode++);
    if (i == 3) {
      char_array_4[0] = (char_array_3[0] & 0xfc) >> 2;
      char_array_4[1] =
          ((char_array_3[0] & 0x03) << 4) + ((char_array_3[1] & 0xf0) >> 4);
      char_array_4[2] =
          ((char_array_3[1] & 0x0f) << 2) + ((char_array_3[2] & 0xc0) >> 6);
      char_array_4[3] = char_array_3[2] & 0x3f;

      for (i = 0; (i < 4); i++) ret += base64_chars[char_array_4[i]];
      i      = 0;
    }
  }

  if (i) {
    for (j = i; j < 3; j++) char_array_3[j] = '\0';

    char_array_4[0] = (char_array_3[0] & 0xfc) >> 2;
    char_array_4[1] =
        ((char_array_3[0] & 0x03) << 4) + ((char_array_3[1] & 0xf0) >> 4);
    char_array_4[2] =
        ((char_array_3[1] & 0x0f) << 2) + ((char_array_3[2] & 0xc0) >> 6);
    char_array_4[3] = char_array_3[2] & 0x3f;

    for (j = 0; (j < i + 1); j++) ret += base64_chars[char_array_4[j]];

    while ((i++ < 3)) ret += '=';
  }

  return ret;
}

/**
   * @brief Decodes the provided string in base64.
   *
   * @param encoded_string
   *
   * @return Returns a decoded vector of unsigned char of the string in base64.
   */
static std::vector<unsigned char> base64_decode(
    std::string const& encoded_string) {
  int in_len = encoded_string.size();
  int i      = 0;
  int j      = 0;
  int in_    = 0;
  unsigned char char_array_4[4], char_array_3[3];
  std::vector<unsigned char> ret;

  while (in_len-- && (encoded_string[in_] != '=') &&
         is_base64(encoded_string[in_])) {
    char_array_4[i++] = encoded_string[in_];
    in_++;
    if (i == 4) {
      for (i            = 0; i < 4; i++)
        char_array_4[i] = base64_chars.find(char_array_4[i]);

      char_array_3[0] =
          (char_array_4[0] << 2) + ((char_array_4[1] & 0x30) >> 4);
      char_array_3[1] =
          ((char_array_4[1] & 0xf) << 4) + ((char_array_4[2] & 0x3c) >> 2);
      char_array_3[2] = ((char_array_4[2] & 0x3) << 6) + char_array_4[3];

      for (i = 0; (i < 3); i++) ret.insert(ret.end(), char_array_3[i]);
      i      = 0;
    }
  }

  if (i) {
    for (j = i; j < 4; j++) char_array_4[j] = 0;

    for (j            = 0; j < 4; j++)
      char_array_4[j] = base64_chars.find(char_array_4[j]);

    char_array_3[0] = (char_array_4[0] << 2) + ((char_array_4[1] & 0x30) >> 4);
    char_array_3[1] =
        ((char_array_4[1] & 0xf) << 4) + ((char_array_4[2] & 0x3c) >> 2);
    char_array_3[2] = ((char_array_4[2] & 0x3) << 6) + char_array_4[3];

    for (j = 0; (j < i - 1); j++) ret.insert(ret.end(), char_array_3[j]);
  }

  return ret;
}

#endif  // USERENCRYPTDECRYPT_USERENCRYPTDECRYPT_UTILS_H
