/******************************************************************
 *FILE: UserEncryptDecrypt_Diagnosis.cpp
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UserEncryptDecrypt
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author José Oliveira  (josearmando.oliveira@altran.com)
 * @date Dec, 2017
 */

#include "UserEncryptDecrypt_Diagnosis.h"
#include <boost/shared_ptr.hpp>
#include "backends/UserEncryptDecrypt_BackendManager.h"
#include "core/UserEncryptDecrypt_Configurations.h"
#include "core/authentication/UserEncryptDecrypt_Authentication.h"
#include "core/user/UserEncryptDecrypt_UserManager.h"

std::mutex Diagnosis::_mutex;
std::shared_ptr<Diagnosis> Diagnosis::_instance = nullptr;

void Diagnosis::resetToFactory() {
  std::shared_ptr<Logger1> logger = Logger1::getLogger();
  std::shared_ptr<LogContext> ctx =
      logger->registerContext("RetF", "resetToFactory");

  logger->log(ctx, LogLevel::INFO, "Reset to Factory method");

  std::shared_ptr<Authentication> auth = Authentication::getInstance();
  std::shared_ptr<SDCBackend> sdc =
      BackendManager::getInstance()->get<SDCBackend>();

  std::list<unsigned int> users = UserManager::getInstance()->getCreatedUsers();

  for (auto u : users) {
    auth->delUserDiagnosis(u);
  }

  std::shared_ptr<Configurations> conf = Configurations::getInstance();
  std::string filename = conf->get<std::string>(Configurations::UED_AUTH_FILE);

  std::remove(filename.c_str());
  std::remove(std::string("." + filename + ".bkp").c_str());
  std::remove(std::string("." + filename + ".sig").c_str());

  ConsistentFile::getFile(filename)->openFile(true);

  auth = Authentication::getInstance(true);

  unsigned int defaultProfileNumber =
      conf->get<unsigned int>(Configurations::NR_DEFAULT_PROFILES);

  for (unsigned int i = 1; i <= defaultProfileNumber; i++) {
    auth->createUserCredentials(i);
  }
}

Diagnosis::Diagnosis() {
  _logger = Logger1::getLogger();
  _ctx    = _logger->registerContext("DIAG", "Diagnosis");
}

Diagnosis::~Diagnosis() { _logger->unregisterContext(_ctx); }

std::shared_ptr<Diagnosis> Diagnosis::getInstance() {
  if (!_instance) {
    std::lock_guard<std::mutex> lock(_mutex);
    if (!_instance) {
      try {
        _instance.reset(new Diagnosis());
      } catch (std::bad_alloc &) {
        THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
      }
    }
  }
  return _instance;
}
