/**************************************************************************//**
 * \file       Sds_ViewDB.cpp
 *
 * Exported interface of the view data base.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "Sds_ViewDB.h"
#include "Sds_TextDB.h"
#include "Sds_ViewDB_Trace.h"
#include <string.h>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_VIEW_DB
#include "trcGenProj/Header/Sds_ViewDB.cpp.trc.h"
#endif


enum Sds_ViewId
{
#define SDS_VIEW(id, layout, headline, a, helpline, b, info, c, listItemPrompt)     id,
#include "Sds_ViewDB.dat"
#undef SDS_VIEW
   SDS_VIEW_ID_LIMIT
};


enum Sds_ConditionId
{
#define SDS_COND(id, expression)                                           id,
#include "Sds_ViewDB.dat"
#undef SDS_COND
   SDS_CONDITION_ID_LIMIT
};


enum Sds_HeaderTemplateId
{
#define SDS_HEADER_TEMPLATES(id, textId, templateName)                     id,
#include "Sds_ViewDB.dat"
#undef SDS_HEADER_TEMPLATES
   SDS_HEADER_TEMPLATE_ID_LIMIT
};


struct SubCommand
{
   Sds_ConditionId enVisibe;
   Sds_TextId enTextId;
};


struct Command
{
   Sds_ConditionId enVisibe;
   Sds_ConditionId enSelectable;
   const char* grammarId;
   const char* promptId;
   Sds_TextId enTextId;
   const SubCommand* subCommandList;
   bool isSelected;
   Sds_BehaviorId enBehavior;
};


struct View
{
   const char* idName;
   const char* layout;
   Sds_TextId headline;
   Sds_BehaviorId headlineBehavior;
   Sds_TextId helpline;
   Sds_BehaviorId helplineBehavior;
   Sds_TextId info;
   Sds_BehaviorId infoBehavior;
   const char* listItemPrompt;   // generic prompt for items of dynamic lists (HMI lists)
   const Command* commandList;
};


#define SUB_COMMAND_LIST_BEGIN(command)                     static const SubCommand subCommandList_##command[] = {
#define SUB_COMMAND(condition, textId)                         { condition, textId },
#define SUB_COMMAND_LIST_END                                   { SDS_CONDITION_ID_LIMIT, SDS_TEXT_ID_LIMIT } };
#include "Sds_ViewDB.dat"
#undef SUB_COMMAND_LIST_BEGIN
#undef SUB_COMMAND
#undef SUB_COMMAND_LIST_END


#define subCommandList_NONE                                                     NULL
#define COMMAND_LIST_BEGIN(id)                                                  static const Command commandList_##id[] = {
#define COMMAND(visible, selectable, grammarId, promptId, textId, command, isselected, behaviorId)      { visible, selectable, #grammarId, #promptId, textId, subCommandList_##command, isselected, behaviorId },
#define COMMAND_LIST_END                                                        { SDS_CONDITION_ID_LIMIT, SDS_CONDITION_ID_LIMIT, "0", "0", SDS_TEXT_ID_LIMIT, subCommandList_NONE } };
#include "Sds_ViewDB.dat"
#undef subCommandList_NONE
#undef COMMAND_LIST_BEGIN
#undef COMMAND
#undef COMMAND_LIST_END


static const View viewList[] =
{
#define SDS_VIEW(id, layout, headline, headlineBehavior, helpline, helplineBehavior, info, infoBehavior, listItemPrompt)   { #id, #layout, headline, headlineBehavior, helpline, helplineBehavior, info, infoBehavior, #listItemPrompt, commandList_##id },
#include "Sds_ViewDB.dat"
#undef SDS_VIEW
};


struct Condition
{
   const char* operation;
   const void* left;
   const void* right;
};


static const Condition g_aConditionTable[SDS_CONDITION_ID_LIMIT] =
{
//lint -esym(750, AND) "local macro not referenced - jnd2hi"
#define AND(left, right)                                "AND", (const void*) left, (const void*) right
#define OR(left, right)                                 "OR", (const void*) left, (const void*) right
#define EQUALS(left, right)                             "EQUALS", (const void*) #left, (const void*) #right
#define SDS_COND(id, expression)                        { expression },
#include "Sds_ViewDB.dat"
#undef SDS_COND
};


static const char* const aTemplateNames[] =
{
#define SDS_HEADER_TEMPLATES(id, textId, templateName)                  #templateName,
#include "Sds_ViewDB.dat"
#undef SDS_HEADER_TEMPLATES
};


/**
 * maps a template id to a text id
 */
static const Sds_TextId g_aTemplateTextIds[SDS_HEADER_TEMPLATE_ID_LIMIT] =
{
#define SDS_HEADER_TEMPLATES(id, textId, templateName)      textId,
#include "Sds_ViewDB.dat"
#undef SDS_HEADER_TEMPLATES
};


struct HandOverData
{
   const int HMIScreenId;
   const char* HandOverType;   // list size
   const char* SDSScreenIdwithState;
   const char* SDSScreenIdwithoutState;
};


static const HandOverData handOverDataList[] =
{
#define SDS_HANDOVER(HMIScreenId, HandOverType, SDSScreenIdwithState, SDSScreenIdwithoutState )   { HMIScreenId, #HandOverType, #SDSScreenIdwithState, #SDSScreenIdwithoutState  },
#include "Sds_HandOverDB.dat"
#undef SDS_HANDOVER
};


static std::string mapUnknownViewId(const std::string& viewId)
{
   if (viewId.find("Confirm") != std::string::npos)
   {
      ETG_TRACE_USR1(("warning: mapped screenId '%s' to 'SR_GLO_Confirm'", viewId.c_str()));
      return "SR_GLO_Confirm";
   }
   if (viewId.find("Query") != std::string::npos)
   {
      ETG_TRACE_USR1(("warning: mapped screenId '%s' to 'SR_GLO_Query'", viewId.c_str()));
      return "SR_GLO_Query";
   }
   if ((viewId.find("NbestList") != std::string::npos) ||
         (viewId.find("NBestList") != std::string::npos))
   {
      ETG_TRACE_USR1(("warning: mapped screenId '%s' to 'SR_GLO_NbestList'", viewId.c_str()));
      return "SR_GLO_Nbest_List";
   }
   if (viewId == "SR_SMS_SendText")
   {
      ETG_TRACE_USR1(("warning: mapped screenId '%s' to 'SR_PHO_SelectMessage'", viewId.c_str()));
      return "SR_PHO_SelectMessage";
   }
   if (viewId == "SR_PHO_DialNumber")
   {
      ETG_TRACE_USR1(("warning: mapped screenId '%s' to 'SR_PHO_DialNumberSendText'", viewId.c_str()));
      return "SR_PHO_DialNumberSendText";
   }
   if (viewId == "SR_PHO_DialNumber2")
   {
      ETG_TRACE_USR1(("warning: mapped screenId '%s' to 'SR_PHO_SelectMessage'", viewId.c_str()));
      return "SR_PHO_DialNumberSendText2";
   }

   ETG_TRACE_FATAL(("error: unknown screen id '%s'", viewId.c_str()));
   return viewId;
}


/***********************************************************************//**
*
***************************************************************************/
static Sds_ViewId Sds_ViewDB_enFindViewId(const std::string& viewId)
{
   const size_t numberOfViews = sizeof viewList / sizeof viewList[0];
   ETG_TRACE_USR1(("Sds_ViewDB_enFindViewId  numberOfViews %d", numberOfViews));
   for (size_t i = 0; i < numberOfViews; i++)
   {
      if (viewId == viewList[i].idName)
      {
         return (Sds_ViewId) i;
      }
   }

   // view id was not found - try to map it to existing view id
   std::string patchedViewId = mapUnknownViewId(viewId);
   for (size_t i = 0; i < numberOfViews; i++)
   {
      if (patchedViewId == viewList[i].idName)
      {
         return (Sds_ViewId) i;
      }
   }
   return SDS_VIEW_ID_LIMIT;
}


/**************************************************************************//**
*
******************************************************************************/
std::string Sds_ViewDB_getLayout(const std::string& viewId)
{
   Sds_ViewId enViewId = Sds_ViewDB_enFindViewId(viewId);
   if (enViewId < SDS_VIEW_ID_LIMIT)
   {
      return viewList[enViewId].layout;
   }
   ETG_TRACE_USR1(("Sds_ViewDB_getLayout  Returning ?"));
   return "?";
}


/**************************************************************************//**
*
******************************************************************************/
static Sds_TextId Sds_ViewDB_enGetHeadlineTextId(Sds_ViewId enViewId)
{
   if (enViewId < SDS_VIEW_ID_LIMIT)
   {
      return viewList[enViewId].headline;
   }
   return SDS_TEXT_ID_LIMIT;
}


/**************************************************************************//**
*
******************************************************************************/
static Condition Sds_ViewDB_stGetCondition(Sds_ConditionId enConditionId)
{
   return g_aConditionTable[enConditionId];
}


/**************************************************************************//**
*
******************************************************************************/
std::string Sds_ViewDB_oGetTemplateString(const std::string& templateId)
{
   const size_t numberOfTemplates = sizeof aTemplateNames / sizeof aTemplateNames[0];
   for (size_t i = 0; i < numberOfTemplates; i++)
   {
      if (templateId == aTemplateNames[i])
      {
         return Sds_TextDB_vGetText(g_aTemplateTextIds[i]);
      }
   }
   ETG_TRACE_ERR(("unknown template id '%s'", templateId.c_str()));
   return "";
}


/**************************************************************************//**
*
******************************************************************************/
std::string Sds_ViewDB_oGetHeadline(const std::string& viewId)
{
   Sds_ViewId enViewId = Sds_ViewDB_enFindViewId(viewId);
   return Sds_TextDB_vGetText(Sds_ViewDB_enGetHeadlineTextId(enViewId));
}


/***********************************************************************//**
*
***************************************************************************/
static std::string convertTrueFalse(const std::string& str)
{
   if (str == "TRUE")
   {
      return "1";
   }
   if (str == "FALSE")
   {
      return "0";
   }
   return str;
}


/***********************************************************************//**
*
***************************************************************************/
static bool conditionValuesMatch(const std::string& str1, const std::string& str2)
{
   return (convertTrueFalse(str1) == convertTrueFalse(str2));
}


/***********************************************************************//**
*
***************************************************************************/
static tBool Sds_ViewDB_bIsConditionTrue(Sds_ConditionId conditionId, std::map<std::string, std::string> const& variables)
{
   if (conditionId == SDS_COND__DEFAULT)
   {
      return TRUE;
   }

   const Condition& condition = Sds_ViewDB_stGetCondition(conditionId);
   std::string operation = condition.operation;

   if (operation == "EQUALS")
   {
      std::string left = (const char*) condition.left;
      std::string right = (const char*) condition.right;

      std::map<std::string, std::string>::const_iterator iter;
      for (iter = variables.begin(); iter != variables.end(); ++iter)
      {
         if ((iter->first == left) && conditionValuesMatch(iter->second, right))
         {
            return TRUE;
         }
      }
   }
   if (operation == "AND")
   {
      Sds_ConditionId left = (Sds_ConditionId)(intptr_t) condition.left;
      Sds_ConditionId right = (Sds_ConditionId)(intptr_t) condition.right;
      return Sds_ViewDB_bIsConditionTrue(left, variables) && Sds_ViewDB_bIsConditionTrue(right, variables);
   }
   if (operation == "OR")
   {
      Sds_ConditionId left = (Sds_ConditionId)(intptr_t) condition.left;
      Sds_ConditionId right = (Sds_ConditionId)(intptr_t) condition.right;
      return Sds_ViewDB_bIsConditionTrue(left, variables) || Sds_ViewDB_bIsConditionTrue(right, variables);
   }

   return FALSE;
}


/**************************************************************************//**
 *
 ******************************************************************************/
static const std::vector<const Command*> Sds_ViewDB_getVisibleCommands(Sds_ViewId enViewId, std::map<std::string, std::string> screenVariableData)
{
   std::vector<const Command*> commandList;
   ETG_TRACE_USR1(("Sds_ViewDB_getVisibleCommands with cursor index %d", enViewId));
   if (enViewId < SDS_VIEW_ID_LIMIT)
   {
      const Command* command = viewList[enViewId].commandList;
      if (command)
      {
         while (command->enTextId != SDS_TEXT_ID_LIMIT)
         {
            if (Sds_ViewDB_bIsConditionTrue(command->enVisibe, screenVariableData))
            {
               if (command->enBehavior == Default || command->enBehavior == Icon)
               {
                  commandList.push_back(command);
               }
            }
            command++;
         }
      }
   }
   return commandList;
}


/**************************************************************************//**
 *
 ******************************************************************************/
static const std::vector<const Command*> Sds_ViewDB_getEntireCommandList(const std::string& viewId, std::map<std::string, std::string>const& screenVariableData)
{
   Sds_ViewId enViewId = Sds_ViewDB_enFindViewId(viewId);
   std::vector<const Command*> commandList;
   if (enViewId < SDS_VIEW_ID_LIMIT)
   {
      const Command* command = viewList[enViewId].commandList;
      if (command)
      {
         while (command->enTextId != SDS_TEXT_ID_LIMIT)
         {
            if (Sds_ViewDB_bIsConditionTrue(command->enVisibe, screenVariableData))
            {
               commandList.push_back(command);
            }
            command++;
         }
      }
   }
   return commandList;
}


/***********************************************************************//**
*
***************************************************************************/
std::vector<SelectableText> Sds_ViewDB_oGetCommandList(std::map<std::string, std::string> const& oScreenVariableData, const std::string& viewId)
{
   Sds_ViewId enViewId = Sds_ViewDB_enFindViewId(viewId);
   std::vector<SelectableText> commandStrings;
   const std::vector<const Command*>& visibleCommands = Sds_ViewDB_getVisibleCommands(enViewId, oScreenVariableData);
   for (size_t i = 0; i < visibleCommands.size(); ++i)
   {
      SelectableText text;
      text.text = Sds_TextDB_vGetText(visibleCommands[i]->enTextId);
      text.isSelectable = (Sds_ViewDB_bIsConditionTrue(visibleCommands[i]->enSelectable, oScreenVariableData) == TRUE);
      //text.isSelectable = true;
      text.enBehavior = visibleCommands[i]->enBehavior;
      commandStrings.push_back(text);
   }
   return commandStrings;
}


/***********************************************************************//**
*
***************************************************************************/
std::vector<std::string> Sds_ViewDB_oGetExampleCommandList(std::map<std::string, std::string> const& variables,  const std::string& viewId)
{
   std::vector<std::string> exampleStrings;
   const std::vector<const Command*>& visibleCommands = Sds_ViewDB_getEntireCommandList(viewId, variables);
   for (size_t i = 0; i < visibleCommands.size(); ++i)
   {
      if (visibleCommands[i]->enBehavior == Example)
      {
         exampleStrings.push_back(Sds_TextDB_vGetText(visibleCommands[i]->enTextId));
      }
   }
   return exampleStrings;
}


/**************************************************************************//**
*
******************************************************************************/
std::vector<std::string> Sds_ViewDB_oGetSubCommandList(std::map<std::string, std::string> const& variables,  const std::string& viewId, tU32 cursorIndex)
{
   Sds_ViewId enViewId = Sds_ViewDB_enFindViewId(viewId);
   std::vector<std::string> subCommandStrings;
   const std::vector<const Command*>& visibleCommands = Sds_ViewDB_getVisibleCommands(enViewId, variables);
   ETG_TRACE_USR1(("Sds_ViewDB_oGetSubCommandList with cursor index"));
   if (cursorIndex < visibleCommands.size())
   {
      const SubCommand* subCommand = visibleCommands[cursorIndex]->subCommandList;

      if (subCommand)
      {
         while (subCommand->enTextId != SDS_TEXT_ID_LIMIT)
         {
            if (Sds_ViewDB_bIsConditionTrue(subCommand->enVisibe, variables))
            {
               subCommandStrings.push_back(Sds_TextDB_vGetText(subCommand->enTextId));
            }
            subCommand++;
         }
      }
   }
   return subCommandStrings;
}


/**************************************************************************//**
*
******************************************************************************/
std::vector<std::string> Sds_ViewDB_oGetSubCommandList(std::map<std::string, std::string> const& variables,  const std::string& viewId)
{
   Sds_ViewId enViewId = Sds_ViewDB_enFindViewId(viewId);
   std::vector<std::string> subCommandStrings;
   const std::vector<const Command*>& visibleCommands = Sds_ViewDB_getVisibleCommands(enViewId, variables);

   ETG_TRACE_USR1(("Sds_ViewDB_oGetSubCommandList without cursor index"));
   for (size_t i = 0; i < visibleCommands.size(); i++)
   {
      const SubCommand* subCommand = visibleCommands[i]->subCommandList;

      if (subCommand)
      {
         while (subCommand->enTextId != SDS_TEXT_ID_LIMIT)
         {
            if (Sds_ViewDB_bIsConditionTrue(subCommand->enVisibe, variables))
            {
               subCommandStrings.push_back(Sds_TextDB_vGetText(subCommand->enTextId));
            }
            subCommand++;
         }
      }
   }
   return subCommandStrings;
}


/**************************************************************************//**
*
******************************************************************************/
std::string Sds_ViewDB_oGetHelpline(const std::string& viewId)
{
   Sds_ViewId enViewId = Sds_ViewDB_enFindViewId(viewId);
   if (enViewId < SDS_VIEW_ID_LIMIT)
   {
      return Sds_TextDB_vGetText(viewList[enViewId].helpline);
   }
   return "";
}


/**************************************************************************//**
*
******************************************************************************/
std::string Sds_ViewDB_oGetInfo(const std::string& viewId)
{
   Sds_ViewId enViewId = Sds_ViewDB_enFindViewId(viewId);
   if (enViewId < SDS_VIEW_ID_LIMIT)
   {
      return Sds_TextDB_vGetText(viewList[enViewId].info);
   }
   return "";
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::string Sds_ViewDB_getGrammarId(const std::string& viewId, unsigned int itemIndex, std::map<std::string, std::string> screenVariableData)
{
   Sds_ViewId enViewId = Sds_ViewDB_enFindViewId(viewId);
   const std::vector<const Command*>& visibleCommands = Sds_ViewDB_getVisibleCommands(enViewId, screenVariableData);
   if (itemIndex < visibleCommands.size())
   {
      return visibleCommands[itemIndex]->grammarId;
   }
   return "";
}


/**************************************************************************//**
*
******************************************************************************/
tBool Sds_ViewDB_getIsSelected(const std::string& viewId, unsigned int itemIndex, std::map<std::string, std::string> screenVariableData)
{
   Sds_ViewId enViewId = Sds_ViewDB_enFindViewId(viewId);
   const std::vector<const Command*>& visibleCommands = Sds_ViewDB_getVisibleCommands(enViewId, screenVariableData);
   if (itemIndex < visibleCommands.size())
   {
      return visibleCommands[itemIndex]->isSelected;
   }
   return false;
}


/**************************************************************************//**
 *
 ******************************************************************************/

std::string Sds_ViewDB_getPromptId(const std::string& viewId, unsigned int itemIndex, std::map<std::string, std::string> screenVariableData)
{
   Sds_ViewId enViewId = Sds_ViewDB_enFindViewId(viewId);
   if (enViewId < SDS_VIEW_ID_LIMIT)
   {
      const char* listItemPrompt = viewList[enViewId].listItemPrompt;
      if (strcmp(listItemPrompt, "0") != 0)
      {
         return listItemPrompt;
      }
   }
   const std::vector<const Command*>& visibleCommands = Sds_ViewDB_getVisibleCommands(enViewId, screenVariableData);
   if (itemIndex < visibleCommands.size())
   {
      return visibleCommands[itemIndex]->promptId;
   }
   return "";
}


/***********************************************************************//**
*
***************************************************************************/
std::string Sds_HandOverDB_getSdsInput(const int hmiScreenId, const std::string& handOverType, const bool stateAvailable)
{
   const size_t numberOfScreens = sizeof handOverDataList / sizeof handOverDataList[0];

   for (size_t i = 0; i < numberOfScreens; ++i)
   {
      if (hmiScreenId == handOverDataList[i].HMIScreenId && handOverType == handOverDataList[i].HandOverType && stateAvailable)
      {
         return handOverDataList[i].SDSScreenIdwithState;
      }
      else if (hmiScreenId == handOverDataList[i].HMIScreenId && handOverType == handOverDataList[i].HandOverType && !stateAvailable)
      {
         return handOverDataList[i].SDSScreenIdwithoutState;
      }
   }
   return "";
}


/**************************************************************************//**
*
******************************************************************************/
Sds_BehaviorId Sds_ViewDB_oGetHeadlineBehaviorId(const std::string& viewId)
{
   Sds_ViewId enViewId = Sds_ViewDB_enFindViewId(viewId);
   if (enViewId < SDS_VIEW_ID_LIMIT)
   {
      Sds_BehaviorId behariorId =  viewList[enViewId].headlineBehavior;
      return behariorId;
   }
   return SDS_BEHAVIOR_ID_LIMIT;
}


/**************************************************************************//**
*
******************************************************************************/
Sds_BehaviorId Sds_ViewDB_oGetInfoBehaviorId(const std::string& viewId)
{
   Sds_ViewId enViewId = Sds_ViewDB_enFindViewId(viewId);
   if (enViewId < SDS_VIEW_ID_LIMIT)
   {
      Sds_BehaviorId behariorId =  viewList[enViewId].infoBehavior;
      return behariorId;
   }
   return SDS_BEHAVIOR_ID_LIMIT;
}


/**************************************************************************//**
*
******************************************************************************/
Sds_BehaviorId Sds_ViewDB_oGetHelplineBehaviorId(const std::string& viewId)
{
   Sds_ViewId enViewId = Sds_ViewDB_enFindViewId(viewId);
   if (enViewId < SDS_VIEW_ID_LIMIT)
   {
      Sds_BehaviorId behariorId =  viewList[enViewId].helplineBehavior;
      return behariorId;
   }
   return SDS_BEHAVIOR_ID_LIMIT;
}
