/**************************************************************************//**
 * \file       SqliteDB.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "SqliteDB.h"
#include "FileUtils.h"
#include "SdsUtilsTrace.h"
#include <sqlite3.h>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSA_SQLITE
#include "trcGenProj/Header/SqliteDB.cpp.trc.h"
#endif


namespace sdsa {
namespace sqlite {


SqliteDB::SqliteDB(const ::std::string& path)
   : _path(path)
   , _database(NULL)
   , _statement(NULL)
   , _lastResult(SQLITE_OK)
   , _deleteDatabase(false)
{
}


SqliteDB::~SqliteDB()
{
   close();
}


static void traceSqliteError(int error)
{
   switch (error)
   {
      case SQLITE_OK:
      case SQLITE_ROW:
      case SQLITE_DONE:
         // no error - do nothing
         break;

      default:
         ETG_TRACE_ERR(("error: %d '%s'", error, sqlite3_errstr(error)));
         break;
   }
}


void SqliteDB::open()
{
   ETG_TRACE_USR4(("open '%s'", _path.c_str()));
   bool needsSetup = !fileExists(_path);
   makePath(dirname(_path));
   _lastResult = sqlite3_open(_path.c_str(), &_database);
   traceSqliteError(_lastResult);
   if ((_lastResult != SQLITE_OK) || (_database == NULL))
   {
      close();
      return;
   }
   if (needsSetup)
   {
      ETG_TRACE_USR4(("setup new database"));
      exec("PRAGMA synchronous = OFF;");
      exec("PRAGMA journal_mode = MEMORY;");
      setup();    // TODO jnd2hi: what if setup gets interrupted, e.g. by low-power?
   }
}


void SqliteDB::close()
{
   finalize();

   if (_database)
   {
      ETG_TRACE_USR4(("close '%s'", _path.c_str()));
      int result = sqlite3_close(_database);
      traceSqliteError(result);
      _database = NULL;
   }
   else
   {
      ETG_TRACE_USR4(("close '%s' - already closed", _path.c_str()));
   }

   if (_deleteDatabase)
   {
      ETG_TRACE_USR4(("deleting database!"));
      deleteFile(_path);
      _deleteDatabase = false;
   }
}


int SqliteDB::exec(const std::string& query)
{
   if (_database)
   {
      ETG_TRACE_USR4(("exec: '%s'", query.c_str()));
      _lastResult = sqlite3_exec(_database, query.c_str(), NULL, NULL, NULL);
      traceSqliteError(_lastResult);
      checkDeletionRequired(_lastResult);
      return _lastResult;
   }
   return SQLITE_ERROR;
}


void SqliteDB::prepare(const std::string& query)
{
   finalize();
   _lastResult = sqlite3_prepare_v2(_database, query.c_str(), (unsigned int) query.length(), &_statement, NULL);
   ETG_TRACE_USR4(("prepare %p: '%s'", _statement, query.c_str()));
   traceSqliteError(_lastResult);
   checkDeletionRequired(_lastResult);
}


void SqliteDB::step()
{
   if (_statement != NULL)
   {
      ETG_TRACE_USR4(("step %p", _statement));
      _lastResult = sqlite3_step(_statement);
      traceSqliteError(_lastResult);
      checkDeletionRequired(_lastResult);
   }
   else
   {
      ETG_TRACE_ERR(("step: error - no statement"));
   }
}


bool SqliteDB::columnSelected()
{
   return (_lastResult == SQLITE_ROW);
}


int SqliteDB::getColumnInt(int column, int defaultValue)
{
   int value = defaultValue;
   if (_statement && (_lastResult == SQLITE_ROW))
   {
      value = sqlite3_column_int(_statement, column);
      ETG_TRACE_USR4(("colum_int(%d) = %d", column, value));
   }
   else
   {
      ETG_TRACE_ERR(("colum_int: error - no row selected - returning %d", value));
   }
   return value;
}


::std::string SqliteDB::getColumnText(int column, const ::std::string& defaultValue)
{
   ::std::string text = defaultValue;
   if (_statement && (_lastResult == SQLITE_ROW))
   {
      return reinterpret_cast<const char*>(sqlite3_column_text(_statement, column));
      ETG_TRACE_USR4(("colum_text(%d) = '%s'", column, text.c_str()));
   }
   else
   {
      ETG_TRACE_ERR(("colum_text: error - no row selected - returning '%s'", defaultValue.c_str()));
   }
   return text;
}


void SqliteDB::finalize()
{
   if (_statement != NULL)
   {
      ETG_TRACE_USR4(("finalize %p", _statement));
      sqlite3_finalize(_statement);
      _statement = NULL;
   }
}


void SqliteDB::checkDeletionRequired(int errorCode)
{
   switch (errorCode)
   {
      case SQLITE_CANTOPEN:
      case SQLITE_SCHEMA:
      case SQLITE_MISMATCH:
      case SQLITE_AUTH:
      case SQLITE_FORMAT:
      case SQLITE_RANGE:
      case SQLITE_NOTADB:
      case SQLITE_CORRUPT:
      case SQLITE_FULL:
         _deleteDatabase = true;
         ETG_TRACE_ERR(("deletetion of database required - error code = %d", errorCode));
         break;

      default:
         break;
   }
}


void SqliteDB::beginTransaction()
{
   exec("BEGIN TRANSACTION;");
}


void SqliteDB::endTransaction()
{
   exec("END TRANSACTION;");
}


std::string SqliteDB::escape(const std::string& text)
{
   std::string escaped = text;
   size_t pos = escaped.find("'");
   while (pos != std::string::npos)
   {
      escaped.insert(pos + 1, "'");
      pos = escaped.find('\'', pos + 2);
   }
   return escaped;
}


};
};
